/*

 step 1: 	read file 1 + 2 parallel
 			compare values of file 1 and 2 in order to detect values which must be animated
 			(remember these values by saving line number and number of value in line)
			
			read file 2 + 3 parallel 
			...
			
 step 2:	read all values in all files and save them (to memory)
 
 step 3:    calculate the interpolated values
 
 step 4:	create all the animation files 
            (using file 1 as template, filling in the calculated values)
            (write a header to every file, containing important informations)

*/

#include <conio.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#define FALSE 0
#define TRUE 1
#define LINE_LEN 256
#define TEMP_FILE_NAME xy060100.abc
#define EXCEL_FILE_NAME exceldat.xls

#include "list.h"

struct LGS {
	double y1, y2, y3;
	double a1, a2, a3;
	double b1, b2, b3;
	double c1, c2, c3;		
	double a, b, c;	// resulting parameters
};

struct parabola {
	double a, b, c;	// parameters, y = ax2 + bx + c
};

// converts long to double
double d_value(long value)
{
	char str[50];
	double d_val;
	
	sprintf(str, "%i", value);
	d_val = atof(str);
	return( d_val );
}

// solve 3-dim. LGS
void solve_lgs(struct LGS *lgs)
{
	double z1, z2, n1, n2;
	
	// print LGS
/*
	printf("\n%8.2f | %7.3f %7.3f %7.3f", lgs->y1, lgs->a1, lgs->b1, lgs->c1);
	printf("\n%8.2f | %7.3f %7.3f %7.3f", lgs->y2, lgs->a2, lgs->b2, lgs->c2);
	printf("\n%8.2f | %7.3f %7.3f %7.3f", lgs->y3, lgs->a3, lgs->b3, lgs->c3);
	getch();
*/
	if ((lgs->c2 == 0) || (lgs->c3 == 0) || (lgs->b2*lgs->c1-lgs->b1*lgs->c2 == 0))
	{
		printf("ERROR: null value, can't solve LGS!");
		exit(1);
	}

	z1 = (lgs->b3*lgs->c1-lgs->b1*lgs->c3)*(lgs->y2*lgs->c1-lgs->y1*lgs->c2);
	n1 = lgs->b2*lgs->c1-lgs->b1*lgs->c2;
	
	z2 = (lgs->b3*lgs->c1-lgs->b1*lgs->c3)*(lgs->a2*lgs->c1-lgs->a1*lgs->c2);
	n2 = lgs->b2*lgs->c1-lgs->b1*lgs->c2;
	
	lgs->a = (lgs->y3*lgs->c1-lgs->y1*lgs->c3-(z1/n1)) / (lgs->a3*lgs->c1-lgs->a1*lgs->c3-(z2/n2));
	
	z1 = lgs->y2*lgs->c1-lgs->y1*lgs->c2-lgs->a*(lgs->a2*lgs->c1-lgs->a1*lgs->c2);
	n1 = lgs->b2*lgs->c1-lgs->b1*lgs->c2;
	
	lgs->b = z1 / n1;
	
	lgs->c = (lgs->y1-lgs->a*lgs->a1-lgs->b*lgs->b1)/lgs->c1;
}

// returns TRUE if given file exists, otherwise FALSE
int file_exists(char *filename)
{
	FILE *test;

    if( (test = fopen( filename, "r" )) == NULL )
    {
    	return( FALSE );
    }               
    else
    {
    	fclose(test);
    	return( TRUE );
    }
}

// prepocessor: removes all '//'-comment lines
//              removes all empty lines
int prepocess(struct ini_list_struct *start)
{
	struct ini_list_struct *run = start;
	FILE *read, *write;
	char line[LINE_LEN];
	//char *keyword;
	int print;
	
	while (run != NULL)
	{
		// open input file
	    if( (read = fopen( run->keyframe, "r" )) == NULL )
	    {
	    	printf("\nERROR: could not open file: %s", run->keyframe);
	    	printf("\nterminated.");
	    	exit(1);
	    }               
		// open output file
	    if( (write = fopen( "TEMP_FILE_NAME", "w" )) == NULL )
	    {
	    	printf("\nERROR: could not open temp output file: %s", "TEMP_FILE_NAME");
	    	printf("\nterminated.");
	    	exit(1);
	    }
	    
		while (fgets(line, LINE_LEN-1, read) != NULL)
		{
			print = TRUE;
			if ((line[0] == '/') && (line[1] == '/')) print = FALSE;
			if (line[0] == '\n') print = FALSE;
			
			if (print)
			{
				/*keyword = strstr(line, "rotate");
				if (keyword != NULL)
				{
					if (*(keyword+strlen("rotate")+1) != '<')
					{
						printf("\n in dieser Zeile ersetzen:");
						printf("\n%s", line);
						getch();
					}
				}*/
				fputs(line, write);
			}
			
		}
	    
	    fclose(read);
	    fclose(write);
	    
	    if( remove( run->keyframe ) == -1 )
	    {
	    	printf("\nERROR: preprocessor was unable to remove file %s", run->keyframe);
	    	printf("\nterminated.");
	    	exit(1);
	    }	
	    if (rename( "TEMP_FILE_NAME", run->keyframe ) != 0)
	    {
	    	printf("\nERROR: preprocessor was unable to rename %s to %s", "TEMP_FILE_NAME", run->keyframe);
	    	printf("\nterminated.");
	    	exit(1);
	    }

	    run = (struct ini_list_struct*)run->next;              
	}
	return( 1 );
}


// Sucht end_of_comment-Marke '*/'
// return value:
// 0 --> nothing found
// 1 --> found end of comment (pos is set to the position of the '/')
int find_end_of_comment(char *line, unsigned long *pos)
{
	char *run = line;
	int result = 0;
		
	while ((*run != 0) && (result == 0))
	{
		if (*run == '*')
			if (*(run+1) == '/')
			{
				result = 1;
				*pos = run - line + 1;
			}	
		run++;	
	}
	
	return ( result );
}

// Sucht in einem String nach gltigen Zahlen
/* return value:
   0 --> nothing found
   1 --> found value
   2 --> found start of comment

   if a '//' comment is found pos is set to length(line) and
   the return value is 0 (--> nothing found)
   
   if a '#include' is found, this is treated as comment, because
   normally here's a file included, if the filename contains
   number, these are interpolated too (-->ani1.24535.inc)
   
   if a '/ *' comment is found pos is set to pos of '*' and
   the return value is 2 (--> found start of comment)
*/
int read_value(char *line, unsigned long *pos, unsigned long* len)
{
	char *run = line;
	char *run2;
	
	int status = 99;
	// 99 --> not finished
	// 0 --> finished, found nothing
	// 1 --> finished, found value
	// 2 --> finished, found start of comment

	int found_digit = FALSE;
	int found_point = FALSE;
	int found_sign = FALSE;
	int search_value;
			
	if (*run == 0) return( 0 );  // if empty string --> return 'nothing found'

	do
	{
		// Prfen ob Zeile mit '#include' beginnt
		if (strlen(run) > 7) 
			if (strncmp(run, "#include", 8) == 0)
				return( 0 );
				
		// Prfen auf Anfang eines Kommentars
		if (*run == '/')
		{
			if (*(run+1) == '/')
			{   // found // comment --> ignore rest of line
				*pos = strlen(line);
				*len = 0;
				status = 0;
			}
			else
				if (*(run+1) == '*')
				{   // found start of / * comment
			    	*pos = run - line + 1;
			    	*len = 0;
			    	status = 2;
				}
		}

		// Prfen, auf Anfang einer Zahl (0..9, '.', '-')
		if (isdigit(*run)) found_digit = TRUE;
		else 
			if ((*run == '.') && (isdigit(*(run+1))) ) found_point = TRUE;
				else if ( ((*run == '-') && (isdigit(*(run+1)))) || ((*run == '-') && (*(run+1) == '.')) ) found_sign = TRUE;
		
		if (found_digit || found_point || found_sign)
		{   // Falls Anfang einer Zahl bis zum Ende der Zahl vorrcken
			search_value = TRUE;		
			run2 = run+1;
			while ((*run2 != 0) && (search_value == TRUE))
			{
				if (isdigit(*run2))
				{
					run2++;
					found_digit = TRUE;
				}
				else
				{
					if (*run2 == '.')
					{
						if (! found_point) 
						{
							found_point = TRUE;
							run2++;
						}	
						else search_value = FALSE;
					}
					else search_value = FALSE;
				}
			}
			
			// Falls in der Zahl digits vorkamen, wurde gltige Zahl gefunden
			if (found_digit) 
			{
				*pos = run - line;
				*len = run2 - run;
				status = 1;
			}
			else  // weitersuchen
			{
				found_point = FALSE;
				found_digit = FALSE;
				found_sign = FALSE;
			}
		}
			
		//printf("\n%c -- %d", *run, *run);
		//getch();
		if (*run != 0) run++;	// nur weitergehen, wenn nicht schon am Ende der Zeile
	}
	while ((status == 99) && (*run != 0));
	
	if (status == 99) status = 0;
	
	return( status );
}

// Step one:
// find values which must be animated by comparing file 1 with file 2, file 2 with file 3, ...
struct pos_of_numbers *compare_two_files(struct pos_of_numbers *list, char *file1, char *file2)
{
	FILE *f1, *f2;
	char str1[LINE_LEN], str2[LINE_LEN];
	char value1[256], value2[256];
	int search_for_end_of_comment = FALSE;
	int found_end1 = FALSE;
	int found_end2 = FALSE;	
	int result1 = 0;
	int result2 = 0;
	unsigned long line, number;
	unsigned long pos1, len1, abs_pos1;
	unsigned long pos2, len2, abs_pos2;
	double val1, val2;
        
	// open both files
    if( (f1 = fopen( file1, "r" )) == NULL )
    {
    	printf("\nERROR: could not open file: %s", file1);
    	printf("\nterminated.");
    	exit(1);
    }               
    if( (f2 = fopen( file2, "r" )) == NULL )
    {
    	printf("\nERROR: could not open file: %s", file2);
    	printf("\nterminated.");
    	exit(1);
    }               

	line = 0;
	
	// repeat until eof
	while ( (found_end1 != TRUE) && (found_end2 != TRUE) )
	{
		// read data
		if (fgets(str1, LINE_LEN-1, f1) == NULL)
		{
			found_end1 = TRUE;			
			*str1 = 0;
		}
		//printf("\na%4lu:%s", line, str1);
		if (fgets(str2, LINE_LEN-1, f2) == NULL)
		{
			found_end2 = TRUE;			
			*str2 = 0;
		}
		//printf("\nb%4lu:%s", line, str2);
		
		pos1 = 0;
		pos2 = 0;
		len1 = 0;
		len2 = 0;
		number = 0;
			
		if (!search_for_end_of_comment)
		{
			do {
				// search values		
				abs_pos1 = pos1 + len1; // read_value liefert nur relative positionen zurck
				abs_pos2 = pos2 + len2; // weil es auch nur relative bergeben bekommt
				result1 = read_value(str1+pos1+len1, &pos1, &len1);
				result2 = read_value(str2+pos2+len2, &pos2, &len2);
			    pos1 = pos1 + abs_pos1;
			    pos2 = pos2 + abs_pos2;
			    
				// check results
				if (result1 != result2)
				{
					printf("\nERROR: difference in files %s and %s, line %lu", file1, file2, line+1);
					printf("\n%s: %s", file1, str1);
					printf("\nresult: %i   pos: %lu   len: %lu", result1, pos1, len1);
					printf("\n%s: %s", file2, str2);
					printf("\nresult: %i   pos: %lu   len: %lu", result2, pos2, len2);
					printf("\nterminated.");
					exit(1);
				}
				
				// Werte auswerten
				if (result1 == 1)			
				{
					strncpy(value1, (str1+pos1), (size_t)len1);
					*(value1 + len1) = 0;
					strncpy(value2, (str2+pos2), (size_t)len2);
					*(value2 + len2) = 0;
					//printf("\n%30s <---> %30s", value1, value2);
					//printf("\n%30d <---> %30d", pos1, pos2);
					val1 = atof(value1);
					val2 = atof(value2);
					if (val1 != val2)
					{   // unterschiedliche Werte gefunden, diese Stelle der Liste hinzufgen
						if (!value_exists(list, line, number))						
						{
							list = add_value(list, line, number);				
							printf("\nAdd line %lu, number %lu", line+1, number+1);
						}	
						//_getch();
					}			
					number++;
				}
				
				if (result1 == 2) search_for_end_of_comment = TRUE;
	
			} while (result1 == 1);
		}

		// not so good:
		// if there are values behind the '*/' in the same line, they won't be recognized
		// because the program is switsching to the next line!!!
		if (search_for_end_of_comment)
		{	// found start of comment '/*-comment" --> search for end of comment
			result1 = find_end_of_comment(str1+pos1+len1, &pos1);					
			result2 = find_end_of_comment(str2+pos2+len2, &pos2);					
			if (result1 != result2)
			{
				printf("\nERROR: difference in files %s and %s, line %ld", file1, file2, line+1);
				printf("\nterminated.");
				exit(1);
			}
			if (result1 == 1)  // found end of comment
			{
			 	search_for_end_of_comment = FALSE;
			}
		}		

		line++;
		//printf("\nEnd of this line ... now next line");
		//getch();
	}
	
	if ( found_end1 != found_end2 )
	{
		printf("\nWARNING: files %s and %s", file1, file2);
		printf("\n         differnt number of lines!");
	}

	fclose(f1);
	fclose(f2);
	
	return( list );    
}

// Main step-one function
// compares file1 with file2, file2 with file3, ...
// in order to find the values which must be animated
struct pos_of_numbers *find_animation_values(struct pos_of_numbers *list, struct ini_list_struct *start)
{
	struct ini_list_struct *run, *run2;
	int count = 0;
	int finished = FALSE;	     
	     
	run = start;
	run2 = start->next;
	
	if ((run2 == NULL) || (run == NULL))
	{
		printf("\nERROR: need at least to keyframes to do an animation\nterminated.\n");
		exit(1);
	}

	while (run2 != NULL)
	{	
		count++;
		// compare the two files
		printf("\ncomparing %s with %s", run->keyframe, run2->keyframe);		
		list = compare_two_files(list, run->keyframe, run2->keyframe);
		run = run2;
		run2 = (struct ini_list_struct *)run2->next;
	}
	return( list );
}

// Main step-two function
// fetches the animation values from each keyframe file
int read_animation_values(struct values_struct *array, struct pos_of_numbers *positions, struct ini_list_struct *start, int files)
{
	struct ini_list_struct *inilist = start;
    FILE *keyframe;
	char str[LINE_LEN], value_str[LINE_LEN];
	int file_count = 0;
	int value_count = 0;
	unsigned long line_count, pos_count, len, pos, abs_pos;
	int values; 
	int result;
	struct pos_of_numbers *run;
			     
	// number of values to read from each file
	values = count_list(positions);
	
	// loop for each keyframe file
	for (file_count = 0; file_count < files; file_count++)
	{
	    // open keyframe file
	    if( (keyframe = fopen( inilist->keyframe, "r" )) == NULL )
	    {
	    	printf("\nERROR: could not open keyframe file: %s", inilist->keyframe);
	    	printf("\nterminated.");
	    	exit(1);
	    }               
        
		// read first line
		if (fgets(str, LINE_LEN-1, keyframe) == NULL)
		{
			printf("\nERROR: unable to read line %lu in keyframe file <%s>", run->line, inilist->keyframe);
			printf("\nterminated\n");
			exit(1);
		}
		//printf("\nreading file %s:", inilist->keyframe);
		
		run = positions;
		line_count = 0;
		pos_count = 0;  // value number x in line
		pos = 0;		// position of value in line
		len = 0;		// length of value
		
		// read values from keyframe file
		for (value_count = 0; value_count < values; value_count++)
		{
			// goto line according to list of value positions
			while (line_count < run->line)
			{
				if (fgets(str, LINE_LEN-1, keyframe) == NULL)
				{
					printf("\nERROR: unable to read line %lu in keyframe file <%s>", run->line, inilist->keyframe);
					printf("\nterminated\n");
					exit(1);
				}
				//printf("\n%lu:%s", line_count, str);
				line_count++;
				pos_count = 0;  // value number x in line
				pos = 0;		// position of value in line
				len = 0;		// length of value
			}
			
			// goto value according to list of value positions
			do {
			    // 0 --> nothing found
				// 1 --> found value
				// 2 --> found start of comment
				abs_pos = pos + len;	// read_value liefert nur relative positionen zurck weil
				result = read_value(str+pos+len, &pos, &len); // es auch nur relative bergeben bekommt
			    pos = pos + abs_pos;
			    //printf("\nresult: %i", result);
			    if (result == 1) pos_count++;
			} while ((pos_count < run->number+1) && (result == 1));
			if ((pos_count != run->number+1) || (result != 1))
			{
				printf("\nERROR: couldn't find value number %lu in line %lu in file %s", run->number, line_count, inilist->keyframe);
				printf("\nterminated.");
				exit(1);
			}
			strncpy(value_str, (str+pos), (size_t)len);
			*(value_str + len) = 0;
			//printf("\nset value (number %i / file %i) to %s", value_count, (int)file_count, value_str);
			//getch();
			set_value(array, atof(value_str), value_count, (int)file_count);
			run = (struct ini_list_struct *)run->next;
		}		
		inilist = inilist->next;
	}
	
	return( TRUE );
}

// Parabeltyp von p1 bestimmen (fr Verbindung y2 nach y3)
char determine_ptype1(double y1, double y2, double y3)
{
	// 'A' --> Anfangssteigung 0 (nachfolg. Punkt wird nach vorne gespiegelt)
	// 'E' --> Endsteigung 0 (vorheriger Punkt wird nach hinten gespiegelt)
	// 'D' --> Dreipunktparabel aus vorherigem, aktuellem und nachfolg. Punkt
	
	if (y2 == y3) 
	{
	 	printf("\nERROR: cannot determine p type for constant connection!");
	 	printf("\nterminated.");
	 	exit(1);
	}

	// y2 != y3, vorher aber Gerade --> 'A'
	if (y1 == y2) return( 'A' );	
	// streng monoton steigend --> Dreipunktparabel
	if ((y1 < y2) && (y2 < y3)) return( 'D' );  
	// streng monoton fallend --> Dreipunktparabel
	if ((y1 > y2) && (y2 > y3)) return( 'D' ); 
	// y2 ist Hochpunkt --> Parabel mit Anfangssteigung 0
	if ((y1 < y2) && (y3 < y2)) return( 'A' ); 
	// y2 ist Tiefpunkt --> Parabel mit Anfangssteigung 0
	if ((y1 > y2) && (y3 > y2)) return( 'A' ); 

	printf("\nERROR: this should not happen!!!");
	printf("\ninternal error\nterminated.\n");
	exit(1);
}

// Parabeltyp von p2 bestimmen (fr Verbindung y1 nach y2)
char determine_ptype2(double y1, double y2, double y3)
{
	// 'A' --> Anfangssteigung 0 (nachfolg. Punkt wird nach vorne gespiegelt)
	// 'E' --> Endsteigung 0 (vorheriger Punkt wird nach hinten gespiegelt)
	// 'D' --> Dreipunktparabel aus vorherigem, aktuellem und nachfolg. Punkt
	
	if (y1 == y2) 
	{
	 	printf("\nERROR: cannot determine p type for constant connection!");
	 	printf("\nterminated.");
	 	exit(1);
	}

	// y1 != y2, nachher aber Gerade --> 'E'
	if (y2 == y3) return( 'E' );	
	// streng monoton steigend --> Dreipunktparabel
	if ((y1 < y2) && (y2 < y3)) return( 'D' );  
	// streng monoton fallend --> Dreipunktparabel
	if ((y1 > y2) && (y2 > y3)) return( 'D' ); 
	// y2 ist Hochpunkt --> Parabel mit Endsteigung 0
	if ((y1 < y2) && (y3 < y2)) return( 'E' ); 
	// y2 ist Tiefpunkt --> Parabel mit Endsteigung 0
	if ((y1 > y2) && (y3 > y2)) return( 'E' ); 

	printf("\nERROR: this should not happen!!!");
	printf("\ninternal error\nterminated.\n");
	exit(1);
}


// key values: Zeiger auf 1-dim. array der values in den keyframes
// keyframes : Anzahl der keyframes
// values: Zeiger auf 1-dim. array mit allen Werten des Animationsparameters
// ini_list: Werte des Ini-Files (enthlt Anzahl frames zwischen den Keyframes)
void calc_one_value(double *key_values, int keyframes, double *values, struct ini_list_struct *ini_list)
{
	long key_count;
	long frame_count = 0;
	long frames = 0, n, len;
	double x, d_len, x1, x2, y1, y2, n_max;
	char connect;  // --> 'G' --> Gerade
	               // --> 'P' --> Parabeln
	char par1; // A --> Anfangssteigung 0 (nachfolg. Punkt wird nach vorne gespiegelt)
	char par2; // E --> Endsteigung 0 (vorheriger Punkt wird nach hinten gespiegelt)
	           // D --> Dreipunktparabel aus vorherigem, aktuellem und nachfolg. Punkt
	struct parabola p1, p2;
	struct LGS lgs;
	struct ini_list_struct *run_ini_list = ini_list;
		
	//printf("\nkeyframes = %i\n", keyframes);
	//printf("\n");
	for (key_count = 0; key_count < keyframes-1; key_count++)
	{
		//printf("\n%8.3f", key_values[key_count]);		
		
		// *** Verbindungstypen bestimmen ***
		if (key_values[key_count] == key_values[key_count+1])
		{   // Fall 1: y == y+1 --> Verbindung ist Gerade (Konstante)
			connect = 'G';
		}
		else
		{	// Fall 2: y != y+1  --> Verbindung besteht aus zwei Parabeln
			connect = 'P';
			// Parabel 1 ist von vorheriger Verbindung abhngig
			if (key_count == 0) par1 = 'A'; // ganz am Anfang: immer Parabel mit Anf.-Steigung 0
			else par1 = determine_ptype1(key_values[key_count-1], key_values[key_count], key_values[key_count+1]);
			// Parabel 2 ist von nachfolgender Verbindung abhngig
			if (key_count+2 == keyframes) par2 = 'E'; // ganz am Ende: immer Par. mt Endsteigung 0
			else par2 = determine_ptype2(key_values[key_count], key_values[key_count+1], key_values[key_count+2]);
		}
		/*
		if (connect == 'G')
			printf(" G ");
		if (connect == 'P')
			printf(" %c%c ",par1, par2);*/
			
		// *** Parabeln bestimmen ***	
		// relevante Parabelhlfte liegt immer bei x=0 (Scheitelpunkt) bis x=1
		if (connect == 'P')
		{
			// y = axx + bx + c
			lgs.a1 = 1;
			lgs.b1 = -1;
			lgs.c1 = 1;
			lgs.a2 = 0;
			lgs.b2 = 0;
			lgs.c2 = 1;
			lgs.a3 = 1;
			lgs.b3 = 1;
			lgs.c3 = 1;
			switch (par1)
			{
				case 'A': // Parabel mit Anfangssteigung 0, nachfolgenden Punkt nach vorne kopieren
					// linker Punkt
					lgs.y1 = key_values[key_count+1];
					// Scheitelpunkt
					lgs.y2 = key_values[key_count];
					// rechter Punkt
					lgs.y3 = key_values[key_count+1];
					// LGS lsen
					solve_lgs(&lgs);
					p1.a = lgs.a;
					p1.b = lgs.b;
					p1.c = lgs.c;
					//printf("\nPar1 type A: von %6.2f ber %6.2f nach %6.2f", lgs.y1, lgs.y2, lgs.y3);
					//printf("\n             %6.2f xx + %6.2f x + %6.2f", p1.a, p1.b, p1.c);
					break;	
				case 'D': // Dreipunktparabel
					// linker Punkt
					lgs.y1 = key_values[key_count-1];
					// mittlerer Punkt
					lgs.y2 = key_values[key_count];
					// rechter Punkt
					lgs.y3 = key_values[key_count+1];
					// LGS lsen
					solve_lgs(&lgs);
					p1.a = lgs.a;
					p1.b = lgs.b;
					p1.c = lgs.c;
					//printf("\nPar1 type D: von %6.2f ber %6.2f nach %6.2f", lgs.y1, lgs.y2, lgs.y3);
					//printf("\n             %6.2f xx + %6.2f x + %6.2f", p1.a, p1.b, p1.c);
				break;	
			}	
			switch (par2)
			{
				case 'E': // Parabel mit Endsteigung 0, vorherigen Punkt nach hinten kopieren
					// linker Punkt
					lgs.y1 = key_values[key_count];
					// Scheitelpunkt
					lgs.y2 = key_values[key_count+1];
					// rechter Punkt
					lgs.y3 = key_values[key_count];
					// LGS lsen
					solve_lgs(&lgs);
					p2.a = lgs.a;
					p2.b = lgs.b;
					p2.c = lgs.c;
					//printf("\nPar2 type E: von %6.2f ber %6.2f nach %6.2f", lgs.y1, lgs.y2, lgs.y3);
					//printf("\n             %6.2f xx + %6.2f x + %6.2f", p2.a, p2.b, p2.c);
					break;	
				case 'D': // Dreipunktparabel
					// linker Punkt
					lgs.y1 = key_values[key_count];
					// mittlerer Punkt
					lgs.y2 = key_values[key_count+1];
					// rechter Punkt
					lgs.y3 = key_values[key_count+2];
					// LGS lsen
					solve_lgs(&lgs);
					p2.a = lgs.a;
					p2.b = lgs.b;
					p2.c = lgs.c;
					//printf("\nPar2 type D: von %6.2f ber %6.2f nach %6.2f", lgs.y1, lgs.y2, lgs.y3);
					//printf("\n             %6.2f xx + %6.2f x + %6.2f", p2.a, p2.b, p2.c);
				break;	
			}
		}
		
		// *** Animationswerte des aktuellen Abschnittes berechnen ***
		len = run_ini_list->frames+1;  // Anzahl frames fr diesen Abschnitt
		d_len = d_value(len);
		n_max = d_len - 1; // fr gewichteten Durchschnitt aus p1 und p2
		x = 0;
		for (n = 0; n < len; n++)
		{
			if (connect == 'G')
				values[frames] = key_values[key_count];
			else    // connect = 'P'
			{
				// x-Pos fr p1 (relevant von 0 .. 1)
				x1 = x;
				// x-Pos fr p2 (relevant von -1 .. 0)
				x2 = -1 + x;
				// y-Wert von p1 bei x1
				y1 = p1.a*x1*x1 + p1.b*x1 + p1.c;
				// y-Wert von p2 bei x2
				y2 = p2.a*x2*x2 + p2.b*x2 + p2.c;
				// gewichteter Endwert
				values[frames] = (1-x)*y1 + x*y2;
				x = x + 1 / d_len;
			}	
			frames++;
		}
		run_ini_list = run_ini_list->next;		
		//getch();
	}
	values[frames] = key_values[key_count];
	//printf("\n%8.3f", key_values[keyframes-1]);		

	//printf("\nfinal frame value: %i", frames);
	
	/*
	printf("\n*** results ***\n");
	for (n = 0; n <= frames; n++)
	{
		printf("%7.3f ",values[n]);
	}
	getch();*/
}

// key_values: einfache verkette Liste der bereits ausgelesenen keyvalues
// keyframes: Anzahl der keyframes
// all_values: 2-dim array enthlt berechnete Werte der Animationsparameter fr alle frames
// ini_list: Werte des Ini-Files (enthlt Anzahl frames zwischen den Keyframes
void calc_values(struct values_struct *key_values, int keyframes, struct values_struct *all_values, struct ini_list_struct *ini_list)
{
	struct values_struct *run_key_values = key_values;
	struct values_struct *run_all_values = all_values;
	
	int value_count = 1;
	
	//printf("\n");
	while (run_key_values != NULL)
	{
		//printf("\rcalculating animation value %i", value_count);
		calc_one_value(run_key_values->value_array, keyframes, run_all_values->value_array, ini_list);
		value_count++;
				
		run_key_values = run_key_values->next;
		run_all_values = run_all_values->next;
	}
}

// open file for writing, first checking if it exists
// if it exists, ask for overwriting
FILE *open_file(char *path_buffer, char *overwrite)
{
	FILE *out;
		
	if (*overwrite != 'a')
	{
		// check if file exists
		if ((out = fopen(path_buffer, "r")) != NULL)
		{
			fclose(out);
			printf("\nfile %s already exists.\nOverwrite ? (<y>es, <a>lways overwrite, <q>uit)\n", path_buffer);
			*overwrite = getch();
			*overwrite = tolower(*overwrite);
			if ((*overwrite == 'y') || (*overwrite == 'a'))
			{
				if ((out = fopen(path_buffer, "w")) == NULL)
				{
					printf("\nERROR: unable to open %s for writing\nterminated.\n", path_buffer);
					exit(1);
				}
				if (*overwrite == 'y')
				{
					*overwrite = 'n';
					printf("\noverwrite %s", path_buffer);
				}
				else
				{
					printf("\noverwriting all files");
				}
			}
			else
			{
				printf("\nterminated.\n");
				exit(1);
			}
		}
		else
		{
			if ((out = fopen(path_buffer, "w")) == NULL)
			{
				printf("\nERROR: unable to open %s for writing\nterminated.\n", path_buffer);
				exit(1);
			}
		}
	}									
	else
	{
		if ((out = fopen(path_buffer, "w")) == NULL)
		{
			printf("\nERROR: unable to open %s for writing\nterminated.\n", path_buffer);
			exit(1);
		}
	}
		
	return( out );
}

void write_excel_file(long frames, struct values_struct *all_values)
{
	long frame_count;
	long relative_frame_position = 0;  // frame count relative to last keyframe
	struct values_struct *run_all_values;
	FILE *out;

	// open output file for writing
	if ((out = fopen("exceldat.xls", "w")) == NULL)
	{
		printf("\nERROR: unable to open excel file %s\nterminated.\n", "EXCEL_FILE_NAME");
		exit(1);
	}

    run_all_values = all_values; 	// go to beginning of values_list

	// loop for all values and frames
	while (run_all_values != NULL)
	{
		for (frame_count = 0; frame_count < frames; frame_count++)
		{
			fprintf(out, "%f, ",run_all_values->value_array[frame_count]);
		}	
		run_all_values = run_all_values->next;
		fprintf(out, "\n");
	}
	
	fclose(out);	
}		


// create the animation files
void write_animation_files(long frames, struct pos_of_numbers *positions, struct values_struct *all_values, struct ini_list_struct *ini_list, char *filename, int start_value)
{
	char path_buffer[_MAX_PATH];
	char drive[_MAX_DRIVE];
	char dir[_MAX_DIR];
	char fname[_MAX_FNAME], outname[_MAX_FNAME];
	char ext[_MAX_EXT];

	long frame_count;
	unsigned long pos, len, abs_pos, line_count, pos_count;
	long relative_frame_position = 0;  // frame count relative to last keyframe
	struct ini_list_struct *run_ini_list = ini_list;
	struct pos_of_numbers *run_positions;
	struct values_struct *run_all_values;
	FILE *out, *source;
	char overwrite = 'n';
	char str[LINE_LEN], new_str[LINE_LEN];
	int values, value_count, result;
	
	if (frames > 9999)
	{
		printf("\nERROR: can't number more than 9999 frames!\nterminated.\n");
		exit(1);
	}
	
	// open keyframe file 1 to be used as template
	if ((source = fopen(ini_list->keyframe, "r")) == NULL)
	{
		printf("\nERROR: unable to open keyframe file %s\nterminated.\n", ini_list->keyframe);
		exit(1);
	}
	
	// make filename
	_splitpath( filename, drive, dir, fname, ext );
	fname[4] = 0;

	values = count_list(positions);
    
	// loop for all frames
	for (frame_count = 0; frame_count < frames; frame_count++)
	{
		// create filename
		sprintf(outname, "%s%04i", fname, frame_count+start_value);
		_makepath( path_buffer, drive, dir, outname, ext );
			
		// open output file for writing
		out = open_file(path_buffer, &overwrite);
		
		// write header
		fprintf(out, "// absolute frame no.: %li of %li (starting with 0)\n", frame_count, frames);
		if (relative_frame_position == 0)
			fprintf(out, "// relative position : equivalent to keyframe %s", run_ini_list->keyframe);
		else	
			fprintf(out, "// relative position : frame no. %li after %s", relative_frame_position, run_ini_list->keyframe);
	    fprintf(out, "\n//\n// created with anim.exe, 1999, Roland Ring");
		fprintf(out, "\n//\n// contents of frame starting now!\n");
		
		// *** start - write pov contents ***
	    run_all_values = all_values; 	// go to beginning of values_list
		run_positions = positions;		// go to beginning of positions-list
		fseek(source, 0, SEEK_SET); 	// go to beginning of template file
		value_count = 0;
		line_count = 0;
		// read data
		while (fgets(str, LINE_LEN-1, source) != NULL)
		{
			// check if this is a line, where a value must be replaced
			if ((value_count < values) && (run_positions->line == line_count))
			{	// replace values
				while (run_positions->line == line_count)   // solange in Zeile bleiben bis
				{											// alle Werte ersetzt wurden
					// auch fr den zweiten Wert in einer Zeile wieder ganz vorne anfangen
					// weil die Zeile durch das Einfgen des neuen Wertes verndert wurde
					// und deswegen die Positionen nicht mehr stimmen
					pos_count = 0;  // value number x in line
					pos = 0;		// position of value in line
					len = 0;		// length of value
					do {

					    // 0 --> nothing found
						// 1 --> found value
						// 2 --> found start of comment
						abs_pos = pos + len;	// read_value liefert nur relative positionen zurck weil
						result = read_value(str+pos+len, &pos, &len); // es auch nur relative bergeben bekommt
					    pos = pos + abs_pos;
					    //printf("\nresult: %i", result);
					    if (result == 1) pos_count++;
					} while ((pos_count < run_positions->number+1) && (result == 1));
					if (result != 1)
					{
						printf("\nERROR: expected animation value in line %li in first keyframe file");
						printf("\nterminated.\n");
						exit(1);
					}
					/*printf("\nline: %lu", line_count);
					printf("\nstr: %s", str);
					printf("\npos   : %lu", pos);
					printf("\nlen   : %lu", len);
					printf("\nact_values %f", run_all_values->value_array[frame_count]);
					*/
					strncpy(new_str, str, (size_t)pos);
					new_str[pos] = 0;
					//printf("\nstep_one: new_str: #%s#", new_str);
					//getch();
					sprintf(new_str, "%s%f%s", new_str, run_all_values->value_array[frame_count], str+pos+len);
					strcpy(str, new_str);
					run_positions = (struct pos_of_numbers *)run_positions->next;
					run_all_values = (struct values_struct *)run_all_values->next;
					value_count++;
				}
				fputs(str, out);
			}			
			else
			{	// simply copy the line
				fputs(str, out);
			}
			line_count++;			
		}
						
		// *** end - write pov contents ***			
		
		// used for header data
		relative_frame_position++;
		if (relative_frame_position > run_ini_list->frames)
		{
			run_ini_list = run_ini_list->next;
			relative_frame_position = 0;
		}
		fclose( out );	
	}	
}

void main(int argc, char *argv[])
{
	struct ini_list_struct *ini_list; // Daten des INI-Files (keyframes, number of frames between)
	int value_count; // Anzahl der Animationswerte
	int file_count;  // Anzahl der Keyframes
	int start_value = 0;
	long frame_count;  // Anzahl der frames insgesamt
	struct pos_of_numbers *value_positions = NULL;  // Liste mit Positionen der Animationsparameter
	struct values_struct *value_list = NULL;	// 2-dim-Array, enthlt Werte der Animationsparameter
												// fr jeden keyframe
	struct values_struct *all_values = NULL;    // 2-dim Array fr alle Animationswerte											
	int file, value;
	char ini_file_name[LINE_LEN];
	char out_file_name[LINE_LEN];

	if (argc < 2)
	{
		printf("\nanimator 0.1, 1999, Roland Ring");
		printf("\n");
		printf("\nsyntax: anim <[path]input file> <[path]output file> [n]");
		printf("\ninput file:  text file containing names of keyframes and number of");
		printf("\n             frames between");
		printf("\noutput file: filename for output files in 4.3 notation");
		printf("\n             (eg. anim.pov --> anim0000.pov, anim0001.pov, ...)");
		printf("\nn            start numbering output files with n");
		printf("\n");
		printf("\nthat's all :-)");
		return;
	}
    
    strcpy(ini_file_name, argv[1]);
    strcpy(out_file_name, argv[2]);
	if (argc > 2)
		start_value = atoi(argv[3]);

	// ini file lesen (enthlt Namen der keyframe files und Anzahl der frames
	// zwischen den einzelnen keyframes
	ini_list = read_ini_file(ini_file_name, &file_count);

	// Praprozessieren
	// removing '//' comment lines
	printf("\npreprocessing files ...");
	prepocess(ini_list);
	printf(" done");

	// Animationsparameter finden
	value_positions = find_animation_values(value_positions, ini_list);		
	//printf("\nstep one list:");
	//count = print_list(value_positions);
	//printf("\nfound %d animation values", count);
    
	// Liste der Animationsparameter sortieren
	value_positions = sort_values(value_positions);
	//print_list(value_positions);
	
	// Animationsparameter zhlen
	value_count = count_list(value_positions);
	printf("\nfound %d animation values", value_count);
	
	// Array fr Animationswerte initialsieren
	value_list = init_value_array(value_count, file_count);
	for (file = 0; file < file_count; file++)
	{
		//printf("\n");
		for (value = 0; value < value_count; value++)
		{
			if (!set_value(value_list, 0.0, value, file))
				printf("Ups! ");
			//printf("%10.5f - ", 0.0);
		}
	}	
	
	// Animationswerte aus den keyframes auslesen
	read_animation_values(value_list, value_positions, ini_list, file_count);
	
	printf("\nfound this values\n");
	for (file = 0; file < file_count; file++)
	{
		printf("\n");
		for (value = 0; value < value_count; value++)
		{
			printf("%10.5f - ", get_value(value_list, value, file));
		}
	}	
	
	// frames insgesamt berechnen
	frame_count = count_frames(ini_list);
	printf("\nframes gesamt: %i", frame_count);
	
	// Array fr alle Animationswerte initialsieren
	all_values = init_value_array(value_count, frame_count);

	// Alle Werte der Animationsparameter berechnen / interpolieren
	printf("\ncalculating animations values ...");
	calc_values(value_list, file_count, all_values, ini_list);
	printf(" done");
	
	printf("\nwriting animation files ...");
	write_animation_files(frame_count, value_positions, all_values, ini_list, out_file_name, start_value);
	printf("\ndone.");
	
	printf("\nwriting excel file ...");
	write_excel_file(frame_count, all_values);
	printf("done.\n");
}

