#include "include/functions.inc"
#include "include/transforms.inc"

/*
* range input value a to 0 <= a <= 1
*/
#declare f_clamp = function(a) {
	min(1,max(0,a))
}

/*
* a spherical spark
*/
#declare spark = function(u,v,cu,cv,r) {
	f_clamp(1 - ((u - cu)*(u - cu) + (v - cv)*(v - cv)) / r)
}

/*
* used for dancing light sources
*/
#declare osc = function(n) {
	f_agate(0,n,0) + 0.5
}


/*
* compute horizontal position of flame particles
*/
#declare hv = function(n,ph) {
	(f_agate(n,0,0) * 2 - 1) * sin(ph) * 0.2 + 0.5
}

/*
* a particle flame
*/
#macro flame_function(particles, timeToLive, time)
	#local ttl_p = timeToLive / particles;
	#local p = int(particles / timeToLive * (time - timeToLive));
	#local p1 = p + particles;
	#while (p < p1)
		#local tl = time - p * ttl_p;
		#local fu = hv(p,tl * 3 * osc(p));
		#local tl_ttl = tl / timeToLive;
		#local fv = tl_ttl * 1;
		#local fs = pow(0.01 + sin(sqrt(tl_ttl) * pi),2) * 0.02;
		+ 0.09 * spark(u,v,fu,fv,fs)
		#local p = p + 1;
	#end
#end

/*
* start time of particle #n
*/
#macro flame_starttime(n)
	n * flame_TTL / flame_P
#end

/*
* generate flame uv pigment
*/
#macro flame_pigment(time)
	pigment {
		uv_mapping
		function {
			f_clamp(flame_function(100,1,time))
		}
		color_map {
			[0.0 color rgbt <1,0.2,0.1,1>]
			[0.5 color rgbt <1,0.2,0.1,0>]
			[0.8 color rgbt <1,1,0.2,0>]
			[1.0 color rgbt <1,1,1,0>]
		}
	}
#end

/*
* generate flame object
*/
#macro flame(camera_position,flame_position,flame_height,time,hotair)
	flame_object(camera_position,flame_position,flame_height,time,false,hotair,0)
#end

/*
* generate sheared flame object
*/
#macro sheared_flame(camera_position,flame_position,flame_height,time,hotair,shear)
	flame_object(camera_position,flame_position,flame_height,time,false,hotair,shear)
#end

/*
* generate test-flame object (to test alignment)
* produces just a translucent red rectangle (where the flame pigment would be mapped to otherwise)
*/
#macro test_flame(camera_position,flame_position,flame_height,time)
	flame_object(camera_position,flame_position,flame_height,time,true,false,0)
#end

/*
* the flame macro itself
*/	
#macro flame_object(camera_position,flame_position,flame_height,time,test,hotair,shear)
	
	union {
		
		//
		//a rectangle with uv-mapped pigment, implemented as two triangles
		//
		mesh {
			triangle {
				<-0.3,1,0>,<0.3,1,0>,<0.3,0,0>
				uv_vectors <0,1>,<1,1>,<1,0>
			}
			triangle {
				<0.3,0,0>,<-0.3,0,0>,<-0.3,1,0>
				uv_vectors <1,0>,<0,0>,<0,1>
			}
			hollow
			//
			//use flame pigment if test = false
			//
			#if (!test)
				flame_pigment(time)
			#else
				pigment { color rgbt <1,0,0,0.5> }
			#end
			finish { ambient 1 diffuse 0 }
			no_shadow
			matrix <1,0,0,
				 shear,1,0,
				 0,0,1,
				 0,0,0>	
			scale flame_height
			#local V = <camera_position.x - flame_position.x,0,camera_position.z - flame_position.z>;
			
			//
			//eorient rectangle to face the camera
			//
			transform { Reorient_Trans(-z,V) }
			
			//
			//move it to the desired position
			//
			translate flame_position
		}
		
		//
		//if test = false, add three dancing light-sources
		//
		#if (!test)
			#local LIGHT_POS = flame_position + y * flame_height/3;
			#local LIGHT_BOUNDS = <flame_height/10, flame_height/5, flame_height/10>;
			#local AREA_LIGHT_SIZE = flame_height * 0.05;
			#local FADE_DISTANCE = 2 * flame_height;
			dancing_light(LIGHT_POS,LIGHT_BOUNDS,time * 0.4 + 200,AREA_LIGHT_SIZE,FADE_DISTANCE)
			dancing_light(LIGHT_POS,LIGHT_BOUNDS,time * 0.3 + 100,AREA_LIGHT_SIZE,FADE_DISTANCE)
			dancing_light(LIGHT_POS,LIGHT_BOUNDS,time * 0.2,AREA_LIGHT_SIZE,FADE_DISTANCE)
		#end
		
		//
		//if air = true add flickering hot air effect above flame
		//
		#if (hotair)
			hot_air(flame_position,V,<flame_height * 0.4,flame_height * 2,flame_height * 0.33>,time,1/4)
		#end
		
	}
		
#end

/*
* a dancing light source
*/
#macro dancing_light(center,bounds,PHASE,AREA_LIGHT_SIZE,FADE_DISTANCE)
	light_source {
		center
		#local ph = clock * 0.4;
		color rgb <0.7,0.5,0.3> * (0.3 + 0.3*f_agate(PHASE*2+50,0,0).x)
		fade_power 2
		fade_distance FADE_DISTANCE
		area_light <AREA_LIGHT_SIZE,0,0>,<0,0,AREA_LIGHT_SIZE>,2,2 adaptive 1 jitter circular orient
		translate (<f_agate(PHASE,0,0),f_agate(0,PHASE,0),f_agate(0,0,PHASE)> - <0.5,0.5,0.5>) * bounds
	}
#end

/*
* hot air
*/
#macro hot_air(center,cam_vector,height,time,speed)
	#local delta = 0.1;
	#local s = delta;
	union {
		#while (s < 1)
			disc {
				0,cam_vector,s,s-delta
				scale height
				pigment { transmit 1 }
				interior { ior 1 - 0.05*(1-s) }
				normal {
					bumps 0.75
					scale 0.03
					translate <0,speed * time,speed * time / 2>
				}
			}
			#local s = s + delta;
		#end
		hollow
		translate <0,0.66,0>
		translate vnormalize(cam_vector) * 0.001
		translate center
		no_shadow
	}
#end