// Surface subdivision .inc file

// This file uses the Loop surface technique of surface subdivision.
// The edges are split; the split point is placed at the average
// of the ends of the edge, and the far vertices of the two triangles
// that share the edge; the end points are weighted triple.

// The original vertices are replaced with a recalculated vertex,
// taken from the original vertex and the points at the opposite ends
// of the edges that end at the vertex in question.  The weighting
// is dependent on the number of edges that meet at the vertex.

// Each of the triangles in the original surface is replaced with
// four smaller triangles.

// This whole process can be repeated until the user is satisfied with
// the results or runs out of RAM.

// This .inc file requires three arrays to be declared before calling
// this macro:

// sssP[] contains the <x,y,z> locations of the vertices.
// sssE[][4] contains indices:
//   sssE[][0] contains the index to the starting point.
//   sssE[][1] contains the index to the end point.
//   sssE[][2] contains the index to one of the bordering triangles.
//   sssE[][3] contains the index to the other bordering triangle.
//
// sssT[][10] contains indices:
//   sssT[][0] contains the index to the first corner of the
//     triangle.
//   sssT[][1] contains the index to the second corner of the
//     triangle.
//   sssT[][2] contains the index to the third corner of the
//     triangle.
//   The three points *must* be clockwise, or the normals will not be
//     calculated properly!
//   sssT[][7] contains the index to the edge connecting the
//     first and second points.
//   sssT[][8] contains the index to the edge connecting the
//     second and third points.
//   sssT[][9] contains the index to the edge connecting the
//     first and third points.
//
// The macro PrepMesh() will set up the cross-referencing needed.

#macro Divide(Smoothing)
#local cP=dimension_size(sssP,1); // # of points
#local cE=dimension_size(sssE,1); // # of Edges
#local cT=dimension_size(sssT,1); // # of triangles
#ifdef (sssM)
  #local cM=dimension_size(sssM,1); // # of UVW mapping points
#else
  #local cM=0;
#end  

// count the number of new mapping points needed
#local cNM=0;
#local iI=0; #while(iI<cT)
  #if (sssT[iI][4]!=-1) #local cNM=cNM+3; #end
#local iI=iI+1; #end

#if(cNM!=0) #local NewMap=array[cNM] #end
#local pNM=0;

// first we calculate the edge & vertex points

#local EdgePoints=array[cE]
#local PointSum=array[cP]
#local PointCnt=array[cP]

#local iI=0; #while(iI<cP)
  #local PointSum[iI]=<0,0,0>;
  #local PointCnt[iI]=0;
#local iI=iI+1; #end

#local iI=0; #while(iI<cE)

#local pA=sssE[iI][0];
#local pB=sssE[iI][1];

//add the end points to the running sum for the opposite end
//and remember how many have been added, too
#local PointSum[pA]=PointSum[pA]+sssP[pB];
#local PointSum[pB]=PointSum[pB]+sssP[pA];
#local PointCnt[pA]=PointCnt[pA]+1;
#local PointCnt[pB]=PointCnt[pB]+1;

// get the far vertices from the bordering triangles
#local tA=sssE[iI][2];
  #if (tA=-1) #local tA=sssE[iI][3]; #end
#local tB=sssE[iI][3];
  #if (tB=-1) #local tB=sssE[iI][2]; #end

#local pC=sssT[tA][0];
#if (pC=pA | pC=pB)
  #local pC=sssT[tA][1];
#end
#if (pC=pA | pC=pB)
  #local pC=sssT[tA][2];
#end

#local pD=sssT[tB][0];
#if (pD=pA | pD=pB)
  #local pD=sssT[tB][1];
#end
#if (pD=pA | pD=pB)
  #local pD=sssT[tB][2];
#end

#local EdgePoints[iI]=(sssP[pA]*3+sssP[pB]*3
  +sssP[pC]*Smoothing+sssP[pD]*Smoothing)/(Smoothing*2+6);

#local iI=iI+1; #end

// then we calculate the vertex points

#local iI=0; #while(iI<cP)

#local iN=PointCnt[iI];

#local sA=.625-(cos(2*pi/iN)*2+3)*(cos(2*pi/iN)*2+3)*.015625;

#local sA=(iN*(1-sA))/sA;

#local PointSum[iI]=(sssP[iI]*sA+PointSum[iI]*Smoothing)/(sA+iN*Smoothing);

#local iI=iI+1; #end

// then we create all of the edges

#local cEN=cE*2+cT*3; // existing edges are split into two,
                      // and each triangle spawns three new edges

#local NewEdge=array[cEN][4]

#local iJ=0;
#local iI=0; #while(iI<cE)

//  P0 ---------------- P1  old edge
//  P0 ------ NP ------ P1  new edges made from edge

#local NewEdge[iJ][0]=sssE[iI][0];
#local NewEdge[iJ][1]=iI+cP;
#local NewEdge[iJ][2]=-1;
#local NewEdge[iJ][3]=-1;
#local iJ=iJ+1;
#local NewEdge[iJ][0]=sssE[iI][1];
#local NewEdge[iJ][1]=iI+cP;
#local NewEdge[iJ][2]=-1;
#local NewEdge[iJ][3]=-1;
#local iJ=iJ+1;

#local iI=iI+1; #end

// then we create the new interior edges and triangles while we're at it

#local cTN=cT*4; // each triangle is replaced by 4 new ones

#local NewTri=array[cTN][10]
#local iT=0;

#local iI=0; #while (iI<cT)

#local NewEdge[iJ][0]=sssT[iI][7]+cP;
#local NewEdge[iJ][1]=sssT[iI][9]+cP;
#local NewEdge[iJ][2]=-1;
#local NewEdge[iJ][3]=-1;
#local iJ=iJ+1;
#local NewEdge[iJ][0]=sssT[iI][8]+cP;
#local NewEdge[iJ][1]=sssT[iI][7]+cP;
#local NewEdge[iJ][2]=-1;
#local NewEdge[iJ][3]=-1;
#local iJ=iJ+1;
#local NewEdge[iJ][0]=sssT[iI][9]+cP;
#local NewEdge[iJ][1]=sssT[iI][8]+cP;
#local NewEdge[iJ][2]=-1;
#local NewEdge[iJ][3]=-1;
#local iJ=iJ+1;

// create interior triangle
#local NewTri[iT][0]=sssT[iI][7]+cP;
#local NewTri[iT][1]=sssT[iI][8]+cP;
#local NewTri[iT][2]=sssT[iI][9]+cP;
#local NewTri[iT][3]=sssT[iI][3];
#if(sssT[iI][4]=-1)
  #local NewTri[iT][4]=-1;
  #local NewTri[iT][5]=-1;
  #local NewTri[iT][6]=-1;
#else
  #local NewMap[pNM]=  (sssM[sssT[iI][4]]+sssM[sssT[iI][5]])/2;
  #local NewMap[pNM+1]=(sssM[sssT[iI][5]]+sssM[sssT[iI][6]])/2;
  #local NewMap[pNM+2]=(sssM[sssT[iI][4]]+sssM[sssT[iI][6]])/2;
  #local NewTri[iT][4]=cM+pNM;
  #local NewTri[iT][5]=cM+pNM+1;
  #local NewTri[iT][6]=cM+pNM+2;
#end
#local NewTri[iT][7]=iJ-2;
#local NewTri[iT][8]=iJ-1;
#local NewTri[iT][9]=iJ-3;
#local NewEdge[iJ-1][2]=iT;
#local NewEdge[iJ-2][2]=iT;
#local NewEdge[iJ-3][2]=iT;
#local iT=iT+1;

// create triangle at first vertex
#local iEa=sssT[iI][7]*2;
#if(NewEdge[iEa][0]!=sssT[iI][0]) #local iEa=iEa+1; #end
#local iEb=sssT[iI][9]*2;
#if(NewEdge[iEb][0]!=sssT[iI][0]) #local iEb=iEb+1; #end

#local NewTri[iT][0]=sssT[iI][0];
#local NewTri[iT][1]=sssT[iI][7]+cP;
#local NewTri[iT][2]=sssT[iI][9]+cP;
#local NewTri[iT][3]=sssT[iI][3];
#if(sssT[iI][4]=-1)
  #local NewTri[iT][4]=-1;
  #local NewTri[iT][5]=-1;
  #local NewTri[iT][6]=-1;
#else
  #local NewTri[iT][4]=sssT[iI][4];
  #local NewTri[iT][5]=cM+pNM;
  #local NewTri[iT][6]=cM+pNM+2;
#end
#local NewTri[iT][7]=iEa;
#local NewTri[iT][8]=iJ-3;
#local NewTri[iT][9]=iEb;

#local NewEdge[iJ-3][3]=iT;
#if (NewEdge[iEa][2]=-1)
  #local NewEdge[iEa][2]=iT;
#else
  #local NewEdge[iEa][3]=iT;
#end
#if (NewEdge[iEb][2]=-1)
  #local NewEdge[iEb][2]=iT;
#else
  #local NewEdge[iEb][3]=iT;
#end
#local iT=iT+1;

// create triangle at second vertex
#local iEa=sssT[iI][8]*2;
#if(NewEdge[iEa][0]!=sssT[iI][1]) #local iEa=iEa+1; #end
#local iEb=sssT[iI][7]*2;
#if(NewEdge[iEb][0]!=sssT[iI][1]) #local iEb=iEb+1; #end

#local NewTri[iT][0]=sssT[iI][1];
#local NewTri[iT][1]=sssT[iI][8]+cP;
#local NewTri[iT][2]=sssT[iI][7]+cP;
#local NewTri[iT][3]=sssT[iI][3];
#if(sssT[iI][4]=-1)
  #local NewTri[iT][4]=-1;
  #local NewTri[iT][5]=-1;
  #local NewTri[iT][6]=-1;
#else
  #local NewTri[iT][4]=sssT[iI][5];
  #local NewTri[iT][5]=cM+pNM+1;
  #local NewTri[iT][6]=cM+pNM;
#end
#local NewTri[iT][7]=iEa;
#local NewTri[iT][8]=iJ-2;
#local NewTri[iT][9]=iEb;

#local NewEdge[iJ-2][3]=iT;
#if (NewEdge[iEa][2]=-1)
  #local NewEdge[iEa][2]=iT;
#else
  #local NewEdge[iEa][3]=iT;
#end
#if (NewEdge[iEb][2]=-1)
  #local NewEdge[iEb][2]=iT;
#else
  #local NewEdge[iEb][3]=iT;
#end
#local iT=iT+1;

// create triangle at third vertex
#local iEa=sssT[iI][9]*2;
#if(NewEdge[iEa][0]!=sssT[iI][2]) #local iEa=iEa+1; #end
#local iEb=sssT[iI][8]*2;
#if(NewEdge[iEb][0]!=sssT[iI][2]) #local iEb=iEb+1; #end

#local NewTri[iT][0]=sssT[iI][2];
#local NewTri[iT][1]=sssT[iI][9]+cP;
#local NewTri[iT][2]=sssT[iI][8]+cP;
#local NewTri[iT][3]=sssT[iI][3];
#if(sssT[iI][4]=-1)
  #local NewTri[iT][4]=-1;
  #local NewTri[iT][5]=-1;
  #local NewTri[iT][6]=-1;
#else
  #local NewTri[iT][4]=sssT[iI][6];
  #local NewTri[iT][5]=cM+pNM+2;
  #local NewTri[iT][6]=cM+pNM+1;
#end
#local NewTri[iT][7]=iEa;
#local NewTri[iT][8]=iJ-1;
#local NewTri[iT][9]=iEb;

#local NewEdge[iJ-1][3]=iT;
#if (NewEdge[iEa][2]=-1)
  #local NewEdge[iEa][2]=iT;
#else
  #local NewEdge[iEa][3]=iT;
#end
#if (NewEdge[iEb][2]=-1)
  #local NewEdge[iEb][2]=iT;
#else
  #local NewEdge[iEb][3]=iT;
#end
#local iT=iT+1;

#if(sssT[iI][4]!=-1) #local pNM=pNM+3; #end

#local iI=iI+1; #end

// All triangles are generated; now save old arrays, re-initialize them
// to the new larger size, and restore the data

#local cPN=cP+cE;

#undef sssP
#declare sssP=array[cPN]

#local iI=0; #while (iI<cP)
#declare sssP[iI]=PointSum[iI];
#local iI=iI+1; #end

#while (iI<cPN)
#declare sssP[iI]=EdgePoints[iI-cP];
#local iI=iI+1; #end

#undef sssE
#declare sssE=array[cEN][4]

#local iI=0; #while(iI<cEN)
#declare sssE[iI][0]=NewEdge[iI][0];
#declare sssE[iI][1]=NewEdge[iI][1];
#declare sssE[iI][2]=NewEdge[iI][2];
#declare sssE[iI][3]=NewEdge[iI][3];
#local iI=iI+1; #end

#undef sssT
#declare sssT=array[cTN][10]

#local iI=0; #while(iI<cTN)
#declare sssT[iI][0]=NewTri[iI][0];
#declare sssT[iI][1]=NewTri[iI][1];
#declare sssT[iI][2]=NewTri[iI][2];
#declare sssT[iI][3]=NewTri[iI][3];
#declare sssT[iI][4]=NewTri[iI][4];
#declare sssT[iI][5]=NewTri[iI][5];
#declare sssT[iI][6]=NewTri[iI][6];
#declare sssT[iI][7]=NewTri[iI][7];
#declare sssT[iI][8]=NewTri[iI][8];
#declare sssT[iI][9]=NewTri[iI][9];
#local iI=iI+1; #end

#if (cNM!=0)
  #local tMap=array[cM+cNM]
  #local iI=0;#while(iI<cM)
    #local tMap[iI]=sssM[iI];
  #local iI=iI+1; #end
  #local iI=0; #while(iI<cNM)
    #local tMap[iI+cM]=NewMap[iI];
  #local iI=iI+1; #end
  #undef sssM
  #declare sssM=array[cM+cNM]
  #local iI=0; #while(iI<cM+cNM)
    #declare sssM[iI]=tMap[iI];
  #local iI=iI+1; #end
#end

#end // end of macro call
