// Persistence of Vision Ray Tracer Scene Description File
// File: 2_lonely.pov
// Vers: 3.5
// Desc: Being too lonely...
// Date: 14.08.02 (dd.mm.yy)
// Auth: Tim Nikias Wenclawiak
// Last Update: 29.8.02 (dd.mm.yy)

//Required files:
// "2lonnote.png" - Image map for the note
// "pen_impr.png" - Company name on the cap of the pen
// "caroling.ttf" - Carolingia True-Type-Font

//Description:
// Having spent way too much time alone (and we
// don't want to discuss if it was him staying
// alone, or if it was him being left alone, or
// perhaps just his feeling of being left
// alone), he made his choice...

//Homepage:
// www.digitaltwilight.de/no_lights
//Email:
// Tim.Nikias@gmx.de

#declare Radiosity      = off;
#declare Photons        = off;

#declare Save_Data      = off; //Switch to on to save Radiosity/Photon data to disk
#declare Load_Data      = off; //Switch to on to load previously saved lighting data

#declare Area_Light     = on;
#declare Area_Light_Size= 20;
#declare Area_Lights    = 24;

#declare Reflective_Blur_Samples= 1; //Use higher values for blurred reflection on the floor

#declare Random_Stream = seed(3);


//Object-Switches
#declare Note_Object    = on;
#declare Glass_Object   = on;
#declare Water_Object   = on;
#declare Pills          = on;
#declare Pillbox        = on;
#declare PillboxCap     = on;
#declare Pen            = on;
#declare PenCap         = on;

#declare Artist         = on;

#declare Glass_Filled           = .7;
#declare HF_Detail              = 400;
//Spiral details on pen and pillbox 
//(pillbox, due to phtons, has different details)
#declare Pen_Spiral_Detail=200;
#if (Photons)
 #declare Pillbox_Spiral_Detail = 80; #else
 #declare Pillbox_Spiral_Detail = 240; #end

#declare Transparencies = on;
#declare Light = on;

/**********************************/

global_settings{
 max_trace_level 25
 assumed_gamma 1
 ambient_light 0
 #if (Radiosity)
  radiosity{
   always_sample off
   
   #if (Save_Data)
    save_file "2lonely.rad"
    pretrace_start .2
    pretrace_end .05
   #end
   #if (Load_Data)
    load_file "2lonely.rad"
    pretrace_start 1
    pretrace_end 1
   #end
   
   count 800
   error_bound .5
   gray_threshold 0
   recursion_limit 6
   }
 #end
 #if (Photons)
  photons{
   spacing .008
   max_trace_level 25
   #if (Save_Data)
    save_file "2lonely.pho"
   #end
   #if (Load_Data)
    load_file "2lonely.pho"
   #end
   }
  #end
 }

/************************/

camera{
 location <0,6,-8> look_at <-.5,0,0>
 }
 
/**********************************/
#if (Light)
//Spotlight!
light_source{
 <-30,70,-50> rgb 2
 spotlight point_at <0,0,0> radius 1 falloff 4 tightness 60
 fade_power 2 fade_distance 40
 #if (Area_Light)
  area_light x*Area_Light_Size,y*Area_Light_Size,Area_Lights,Area_Lights orient circular adaptive 3
 #end
 photons{reflection on refraction on}
 }
#end

/**********************************/
//Artist name

#if (Artist)
 #declare Artist_Text=
  text{ttf "caroling.ttf" "Tim Nikias 2002" .8,0 hollow
   interior{
    fade_power 2 fade_distance .2
    media{emission 1 method 3 intervals 1 samples 5,20
     density{gradient x scale 10 translate x*-5
      color_map{[0 rgb <1,0,0>][.2 rgb <1,.5,0>][.4 rgb <1,1,0>][.6 rgb <0,1,0>][.8 rgb <0,0,1>][1 rgb <.5,0,1>]}
      }
     }
    }
   texture{
    pigment{rgbt <0,0,0,1>}
    finish{
     reflection{0,1 fresnel exponent .9}
     ambient 0 diffuse 1.2
     }
    }
   translate -z*.2
   scale .2
   }

#declare Artist_Min=min_extent(Artist_Text);
#declare Artist_Max=max_extent(Artist_Text);

object{Artist_Text translate -(Artist_Max-Artist_Min)/2*x
 rotate x*90 rotate y*-3 translate <3,0,-4.75>
 no_reflection //no need for reflections
 no_shadow     //and no need for shadows
 }

#end

/**********************************/
//Positioning/Transformations

//Standing glass with water
#declare Glass_Transform=
transform{
 scale 1.5
 translate <1.25,.05,1.25>
 }

//Note
#declare Note_Transform=
transform{
 rotate y*-21
 translate <-.24,0,-1.25>
 }

//Lying pillbox
#declare Pillbox_Transform=
transform{
 translate z*.75 rotate x*-87
 rotate y*-44
 scale .75
 translate <-3.75,.01,-.25>
 }

#declare PillboxCap_Transform=
transform{
 rotate x*-180 translate <0,.4,0>
 scale .75
 translate <-2.24,0.01,1.15>
 }

//Pen
#declare Pen_Transform=
transform{
 rotate x*-32
 translate y*1.1
 scale .2
 rotate y*-35
 translate <3,0,-.25> 
 }

#declare PenCap_Transform=
transform{
 rotate x*-110 //Rotation to lie on clip
 translate y*1.15
 scale .2
 rotate y*35
 translate <-2,0,-4>
 }

/**********************************/
//Materials

//Floor
#declare Base_Pigment_1 = pigment{crackle rotate y*-62 turbulence .6 lambda 3 omega .6 octaves 15 color_map{[0 rgb .1][1 rgb .4]}}
#declare Base_Pigment_2 = pigment{spiral1 5 rotate <90,53,23> turbulence .3 lambda 2 omega .7 octaves 3 color_map{[0 rgb .2*<1,.9.9>][1 rgb .5*<1,.8,.8>]}}
#declare Base_Pigment   = pigment{wood scale 1 rotate <32,76,42> turbulence .4 lambda 3.5 omega .7 octaves 6 pigment_map{[0 Base_Pigment_1][1 Base_Pigment_2]}}

#declare Base_Normal_GridHeight = .1;
#declare Base_Normal_YRotation  = 35;
#declare Base_Normal_1 = normal{gradient x Base_Normal_GridHeight slope_map{[0 <0,0>][.4 <0,0>][.425 <0,.5>][.5 <1,0>][.575 <1,-.5>][.6 <0,0>][1 <0,0>]} rotate y*Base_Normal_YRotation }
#declare Base_Normal_2 = normal{gradient x Base_Normal_GridHeight slope_map{[0 <0,0>][.4 <0,0>][.425 <0,.5>][.5 <1,0>][.575 <1,-.5>][.6 <0,0>][1 <0,0>]} rotate y*(37+Base_Normal_YRotation) }
#declare Base_Normal   = normal{average scale .2 normal_map{[1 Base_Normal_1][1 Base_Normal_2]}}

#declare Blurred_Finish=
 finish{
  ambient 0
  reflection{.85,1}
  specular .2 roughness .002
  brilliance 5
  }

#declare Base_Material=
material{
 interior{ior 1.23}
 texture{average texture_map{
  #declare Sample_Count=0; #while (Sample_Count<Reflective_Blur_Samples)
  [1 pigment{Base_Pigment}
     //Optional blurred reflection
     #if (Reflective_Blur_Samples!=1)
      normal{average normal_map{[1 bozo .15 scale .1 translate rand(Random_Stream)*10][1 Base_Normal]}}
     #else
      normal{Base_Normal}
     #end
     finish{Blurred_Finish}]
  #declare Sample_Count=Sample_Count+1; #end
   }
  }
 }

#declare Glass_Material=
material{
 interior{
  ior 1.64
  fade_power 2 fade_distance .2 fade_color <0,.2,.3>
  }
 texture{
  #if (Transparencies)
   pigment{rgbt <.6,.8,1,.99>} #else
   pigment{rgb 1} #end
  finish{
   specular 1 roughness .001
   ambient 0
   brilliance 3.5
   reflection{0,.8 exponent .9 fresnel}
   conserve_energy
   }
  }
 }

#declare Water_Material=
material{
 interior{ior 1.33}
 texture{
  #if (Transparencies)
   pigment{rgbt 1} #else
   pigment{rgb <.7,.7,1>} #end
  finish{specular 1 roughness .001 ambient 0 }
  }
 }

#declare Pillbox_Material=
material{
 interior{
  ior 1.29
  fade_power 2 fade_distance .15
  }
 texture{
  #if (Transparencies)
   pigment{rgbt <1,.35,0,.9>} #else
   pigment{rgb <1,.35,0>} #end
  finish{
   diffuse 1.1
   specular 1 roughness .003
   reflection{0,1 exponent .97 fresnel}
   conserve_energy
   brilliance 2
   }
  }
 }

#declare PillboxCap_Material=
material{
 interior{ior 1.31}
 texture{
  pigment{rgb <1,1,1>}
  finish{diffuse 1.1 ambient 0 specular 1 roughness .001 reflection{0,.3 falloff 2 exponent .97}}
  }
 }

//First half of pill...
#declare Pill_1_Texture=
 texture{
  pigment{rgb x}
  finish{diffuse 1 ambient .3 reflection{.3,1 fresnel} specular .8 roughness .003 brilliance 2}
  }

//...second half
#declare Pill_2_Texture=
 texture{
  pigment{rgb <1,1,0>}
  finish{diffuse 1 ambient .3 reflection{.3,1 fresnel} specular .8 roughness .003 brilliance 2}
  }
 
#declare Pill_Interior=interior{ior 1.15} 

//Pen
#declare Pen_Black_Material=
material{
 interior{ior 1.43}
 texture{
  pigment{rgb 0}
  finish{specular 1 roughness .004 brilliance 5 reflection{.2,1 falloff 2 exponent .98 fresnel}}
  }
 }

#declare Pen_Marble_Material=
material{
 interior{ior 1.43}
 texture{
  pigment{wood scale 1.5 turbulence .3 lambda 3 omega .6 octaves 6 color_map{[0 rgb <1,.97,.97>*<.2,.2,1>][1 rgb <.5,.5,.52>*<.2,.2,1>]} rotate <34,76,0>}
  finish{specular .8 roughness .005 brilliance 3 reflection{0,.5 fresnel}}
  }
 }

#declare Pen_Gold_Pigment=pigment{rgb <1,.75,.2>}
#declare Pen_Gold_Finish=finish{specular 2 roughness .002 brilliance 6 reflection{.3,1 metallic .7 falloff 2 exponent .9}}
#declare Pen_Gold_Material=
material{
 interior{ior 1}
 texture{
  pigment{Pen_Gold_Pigment}
  finish{Pen_Gold_Finish}
  }
 }

/**********************************/
//Floor (a box to speed reflections, most rays will miss a box, but not a plane)
box{<-9,-.1,-6>,<9,-.001,9>
 material{Base_Material}
 }
 
/**********************************/
//Pigments used to model the note with its wrinkles...
#declare Paper_1a=pigment{bozo scale .35 color_map{[0 rgb 0][1 rgb 1]}}
#declare Paper_1b=pigment{crackle scale .15 turbulence .3 lambda 2 omega .7 octaves 4 color_map{[0 rgb 1][1 rgb 0]}}

#declare Paper_Flat=pigment{rgb 0}
#declare Paper_1=pigment{average pigment_map{[1 Paper_1a][.5 Paper_1b]}}
#declare Paper_2=pigment{average pigment_map{[1 Paper_Flat][.2 Paper_1b]}}

#declare Paper=pigment{spherical scale <2,1,1>*.75 rotate z*-45 pigment_map{[0 Paper_1 rotate z*45 scale <.5,1,1>][.6 Paper_2][1 Paper_2]} translate <1,0,0>}

#declare HF_Size=<4,.25,4>;

#if (Note_Object)

#declare HF = function{ pigment{Paper}}

 height_field{
  function HF_Detail,HF_Detail{HF(x,y,z).x}
  pigment{ image_map{png "2lonnote.png"} rotate x*90}
  finish{diffuse 1 ambient 0}
  translate <-.5,0,-.5>
  scale HF_Size
  transform{Note_Transform}
  }

#end

/**********************************/
//The Glass

#declare Glass_Wall=.04;
#declare Glass_Base=.1;

#if (Glass_Object)
merge{
 difference{
  cylinder{y*Glass_Base,y*2.5,.75}
  cylinder{y*(Glass_Base+Glass_Wall/2),y*2.6,.75-Glass_Wall}
  //Inner base rim
  cylinder{y*(Glass_Base+Glass_Wall/2),y*(Glass_Base+.0001),.75-Glass_Wall*1.5}
  torus{.75-Glass_Wall*1.5,Glass_Wall/2 translate y*(Glass_Base+Glass_Wall/2)}
  }
 //Upper edge
 torus{.75-Glass_Wall/2,Glass_Wall/2 translate y*2.5}
 //Base rim
 intersection{
  torus{.75-Glass_Base,Glass_Base translate y*Glass_Base}
  cylinder{0,y*Glass_Base,.75}
  }
 cylinder{y*(Glass_Base+.001),y*0,.75-Glass_Base}
 
 material{Glass_Material}
 transform{Glass_Transform}
 photons{target refraction on reflection off collect off}
 }

#end

/***********/
//Water:
#if (Water_Object) 

#declare Tension_Height=.05; 

//Remaining water in glass
#if (Glass_Filled!=0)
merge{
 cylinder{y*(Glass_Base+Glass_Wall/2+.0001),y*Glass_Filled,.75-Glass_Wall-.0001}
 //Inner base rim
 cylinder{y*(Glass_Base+Glass_Wall/2+.0002),y*(Glass_Base+.0001),.75-Glass_Wall*1.5-.0001}
 torus{.75-Glass_Wall*1.5,Glass_Wall/2-.0001 translate y*(Glass_Base+Glass_Wall/2+.0001)}
 //Watertension
 difference{
  difference{
   cylinder{y*Glass_Filled,y*(Glass_Filled+Tension_Height),.75-Glass_Wall-.0001}
   cylinder{y*(Glass_Filled-.1),y*(Glass_Filled+Tension_Height+.01),.75-Glass_Wall-.0001-Tension_Height}
   }
  torus{.75-Glass_Wall-.0001-Tension_Height,Tension_Height translate y*(Glass_Filled-Tension_Height/100+Tension_Height) }
  }
 material{Water_Material}
 transform{Glass_Transform}
 photons{refraction on reflection on collect off}
 }
#end

//Water-Droplets
intersection{
 blob{
  threshold .7
  #declare Drops=0;
  #while (Drops < 30)
   #declare Drop_Pos=<.75-Glass_Wall,sqrt(rand(Random_Stream))*(2.5-Glass_Filled-.25)+Glass_Filled,0>;
   sphere{0,.02+.1*rand(Random_Stream),1
    scale <.5,1,1>+y*rand(Random_Stream)
    translate Drop_Pos
    rotate y*360*rand(Random_Stream)
    } 
  #declare Drops=Drops+1;
  #end
  }
 cylinder{0,y*2.5,.75-Glass_Wall-.001} 
 material{Water_Material}
 photons{reflection on refraction on collect off}
 transform{Glass_Transform}
 }

#end

/**********************************/
//The pills
#declare Pill=
union{
 sphere{0,1 scale <1,.25,1> translate y*.25}
 cylinder{y*.25,y*.5,1}
 difference{
  sphere{0,1 scale <1,.25,1> translate y*.5}
  cylinder{z*-1,z*1,.2 scale <1,.5,1> translate y*.75}
  }
 scale .15
 pigment{rgb 1}
 finish{diffuse 1 brilliance .98 specular .2 roughness .2}
 }

#declare Pill=
union{
 sphere{0,1 scale .25 translate <.5,.25,0> texture{Pill_1_Texture}}
 cylinder{<-.5,.25,0><.5,.25,0>,.25 texture{Pill_1_Texture}}
 
 cone{<-.5,.25,0>,.25,<0,.25,0>,.26 texture{Pill_2_Texture}}
 torus{.255,.005 rotate z*90 translate y*.25 texture{Pill_2_Texture}}
 sphere{0,1 scale .25 translate <-.5,.25,0> texture{Pill_2_Texture}}
 scale .35
 interior{Pill_Interior}
 }

#if (Pills)
//From left to right
object{Pill
 rotate y*33
 translate <-3.5,0,-1.6>
 }

object{Pill
 rotate y*(63+180)
 translate <-2.5,0,-2>
 }

object{Pill
 rotate y*-63
 translate <.6,0,-3.7>
 }

#end

/**********************************/
//Pillbox and -cap
#declare Pillbox_Wall=.08;
#declare Pillbox_Base=.08;
#declare Pillbox_Spiral_Radius=.04;

#if (Pillbox)


merge{
 #if (Pillbox_Spiral_Detail>1)
 //Spiral
 #declare C=0;
 #while (C<Pillbox_Spiral_Detail)
  #declare Pos_2=vrotate(x*.75+y*(.275-Pillbox_Spiral_Radius)/Pillbox_Spiral_Detail*C,y*360*2/Pillbox_Spiral_Detail*C);
  #ifdef( Pos_1)
   sphere{Pos_1,Pillbox_Spiral_Radius translate y*(2-.3)}
   cylinder{Pos_1,Pos_2,Pillbox_Spiral_Radius translate y*(2-.3)}
  #end
  #declare Pos_1=Pos_2;
 #declare C=C+1;
 #end
 sphere{Pos_1,Pillbox_Spiral_Radius translate y*(2-.3)}
 #end

 difference{
  cylinder{y*Pillbox_Base,y*2,.75}
  cylinder{y*(Pillbox_Base-.01),y*2.1,.75-Pillbox_Wall}
  //Inner base rim
  cylinder{y*(Pillbox_Base+Pillbox_Wall/2),y*Pillbox_Base+.0001,.75-Pillbox_Wall*1.5}
  torus{.75-Pillbox_Wall*1.5,Pillbox_Wall/2 translate y*(Pillbox_Base+Pillbox_Wall/2)}
  }
 //Upper edge
 torus{.75-Pillbox_Wall/2,Pillbox_Wall/2 translate y*2}
 //Base rim
 intersection{
  torus{.75-Pillbox_Base,Pillbox_Base translate y*Pillbox_Base}
  cylinder{y*-.0001,y*Pillbox_Base,.7501}
  }
 cylinder{y*(Pillbox_Base+.001),y*.0001,.75-Pillbox_Base}
 //Under-Lid rim
 difference{
  cylinder{y*(2-.3),y*(2-.3-Pillbox_Wall),.82}
  cylinder{y*(2-.2),y*(2-.4-Pillbox_Wall),.75-Pillbox_Wall*.95}
  }
 torus{.82,Pillbox_Wall/2 translate y*(2-.3-Pillbox_Wall/2)}
 material{Pillbox_Material}
 transform{Pillbox_Transform}
 photons{target .5 refraction on reflection on collect off}
 }
#end


//Pillbox-Cap
#declare PillboxCap_Grips=72;

#if (PillboxCap)

union{
 #if (Pillbox_Spiral_Detail>1)
 //Spiral
 #declare C=0;
 #while (C<Pillbox_Spiral_Detail)
  #declare Pos_2=vrotate(x*.75+y*(.275-Pillbox_Spiral_Radius)/Pillbox_Spiral_Detail*C,y*360*2/Pillbox_Spiral_Detail*C);
  #ifdef( Pos_1)
   sphere{Pos_1,Pillbox_Spiral_Radius}
   cylinder{Pos_1,Pos_2,Pillbox_Spiral_Radius}
  #end
  #declare Pos_1=Pos_2;
 #declare C=C+1;
 #end
 sphere{Pos_1,Pillbox_Spiral_Radius}
 #end

 difference{
  cylinder{0,y*.3,.85}
  cylinder{y*-.1,y*.2,.75}
  }
 torus{.8,.05}
 sphere{0,1 scale <.85,.1,.85> translate y*.3}
 #declare C=0; #while (C<PillboxCap_Grips)
  sphere{0,1 scale <.02,.15,.02> translate y*.15
   translate x*.85 rotate y*360/PillboxCap_Grips*C
   }
 #declare C=C+1; #end
 material{PillboxCap_Material}
 transform{PillboxCap_Transform}
 photons{target .25 reflection on refraction off}
 }
#end

/**********************************/
//The pen

#declare Pen_Spiral_Radius=.05;

#if (Pen)
union{
 //Black plastic parts
 union{
  //Rear end
  sphere{0,1 scale <.4,1,1> translate x*10}
  cone{x*10,1 x*7.5,1.1}
  //Before pen-end
  cylinder{x*-2,x*-4,1.1}
  cone{x*-4.5,1.1-Pen_Spiral_Radius, x*-7.5,.9}
  torus{.9,.1 rotate z*90 translate x*-7.4}
  //Underneath the tip
  difference{
   cone{x*-7.5,.6, x*-10.5,.7}
   cylinder{<0,0,-1>,<0,0,1>,.9 scale <2,1,1> translate <-10.5,-.2,0>}
   box{<-11,-.3,-1><-7,-1,1>}
   box{<-10,0,-1><-11,1,1>}
   }
   #declare C=0;
   #while (C<=5)
    torus{.6,.1 rotate z*90 translate x*(-7.5-1.2/5*C)-y*.3/5*C scale <1,1-.7/5*C,1>}
   #declare C=C+1;
   #end
  //Spiral
  #if (Pen_Spiral_Detail>1)
  #declare C=0;
  #while (C<Pen_Spiral_Detail)
   #declare Pos_2=vrotate(z*(1.1-Pen_Spiral_Radius)-x*(.5-Pen_Spiral_Radius)/Pen_Spiral_Detail*C-x*4,x*360*3/Pen_Spiral_Detail*C);
   #ifdef( Pos_1)
    sphere{Pos_1,Pen_Spiral_Radius}
    cylinder{Pos_1,Pos_2,Pen_Spiral_Radius}
   #end
   #declare Pos_1=Pos_2;
  #declare C=C+1;
  #end
  sphere{Pos_1,Pen_Spiral_Radius}
  #end
  cylinder{x*-4,x*-4.5,1.1-Pen_Spiral_Radius}
  material{Pen_Black_Material}
  }
 //Marble part
 cylinder{x*7.5,x*-2,1.1
  material{Pen_Marble_Material}
  }
 //Golden parts
 union{
  difference{
   cone{x*-7.5,.7, x*-10.5,.8}
   cone{x*-7.5001,.6, x*-10.5001,.7}
   cylinder{<0,0,-1>,<0,0,1>,.9 scale <2,1,1> translate <-10.5,-.1,0>}
   box{<-11,0,-1><-7,-1,1>}
   //Inkhole
   cylinder{<-8.5,0,0>,<-8.5,1,0>,.15}
   box{<-8.5,0,-.01>,<-11,1,.01>}
   }
  sphere{<-10.32,.75,0>,.1} 
  material{Pen_Gold_Material}
  photons{target .25 reflection on refraction off collect off}
  }
 transform{Pen_Transform}
}

#end

/**********************************/
#declare Clip_Right = vrotate(<0,1.05,0>,x* 20);
#declare Clip_Left  = vrotate(<0,1.05,0>,x*-20);

//The pen's cap
#if (PenCap)
union{
 //Black plastic parts
 union{
  //Main Hull
  difference{
   cone{x*-1.5,1.2, x*-11,1.1}
   cone{x*-1.4,1.1 x*-11.0001,1}
   }
  //Tip
  cone{x*-11,.85,x*-11.6,.9}
  sphere{0,1 scale <.1,.9,.9> translate x*-11.6}
  cylinder{x*-11.5,x*-11.8,.8}
  sphere{0,1 scale <.1,.8,.8> translate x*-11.8}
  material{Pen_Black_Material}
  }
 //Golden parts
 union{
  torus{1.05,.1 rotate z*90 translate x*-11}
  torus{1.15,.05 rotate z*90 translate x*-1.2}
  //Clip
  //Left on base
  cylinder{<-11,0,0>,<-11,.8,0>,.1 translate Clip_Left} 
  sphere{<-11,.8,0>+Clip_Left,.1}
  //Right on base
  cylinder{<-11,0,0>,<-11,.8,0>,.1 translate Clip_Right} 
  sphere{<-11,.8,0>+Clip_Right,.1}
  //Base-Connectors
  cylinder{<-11,.8,0>+Clip_Left,<-11,.8,0>+Clip_Right,.1}
  //Main stem
  cylinder{<-11,.8,0>,<-5,.7,0>,.1 translate Clip_Left}
  sphere{<-5,.7,0>+Clip_Left,.1}
  cylinder{<-11,.8,0>,<-5,.7,0>,.1 translate Clip_Right}
  sphere{<-5,.7,0>+Clip_Right,.1}
  //End of stem
  cylinder{<-5,.7,0>+Clip_Right,<-4,1.3,.1>,.1}
  cylinder{<-5,.7,0>+Clip_Left,<-4,1.3,-.1>,.1}
  sphere{<-4,1.3,.1>,.1}
  cylinder{<-4,1.3,.1><-4,1.3,-.1>,.1}
  sphere{<-4,1.3,-.1>,.1}
 
  #declare C=1;
  #while (C<=9)
   #declare OnClip_Mover=sqrt(C/9);
   cylinder{Clip_Left,Clip_Right,.1 translate <-11+6*OnClip_Mover,.8-.1*OnClip_Mover,0>}
  #declare C=C+1;
  #end
  material{Pen_Gold_Material}
  photons{target .25 reflection on refraction off collect off}
  }
 //Golden base part with manufacturer's engravement
 difference{
  cylinder{x*-1.5,x*-1.2,1.2
   pigment{Pen_Gold_Pigment}
   normal{bump_map{png "pen_impr.png" map_type 2} rotate z*90 rotate x*-90 scale <.3,1,1> bump_size .2}
   finish{Pen_Gold_Finish}
   }
  cylinder{x*-1.6,x*-1.1,1.1
   pigment{rgb 0}
   }
  photons{target .25 reflection on refraction off}
  }
 transform{PenCap_Transform}
 }
#end
