/*
 * grass.inc - Maurizio Tomasi 2002 (zio_tom78@hotmail.com)
 *             Visit my website at http://www.geocities.com/zio_tom78
 *
 * Macros for rendering grass and reeds.
 */

#declare Pi = 3.141592;

/* This is an array of three objects.  The TraceGrass macro will
   randomly pick one for each blade placed in the scene. */

#declare Blade = array[3]

#declare Blade[0] =
  sphere_sweep {
    b_spline 4
    
    <   0, -1.00, 0>, 0.040
    <   0,  0.00, 0>, 0.040
    <0.25,  1.00, 0>, 0.016
    <0.75,  0.75, 0>, 0.008

    tolerance 0.0001
  }

#declare Blade[1] =
  sphere_sweep {
    b_spline 5
    
    <   0, -1.00, 0>, 0.040
    <   0,  0.00, 0>, 0.040
    <0.25,  0.75, 0>, 0.040
    <0.50,  1.00, 0>, 0.016
    <0.75,  1.50, 0>, 0.008

    tolerance 0.0001
  }

#declare Blade[2] =
  sphere_sweep {
    b_spline 5
    
    <   0, -1.00, 0>, 0.040
    <   0,  0.00, 0>, 0.040
    <0.25,  0.70, 0>, 0.040
    <0.70,  1.00, 0>, 0.016
    <1.40,  1.30, 0>, 0.008

    tolerance 0.0001
  }


/* Each blade stripe is equal to the corresponding blade (see the
   Blade array above), but it is StripeFrac times thinner. */

#declare BladeStripe = array[3]
#declare StripeFrac = 5;

#declare BladeStripe[0] =
  sphere_sweep {
    b_spline 4
    
    <   0, -1.00, 0>, 0.040 / StripeFrac
    <   0,  0.00, 0>, 0.040 / StripeFrac
    <0.25,  1.00, 0>, 0.016 / StripeFrac
    <0.75,  0.75, 0>, 0.008 / StripeFrac
    
    tolerance 0.0001
  }

#declare BladeStripe[1] =
  sphere_sweep {
    b_spline 5
    
    <   0, -1.00, 0>, 0.040 / StripeFrac
    <   0,  0.00, 0>, 0.040 / StripeFrac
    <0.25,  0.75, 0>, 0.040 / StripeFrac
    <0.50,  1.00, 0>, 0.016 / StripeFrac
    <0.75,  1.50, 0>, 0.008 / StripeFrac

    tolerance 0.0001
  }

#declare BladeStripe[2] =
  sphere_sweep {
    b_spline 5
    
    <   0, -1.00, 0>, 0.040 / StripeFrac
    <   0,  0.00, 0>, 0.040 / StripeFrac
    <0.25,  0.70, 0>, 0.040 / StripeFrac
    <0.70,  1.00, 0>, 0.016 / StripeFrac
    <1.40,  1.30, 0>, 0.008 / StripeFrac

    tolerance 0.0001
  }


#declare GrassStream = seed (18);

/* Place a set of grass blades above Obj (typically an height field,
   but any object type is ok).  Their height will be quasi-normally
   distributed around MeanHeight (a small uniform random component is
   still present), with a standard deviation equal to Sigma; blades
   shorter than MinHeight will be removed.  MeanRadius is the average
   base radius of the blades.  Color1 and Color2 are the two color
   extrema (the effective color results from an interpolation).  If
   UseStripes is on, place thin stripes around each blade (warning:
   this slows down both the parsing and the rendering process). */

#macro TraceGrass (Obj, MinHeight, NumOfBlades, MeanHeight,
  Sigma, MeanRadius, Color1, Color2, UseStripes)
union {
  #local BladeCount = 0;

  #local MinBox = min_extent (Obj);
  #local MaxBox = max_extent (Obj);
  
  #local MinX = MinBox.x;
  #local MaxX = MaxBox.x;
  #local MinZ = MinBox.z;
  #local MaxZ = MaxBox.z;
  #local MaxY = MaxBox.y;

  // This will be used to determine the *effective* mean height.
  #local GrassHeightHF =
  height_field {
    jpeg "grass-height.jpg"
    smooth
    
    scale MaxBox - MinBox
    translate MinBox
  }
  
  // This will be used to determine the color shade
  #local GrassColorHF =
  height_field {
    jpeg "grass-color.jpg"
    smooth
    
    scale MaxBox - MinBox
    translate MinBox
  }
  
  Debug_Message (concat ("Extents: <", vstr (3, MinBox, ", ", 0, 1),
			 "> to <", vstr (3, MaxBox, ", ", 0, 1), ">\n"))
  
  #while (BladeCount < NumOfBlades)
    #local PositionX = RRand (MinX, MaxX, GrassStream);
    #local PositionZ = RRand (MinZ, MaxZ, GrassStream);
    
    #local Normal = <0, 0, 0>;

    #local Pos =
    trace (Obj, <PositionX, MaxY + 1.0, PositionZ>, <0, -1, 0>, Normal);
    
    #if (vlength (Normal) != 0)
      #if (Pos.y >= MinHeight)
	// Determine the mean height of this blade ("mean" in the
	// sense that it will be a bit randomized).
	#local TempPoint =
	trace (GrassHeightHF, <PositionX, MaxY + 1.0, PositionZ>,
	  <0, -1, 0>);
	// Height is always in [0, 1]
	#local Height = (TempPoint.y - MinBox.y) / (MaxBox.y - MinBox.y);
	Debug_Message (concat ("Height: ", str (Height, 0, 1), "\n"))
	
	// Determine the shade of this blade
	#local TempPoint =
	trace (GrassColorHF, <PositionX, MaxY + 1.0, PositionZ>,
	  <0, -1, 0>);
	// Color is always in [0, 1]
	#local Color = (TempPoint.y - MinBox.y) / (MaxBox.y - MinBox.y);
	Debug_Message (concat ("Color: ", str (Color, 0, 1), "\n"))

	// Actually, UseStripes is not used in the image because
	// the reeds and the grass are quite far from the observer.
	// But in a previous version, the reeds were very close to
	// the camera, and this option gave a nice "striped" effect
	// (although it slowed down the rendering).
	#if (UseStripes)
	  #local BaseColor = (Color1 + (Color2 - Color1) * Color);
	  #local BladeNum = floor (RRand (0, 3, GrassStream));
	  
	  #local obj =
	  union {
	    object {
	      Blade[BladeNum]
	    }
	    
	    #local StripeCount = 0;
	    #while (StripeCount < 10)
	      object {
		BladeStripe[BladeNum]
		
		translate (<0.040 * cos (StripeCount / 12 * 2 * Pi),
		  0.0, 0.040 * sin (StripeCount / 12 * 2 * Pi)>)
	      }
	      #declare StripeCount = StripeCount + 1;
	    #end

	    texture {
	      pigment {
		bozo
		turbulence 0.4
		color_map {
		  [0.0 color rgb Color1]
		  [1.0 color rgb Color2]
		}
		
		translate VRand (GrassStream)
		scale <0.05, 0.5, 0.05>
	      }
	    }

	    // Dark spots
	    texture {
	      pigment {
		wrinkles
		color_map {
		  [0.00 0.30 color rgbt 1 color rgbt 1]
		  [0.30 0.33 color rgb BaseColor color rgb BaseColor * 0.5]
		  [0.33 0.36 color rgb BaseColor * 0.5 color rgb BaseColor]
		  [0.36 1.00 color rgbt 1 color rgbt 1]
		}
		
		scale 0.1
	      }
	    }
	  }
	  
	#else
	  #local obj =
	  object {
	    Blade[floor (RRand (0, 3, GrassStream))]
	    pigment { color rgb (Color1 + (Color2 - Color1) * Color) }
	  }
	#end
	
	object {
	  obj
	  
	  normal {
	    agate 0.015
	    scale <0.1, 15.0, 0.1>
	  }

	  // Look for a suitable scale (not negative)
	  #local ScaleFactor = -1.0;
	  #while (ScaleFactor <= 0.0)
	    #declare ScaleFactor =
	    Rand_Normal (2.0 * Height * MeanHeight,
	      (0.5 + Height) * Sigma, GrassStream);
	  #end
	  
	  scale <MeanRadius, ScaleFactor, MeanRadius>
	  rotate RRand (-30, 30, GrassStream) * y
	  translate Pos
	}
	
	Debug_Message (concat ("New blade created: <",
                               vstr (3, Pos, ", ", 0, 1), ">, height: ",
                               str (Pos.y, 1, -1), "\n"))
	
	#local BladeCount = BladeCount + 1;
      #end
    #end
  #end
}
#end
