// ==== Standard POV-Ray Includes ====
#include "colors.inc"	  // Standard Color definitions
#include "textures.inc"	  // Standard Texture definitions
#include "functions.inc"  // internal functions usable in user defined functions
// T_Stone4 used for white pieces and T_Stone24 used for black pieces
#include "stones.inc"
 
/* Used for quick rendering 
// create a regular point light source
light_source {
  0*x                  // light's position (translated below)
  color rgb <1,1,1>    // light's color
  translate <20, 40, 20>
}
  
// create a regular point light source
light_source {
  0*x                  // light's position (translated below)
  color rgb <1,1,1>    // light's color
  translate <-40, 40, -20>
}

// perspective (default) camera
camera {
  location  <-18, 13, -18>
  look_at   <-5, 2,  -5>
  right     x*image_width/image_height
}
 */  
  
// An area light (creates soft shadows)
// WARNING: This special light can significantly slow down rendering times!
light_source {
  0*x                 // light's position (translated below)
  color rgb 1.0       // light's color
  area_light
  <8, 0, 0> <0, 0, 8> // lights spread out across this distance (x * z)
  4, 4                // total number of lights in grid (4x*4z = 16 lights)
  adaptive 0          // 0,1,2,3...
  jitter              // adds random softening of light
  circular            // make the shape of the light circular
  orient              // orient light
  translate <20, 40, 20>   // <x y z> position of light
}
   
light_source {
  0*x                 // light's position (translated below)
  color rgb 1.0       // light's color
  area_light
  <8, 0, 0> <0, 0, 8> // lights spread out across this distance (x * z)
  4, 4                // total number of lights in grid (4x*4z = 16 lights)
  adaptive 0          // 0,1,2,3...
  jitter              // adds random softening of light
  circular            // make the shape of the light circular
  orient              // orient light
  translate <-40, 40, -20>   // <x y z> position of light
}   

// focal blur camera
camera {
  location  <-18, 13, -18>
  look_at   <-5, 2, -5>
  right     x*image_width/image_height
  aperture .4               // [0...N] larger is narrower depth of field (blurrier)
  blur_samples 25           // number of rays per pixel for sampling
  focal_point <-10.5, 4,  -10.5>    // point that is in focus <X,Y,Z>
  confidence 0.95           // [0...<1] when to move on while sampling (smaller is less accurate)
  variance 1/200            // [0...1] how precise to calculate (smaller is more accurate)
}

     
// ground plane
plane {
  y, // <X Y Z> unit surface normal, vector points "away from surface"
  -1 // distance from the origin in the direction of the surface normal
  hollow on // has an inside pigment?
  texture {
    pigment {
      color White
    }
  }
} 

// wall plane
plane {
  z, // <X Y Z> unit surface normal, vector points "away from surface"
  40 // distance from the origin in the direction of the surface normal
  hollow on // has an inside pigment?
  texture {
    pigment {
      rgb <.8, .8, .8>
    }
  }
}
 
// wall plane 
plane {
  x, // <X Y Z> unit surface normal, vector points "away from surface"
  40 // distance from the origin in the direction of the surface normal
  hollow on // has an inside pigment?
  texture {
    pigment {
      rgb <.9, .9, .9>
    }
  }
}

// base for all the chess pieces
#declare base = 
  union {
    box { <-1, 0, -1>, <1, 1, 1> }
    box { <.9, 1, .9>, <-.9, 1.25, -.9> }
    translate <0, -1.25, 0>
  }

// definition of a pawn  
#declare pawn = 
  union {
    object { base }
    box { <-1, 0, -1>, <1, 2, 1> } 
    translate <0, 1.25, 0>
  }

// definition of a rook
#declare rook = 
  union {
    object { base } 
    difference {
      box { <-1, 0, -1>, <1, 3, 1> }
      box { <.25, 2.5, -2>, <-.25, 3.5, 2> }
      box { <-2, 2.5, .25>, <2, 3.5, -.25> }       
    }
    translate <0, 1.25, 0>     
  } 

// definition of a bishop  
#declare bishop = 
  union {
    object { base } 
    difference {
      box { <-1, 0, -1>, <1, 4, 1> }
      object { box { <-1, -1, -1>, <1, 1, 1> rotate <45, 0, 0> } translate <0, 4.5, -1> }
      object { box { <-1, -1, -1>, <1, 1, 1> rotate <45, 0, 0> } translate <0, 4.5, 1> } 
      object { box { <-1, -1, -1>, <1, 1, 1> rotate <0, 0, 45> } translate <1, 4.5, 0> }
      object { box { <-1, -1, -1>, <1, 1, 1> rotate <0, 0, 45> } translate <-1, 4.5, 0> }
      object { box { <-.2, -1, -1>, <.2, 1, 1> rotate <0, -45, 30> } translate < -.5, 3.5, -.5> }       
    }
    translate <0, 1.25, 0>     
  }

// definition of a knight  
#declare knight = 
  union {
    object { base }
    difference {
      box { <-1, 0, -1>, <1, 3.5, 1> }
      object { box { <-1, -1, -1.1>, <1, 1, 1.1> rotate <0, 0, -55> } translate <1.25, 3.5, 0> }
      object { box { <-1, -1, -1.1>, <1, 1, 1.1> rotate <0, 0, 55> } translate <2, 1.5, 0> }  
      object { box { <-1, -.1, -1.1>, <1, .1, 1.1> rotate <0, 0, 145> } translate <.5, 1.5, 0> }
      object { box { <-1, -1, -1.1>, <1, 1, 1.1> } translate <.25, 4.25, 0> }
    }
    translate <0, 1.25, 0>
  }

// definition of a king  
#declare king = 
  union {
    object { base }
    box { <-1, 0, -1>, <1, 4, 1> }
    box { <-.5, 4, -.5>, <.5, 6, .5> }
    box { <-1, 4.5, -.5>, <1, 5.5, .5> }
    box { <-.5, 4.5, -1>, <.5, 5.5, 1> }
    translate <0, 1.25, 0>
  }

// definition of a queen
#declare queen = 
  union {
    object { base }
    difference {
      box { <-1, 0, -1>, <1, 5, 1> }
      object { box { <-1.1, -1, -1>, <1.1, 1, 1> rotate <45, 0, 0> } translate <0, 5.5, 0> }
      object { box { <-1, -1, -1.1>, <1, 1, 1.1> rotate <0, 0, 45> } translate <0, 5.5, 0> }
    }
    translate <0, 1.25, 0>
  }    

// seed used for random numbers
#declare R1 = seed(5280);

// macro for placing a chess piece on the board
// randomly tweaks the precise rotation and position of the piece  
#macro PlacePiece (piece, txtr, row, col) 
  object { piece 
    texture {
      txtr
      normal { wrinkles .1 scale .25 } 
      finish {
        reflection { .3 } 
        phong .5
      }
    }
    rotate <0, 15 - rand(R1) * 20, 0>
    translate <(col-4.5)*3 + (.25-rand(R1)*.5), 0, (row-4.5)*3 + (.25-rand(R1)*.5)> 
    
  }
#end


#declare White = T_Stone4;  // white chess piece color
#declare Black = T_Stone24; // black chess piece color

// places the white pawns on the board
PlacePiece (pawn, White, 3, 3)
PlacePiece (pawn, White, 4, 4)
PlacePiece (pawn, White, 5, 5)
PlacePiece (pawn, White, 2, 6)
PlacePiece (pawn, White, 2, 7)
PlacePiece (pawn, White, 2, 8)

// places the black pawns on the board
PlacePiece (pawn, Black, 4, 1)
PlacePiece (pawn, Black, 4, 3)
PlacePiece (pawn, Black, 6, 3)
PlacePiece (pawn, Black, 6, 5)
PlacePiece (pawn, Black, 7, 6)
PlacePiece (pawn, Black, 7, 7)
PlacePiece (pawn, Black, 6, 8)

// places the rook that has a bullet hole
difference { PlacePiece (rook, White, 1, 2) cylinder { <-20, 4, -10.5>, <20, 4, -10.5>, .6 texture { White } } }
// places the other rooks on the board
PlacePiece (rook, White, 6, 2)
PlacePiece (rook, Black, 8, 1)
PlacePiece (rook, Black, 8, 8)
 
// places the night on the board
PlacePiece (knight, Black, 8, 2) 

// places the bishop on the board
PlacePiece (bishop, White, 3, 6)

// places the king that has a bullet hole
difference { PlacePiece (king, White, 1, 7) cylinder { <-20, 4, -10.5>, <20, 4, -10.5>, .6 texture { White } } }
// places the other royal pieces
PlacePiece (queen, White, 3, 1)
PlacePiece (king, Black, 8, 5)
PlacePiece (queen, Black, 8, 3) 

// the board squares
#declare i = 1;
#while (i <= 8)
  #declare j = 1; 
  #while (j <= 8)
    #if (mod(i + j,2) = 0)
      box { <(i-5)*3 +.1, -1, (j-5)*3 +.1>, <(i-4)*3 -.1, 0, (j-4)*3 -.1> texture { pigment { rgb <.15, .5, .5> } finish { reflection { .4 } } } }
    #else 
      box { <(i-5)*3 +.1, -1, (j-5)*3 +.1>, <(i-4)*3 -.1, 0, (j-4)*3 -.1> texture { pigment { rgb <.7, .6, .6> } finish { reflection { .4 } } } }
    #end
    #declare j = j + 1;
  #end
  #declare i = i + 1;
#end

// black base of the board
box {
  <-12.5, -1, -12.5>, <12.5, -.5, 12.5>
  texture {
    pigment { rgb <.1, .1, .1> }
    finish {
      phong .75
      ambient .25
    }
  }
} 
          
// definition of a bullet          
#declare bullet =
difference
  {
  sor { 
    14, // # of points  
    <0, 0>,
    <0, 0>,
    <.05 , .03>,
    <.55, 2>,
    <.6, 2.9>,
    <.6, 3>,
    <.6, 3.5>
    <.57, 3.55>,
    <.6, 3.6>, 
    <.6, 4>,
    <.6, 4.45>,
    <.5, 4.5>,
    <0, 4.53>,
    <.7, 5>
  } 
  object {box { <-1, -1, -1>, <1, 1, 1> rotate < 10, 45, 9> } pigment { rgb <.9, .9, .9> } translate <1.98, 3, 0> }
  cutaway_textures  
  rotate <0, 90, 90>
  translate <5, 0, 0>
}

// the bullet
object { bullet 
  texture {
    pigment { rgb <.75, .75, .75> }
    finish { 
      phong .75
      phong_size 100  
      crand .25
      metallic 
      brilliance 5 
      reflection { .25 }
    }
  }
  rotate <-30, 0, 180> 
  translate <-10, 4, -10.5>
}  

// the bullet trail
#declare dY = 5;
#declare dZ = -14;
#declare dX = -12;
#declare dRadius = .3;
#declare dDelta = 1;
#macro dust (iVal) 
  sphere {
    <0, 0, 0> // center of sphere <X Y Z>
    0.5       // radius of sphere
    scale <2.5,1,1> // <= Note: Spheres can become ellipses by uneven scaling
    texture {
      pigment { 
        color rgb <.75, .75, .75> transmit .6 + iVal*.01
      }                                 
      normal { bumps 1 scale .1 }
    }   
    translate <(dX + dDelta * iVal), dY + dRadius * sin(iVal*.4), dZ + dRadius * cos(iVal*.4)>
    scale 1.02 - rand(R1) * .04
  }
#end

merge {
#declare i = 0;
#while (i < 40) 
dust(i)             
#declare i = i +1;
#end
scale .75 
} 

// random debris for the king bullet hole
#declare Ex = 6;
#declare Ey = 5;
#declare Ez = -10.5;

#declare i = 0;
#while (i < 80) 
  #declare x1 = rand(R1);
  #declare y1 = rand(R1);      
  #declare E_Distance = rand(R1) * 3;

  prism {                                
    linear_sweep  
    linear_spline  
    -.15 + rand(R1)*.3,         
    .15 - rand(R1)*.3,         
    5,           
    <x1*.3, y1*.3>, <-rand(R1)*.3, rand(R1)*.3>, <-rand(R1)*.3, -rand(R1)*.3>, <rand(R1)*.3, -rand(R1)*.3>, <x1*.3, y1*.3>
    texture {
      White
      normal { wrinkles .1 scale .25 } 
      finish {
        reflection { .3 } 
        phong .5
      }
    }
    scale .75
    rotate < rand(R1) * 360, rand(R1) * 360, rand(R1) * 360>
    translate <(Ex-E_Distance), (Ey+(E_Distance + .5) * sin(2*x1*3.14159)), (Ez+(E_Distance + .5) * cos(2*x1*3.14159))>
     
  }
  #declare i = i + 1;
#end  

// random debris for the rook bullet hole
#declare Ex = -8;
#declare Ey = 5;
#declare Ez = -10.5;

#declare i = 0;
#while (i < 50) 
  #declare x1 = rand(R1);
  #declare y1 = rand(R1);      
  #declare E_Distance = rand(R1) * 1.5;

  prism {                                
    linear_sweep  
    linear_spline  
    -.15 + rand(R1)*.3,        
    .15 - rand(R1)*.3,        
    5,           
    <x1*.3, y1*.3>, <-rand(R1)*.3, rand(R1)*.3>, <-rand(R1)*.3, -rand(R1)*.3>, <rand(R1)*.3, -rand(R1)*.3>, <x1*.3, y1*.3>
    texture {
      White
      normal { wrinkles .1 scale .25 } 
      finish {
        reflection { .3 } 
        phong .5
      }
    }
    scale .5
    rotate < rand(R1) * 360, rand(R1) * 360, rand(R1) * 360>
    translate <(Ex-E_Distance), (Ey+(E_Distance + .5) * sin(-x1*3.14159)), (Ez+(E_Distance + .5) * cos(-x1*3.14159))>
     
  }
  #declare i = i + 1;
#end

// end of file