/*

The "Leg" object is equivalent to the following isosurface,
except that it is about 30% faster (because the three contained_by
statements follow more closely the object's shape).

#local Leg =
isosurface
{
    function
    {
        select (
            y,
            pow(x-(1 + pow(y,2)),2) + pow(z,2) - 0.02, // Parabola
            pow(x-(1 + pow(y,2)) * cos(pi*pow(y,4)),2) // Spiral
                + pow(z - (1 + pow(y,2)) * sin(pi*pow(y,4)),2) - 0.02)
    }
    contained_by
    {
        box { <-2.5, -1, -2.5>, <2.5, 0.94, 2.5> }
    }
    accuracy 0.02
    max_gradient 10.25
    translate <0, 1, 0>
    scale <1, 1/2, 1> // "Normalize" its height to the unity
}
*/

#local Leg =
union
{
    isosurface
    {
        function
        {
            pow(x-(1 + pow(y,2)),2) + pow(z,2) - 0.02
        }
        contained_by
        {
            box { <1 - 0.15, -1, -0.15>, <2 + 0.15, 0, 0.15> }
        }
        accuracy 0.02
        max_gradient 3.6
        translate <0, 1, 0>
        scale <1, 1/2, 1> // "Normalize" its height to the unity
    }

    isosurface
    {
        function
        {
            pow(x-(1 + pow(y,2)) * cos(pi*pow(y,4)),2)
                + pow(z - (1 + pow(y,2)) * sin(pi*pow(y,4)),2) - 0.02 * (1 + pow(y,2))
        }
        contained_by
        {
            box { <0.3, 0, -0.15>, <1.5 + 0.15, 0.8, 2 + 0.15> }
        }
        accuracy 0.02
        max_gradient 21
        translate <0, 1, 0>
        scale <1, 1/2, 1> // "Normalize" its height to the unity
    }
    
    isosurface
    {
        function
        {
            pow(x-(1 + pow(y,2)) * cos(pi*pow(y,4)),2)
                + pow(z - (1 + pow(y,2)) * sin(pi*pow(y,4)),2) - 0.02 * (1 + pow(y,2))
        }
        contained_by
        {
            box { <-1.5 - 0.15, 0.8, 0.5>, <1.3 + 0.15, 0.94, 2 + 0.15> }
        }
        accuracy 0.02
        max_gradient 70
        translate <0, 1, 0>
        scale <1, 1/2, 1> // "Normalize" its height to the unity
    }
}

// Metal
#local SupportTex1 =
texture
{
    pigment { color rgb 0.2 }
    
    finish
    {
        ambient 0
        diffuse 0.2
        
        specular 1.3
        reflection { 1 metallic }
    }
}

// Rust
#local SupportTex2 =
texture
{
    pigment { color rgb <0.2, 0.1, 0.1> }
    
    normal
    {
        granite 0.2
        turbulence 0.7
    }
    
    finish
    {
        ambient 0
        diffuse 0.6
    }
}

#local SupportTexture =
texture
{
    wrinkles
    texture_map
    {
        [0.55 SupportTex1]
        [0.95 SupportTex2]
    }
    scale <0.15, 0.15, 0.15>
}

#local Support =
union
{
    object { Leg }
    object { Leg rotate y*120 }
    object { Leg rotate y*-120 }

    torus
    {
        1.92, 0.1
        scale <1, 0.075, 1>
        translate 0.97*y
    }

    torus
    {
        1.4, 0.1
        scale <1, 0.075, 1>
        translate 0.8*y
    }

    torus
    {
        1, 0.1
        scale <1, 0.1, 1>
        translate 0.6*y
    }

    torus
    {
        1.2, 0.1
        scale <1, 0.075, 1>
        translate 0.2*y
    }
    
    disc
    {
        <0, 0.2, 0>, y, 1.2
    }

    scale <0.4, 3, 0.4>
    
    texture { SupportTexture }    
}

#local MirrorWidth = 0.7;
#local MirrorHeight = 0.8;
#local MirrorSuppRadius = 0.02;
#local MirrorBorder = 0.067;

#local MirrorSupport =
union
{
    sphere_sweep
    {
        linear_spline
        6,
        
        <-0.04, 0.7, 1.20>, MirrorSuppRadius
        <-MirrorWidth/2, 0.7, 1.20>, MirrorSuppRadius
        <-MirrorWidth/2, 0.7 + MirrorHeight, 1.20>, MirrorSuppRadius
        < MirrorWidth/2, 0.7 + MirrorHeight, 1.20>, MirrorSuppRadius
        < MirrorWidth/2, 0.7, 1.20>, MirrorSuppRadius
        <+0.04, 0.7, 1.20>, MirrorSuppRadius
    }
    
    sphere_sweep
    {
        cubic_spline
        5,
        
        <-0.08, 0.0, 0.50>, MirrorSuppRadius
        <-0.08, 0.0, 0.77>, MirrorSuppRadius
        <-0.08, 0.0, 1.20>, MirrorSuppRadius
        <-0.04, 0.7, 1.20>, MirrorSuppRadius
        <-0.04, 0.9, 1.20>, MirrorSuppRadius
    }

    sphere_sweep
    {
        cubic_spline
        5,
        
        <+0.08, 0.0, 0.50>, MirrorSuppRadius
        <+0.08, 0.0, 0.77>, MirrorSuppRadius
        <+0.08, 0.0, 1.20>, MirrorSuppRadius
        <+0.04, 0.7, 1.20>, MirrorSuppRadius
        <+0.04, 0.9, 1.20>, MirrorSuppRadius
    }

    cylinder
    {
        <-MirrorWidth/2, 0.7 + MirrorHeight/2, 1.20 + MirrorSuppRadius>,
        <-MirrorWidth/2 + MirrorBorder, 0.7 + MirrorHeight/2, 1.20 + MirrorSuppRadius>,
        MirrorSuppRadius
    }

    cylinder
    {
        <MirrorWidth/2, 0.7 + MirrorHeight/2, 1.20 + MirrorSuppRadius>,
        <MirrorWidth/2 - MirrorBorder, 0.7 + MirrorHeight/2, 1.20 + MirrorSuppRadius>,
        MirrorSuppRadius
    }

    translate 2.91*y
    texture { SupportTexture }    
}

#local Mirror =
union
{
    box
    {
        <-MirrorWidth/2 + MirrorBorder, 0.7 + MirrorBorder, -0.005>,
        < MirrorWidth/2 - MirrorBorder, 0.7 + MirrorHeight - MirrorBorder, 0.005>
        
        texture
        {
            finish { reflection { 0.45 } }
            
            pigment { color rgb 0 }
            
            //normal { crackle 0.02 }
        }
    }

    sphere_sweep
    {
        linear_spline
        5,
        
        <-MirrorWidth/2 + MirrorBorder, 0.7 + MirrorBorder, 0>, MirrorSuppRadius,
        < MirrorWidth/2 - MirrorBorder, 0.7 + MirrorBorder, 0>, MirrorSuppRadius,
        < MirrorWidth/2 - MirrorBorder, 0.7 + MirrorHeight - MirrorBorder, 0>, MirrorSuppRadius,
        <-MirrorWidth/2 + MirrorBorder, 0.7 + MirrorHeight - MirrorBorder, 0>, MirrorSuppRadius,
        <-MirrorWidth/2 + MirrorBorder, 0.7 + MirrorBorder, 0>, MirrorSuppRadius
        
        texture { SupportTexture }
    }

    translate - (0.7 + MirrorHeight/2) * y
    rotate 10*x
    translate (0.7 + MirrorHeight/2) * y

    translate <0, 2.91, 1.20>
}

#include "basin.inc"

#declare ClayTexture =
texture
{
    pigment
    {
        wrinkles
        pigment_map {
            [0.0 color rgb 2]
            [0.83 color rgb 2]
            [0.83 color rgb 1]
            [0.91 color rgb 0.1]
        }
        scale 0.15
    }
    
    normal
    {
        bozo 0.20
        turbulence 0.7
        scale 0.45
    }
    
    finish { specular 0.7 }
}

#include "jug.inc"

#declare BasinSet =
union
{
    union
    {
        object
        {
            Basin
            scale 0.8
            translate <0, 2.7, 0>

            texture { ClayTexture rotate 40*y }
        }
        
        object
        {
            Jug
            scale 0.36
            rotate -130*y
            translate 0.6*y

            texture { ClayTexture scale 1.2}
        }
    }

    object { Support }
    
    union
    {
        object { MirrorSupport }
        object { Mirror }
        
        rotate 35*y
    }
}


///////////////////////////////////////////////////////////////////////

/*
global_settings { max_trace_level 10 }

background { color rgb 0.6 }

camera
{
    location <0, 4, 3>
    look_at <0, 2, 16>
}

light_source
{
    <-3, 10, -6> color rgb 1
}

object
{
    BasinSet
    rotate 160*y
    translate <0, 0, 8>
}
*/