// Persistence of Vision Ray Tracer Scene Description File
// File: pendulum-clock.inc
// Vers: 3.5
// Desc: A "Morbier" pendulum clock 
// Date: 2003/09/22
// Auth: Maurizio Tomasi

#include "rand.inc"
#include "colors.inc"
#include "woods.inc"
#include "shapes.inc"

#include "clock-func.inc"
#include "pyr-base.inc"
#include "column.inc"
#include "sculpture-texture.inc"
#include "statue_pov.inc"

#local ClockSeed = seed (37624);

#declare Wood =
texture
{
    T_Wood2
    
    finish
    {
        specular 1.0
        roughness 0.03
    }
    normal
    {
        bozo 0.3
        scale 0.5
    }
    
    rotate 90*x    
    scale 0.15
}
texture
{
    pigment { color rgbf <0.85, 0.85, 0.85, 0.91> }
}

#local ProfileWood =
texture
{
    T_Wood4
    
    finish
    {
        specular 1.0
        roughness 0.03
    }
    normal
    {
        bozo 0.3
        scale 0.5
    }

    scale 0.15
}
texture
{
    pigment { color rgbf <0.85, 0.85, 0.85, 0.93> }
}


#local Glass =
material
{
    texture
    {
        pigment { color rgbt <0.97, 0.97, 0.99, 0.91> }
        finish { reflection { 0.015 } specular 1 roughness 0.01 }
    }

    interior { ior 1.3 }
    
}


#local ProfileRadius = 0.015;

#macro ProfiledBox (X1, Y1, X2, Y2, Tex1, Tex2)

    union
    {
        box
        {
            <X1, Y1, 0>, <X2, Y2, 0.5>
            
            texture { Tex1 }
        }
        
        sphere_sweep
        {
            linear_spline
            5,
            
            <X1, Y1, 0>, ProfileRadius
            <X1, Y2, 0>, ProfileRadius
            <X2, Y2, 0>, ProfileRadius
            <X2, Y1, 0>, ProfileRadius
            <X1, Y1, 0>, ProfileRadius
            
            texture { Tex2 }
        }
        
        sphere_sweep
        {
            linear_spline
            5,
            
            <X2, Y1, 0.5>, ProfileRadius
            <X2, Y2, 0.5>, ProfileRadius
            <X2, Y2, 0.0>, ProfileRadius
            <X2, Y1, 0.0>, ProfileRadius
            <X2, Y1, 0.5>, ProfileRadius
            
            texture { Tex2 }
        }
    }
    

#end


//////////////////////////////////////////////////////////////////
// Base

#local PrismStartProfile = array [3]
{
    < 0.0, 1.0>,
    <-0.5, 1.0>,
    <-1.0, 0.0>
}

#local Profile = array [8]
{
    <-0.6, 0.0>,
    <-0.5, 0.0>,
    <-0.4, 0.1>,
    <-0.3, 0.3>,
    < 0.3, 0.3>,
    < 0.4, 0.1>,
    < 0.5, 0.0>,
    < 0.6, 0.0>
}

#local PrismEndProfile = array [5]
{
    < 1.0, 0.0>
    < 0.5, 1.0>,
    < 0.0, 1.0>,
    <-0.5, 1.0>
    <-1.0, 0.0>
}

#local PedestalHeight = 0.5;
#local BoxHeight = 1.0;


/*
 * Base structure:
 *
 * +----------+
 * |          |
 * |          |
 * |   Box    |
 * |          |
 * |          |
 * +----------+
 * | Pedestal |
 * |  ______  |
 * | /      \ |
 * |/        \|
 */

#local Base =
union
{
    /****************** Pedestal ********************/
    
    intersection
    {
        prism
        {
            cubic_spline
            -0.5, 0.0, 
            dimension_size (PrismStartProfile, 1) 
                + dimension_size (Profile, 1) 
                + dimension_size (PrismEndProfile, 1)
            
            #local Count = 0;
            #while (Count < dimension_size (PrismStartProfile, 1))
            
                PrismStartProfile[Count]
                
                #local Count = Count + 1;
            #end
            
            #local Count = 0;
            #while (Count < dimension_size (Profile, 1))
            
                Profile[Count]
                
                #local Count = Count + 1;
            #end
    
            #local Count = 0;
            #while (Count < dimension_size (PrismEndProfile, 1))
            
                PrismEndProfile[Count]
                
                #local Count = Count + 1;
            #end
    
            rotate -90*x
        }
        
        box
        {
            <-0.5, -1.0, -1.0>, <0.5, PedestalHeight, 1.0>
        }

        texture { Wood }        
    }


    // Pedestal profile
    
    union
    {
        // Curved profile
    
        sphere_sweep
        {
            cubic_spline
            dimension_size (Profile, 1),
            
            #local Count = 0;
            #while (Count < dimension_size (Profile, 1))
            
                <Profile[Count].x, Profile[Count].y, 0.0>, ProfileRadius
                
                #local Count = Count + 1;
            #end
        }
    
    
        // Straight profile
        
        sphere_sweep
        {
            linear_spline
            4,
            
            <-0.5,            0.0, 0.0>, ProfileRadius
            <-0.5, PedestalHeight, 0.0>, ProfileRadius
            < 0.5, PedestalHeight, 0.0>, ProfileRadius
            < 0.5,            0.0, 0.0>, ProfileRadius
        }

        sphere_sweep
        {
            linear_spline
            4,
            
            < 0.5,            0.0, 0.5>, ProfileRadius
            < 0.5, PedestalHeight, 0.5>, ProfileRadius
            < 0.5, PedestalHeight, 0.0>, ProfileRadius
            < 0.5,            0.0, 0.0>, ProfileRadius
        }

        texture { ProfileWood }
    }
    
    
    /******************** Box **********************/
    
    
    #local BoxTexture = 
    texture 
    {
        Wood 
        rotate <RRand (87, 93, ClockSeed), RRand (-3, 3, ClockSeed), 0> 
        translate VRand (ClockSeed)

        translate -0.5*x
    }

    union
    {
        object
        {
            ProfiledBox (-0.5, 0, 0.5, BoxHeight, BoxTexture, ProfileWood)
        }
        
        sphere_sweep
        {
            linear_spline
            5,
            
            <-0.4,             0.1, 0.0>, ProfileRadius
            <-0.4, BoxHeight - 0.1, 0.0>, ProfileRadius
            < 0.4, BoxHeight - 0.1, 0.0>, ProfileRadius
            < 0.4,             0.1, 0.0>, ProfileRadius
            <-0.4,             0.1, 0.0>, ProfileRadius

            texture { ProfileWood }
        }
        
        height_field
        {
            png "pend-clock-base.png"
            smooth
            water_level 0.01
            
            translate <-0.5, 0, -0.5>
            rotate -90*x
            scale <1, 1, 0.02>
            scale 0.7
            translate 0.5*y
            
            texture { SculptureTexture }
        }
        
                
        translate PedestalHeight*y
    }
}

//////////////////////////////////////////////////////////////////
// Pendulum Box

#local PendulumBoxHeight = 2.5;

#local PendulumBox =
union
{
    difference
    {
        union
        {
            ProfiledBox (-0.45, 0.1, 0.45, PendulumBoxHeight, Wood, ProfileWood)
            
            object
            {
                PyramidBase (1, 0.5, 0.05, 0.1)
                
                translate 0.25*z
                texture { Wood }
            }

            object
            {
                PyramidBase (1, 0.5, 0.05, 0.1)
                
                scale <1, -1, 1>
                translate <0, PendulumBoxHeight + 0.1, 0.25>
                texture { Wood }
            }
        }
        
        union
        {
            box
            {
                <-0.4, 0.05, 0.05>, <0.4, PendulumBoxHeight - 0.05, 0.45>
            }
            
            box
            {
                <-0.25, 0.2, -1>, <0.25, PendulumBoxHeight - 0.2, 0.4>
            }
        }
        
        cutaway_textures
    }

    // Glass
    
    light_group
    {
        object
        {
            PyramidBase (0.5, PendulumBoxHeight - 0.4, 0.05, 0.05)
            
            rotate -90*x
            translate PendulumBoxHeight/2*y
    
            material { Glass }
        }
        
        light_source
        {
            <-1, PendulumBoxHeight, -1>
            color rgb 0.2
        }
        
        global_lights on
    }    
    
    // Glass profile
    
    sphere_sweep
    {
        linear_spline
        5,
        
        <-0.25,                     0.2, 0>, ProfileRadius
        <-0.25, PendulumBoxHeight - 0.2, 0>, ProfileRadius
        < 0.25, PendulumBoxHeight - 0.2, 0>, ProfileRadius
        < 0.25,                     0.2, 0>, ProfileRadius
        <-0.25,                     0.2, 0>, ProfileRadius

        texture { ProfileWood }
    }

    // Pendulum

    light_group
    {    
        height_field
        {
            png "pendulum.png"
            water_level 0.01
            smooth
            
            rotate -90*x
            translate -0.5*x
            scale <0.15, 1, 0.05>
            
            // Give a little tilt
            translate -y
            rotate -1.5*z
            
            scale 2.0
            
            translate <0.0, PendulumBoxHeight, 0.2>
            
            texture
            {
                pigment { color rgb <0.2, 0.2, 0.15>*0.3 }
                finish { specular 1 roughness 0.5 metallic }
            }
        }

        // Shadow-only light
        
        light_source
        {
            <1.0, 0.5, -1.5>
            color rgb 0.2
        }
        
        light_source
        {
            <1.0, 0.5, -1.5>
            color rgb -0.2
            
            shadowless
        }
        
        global_lights
    }
    
    translate (PedestalHeight + BoxHeight)*y
}


//////////////////////////////////////////////////////////////////
// Clock

#local ClockHeight = 1.3;
#local ClockRadius = 0.35;
#local GlassThickness = 0.40;
// The real glass thickness will be GlassThickness * ThicknessScale
#local ThicknessScale = 1.0;

#local HalfHand =
height_field
{
    png "hand.png"
    water_level 0.5
    translate <-1, 0, 0>
    scale <1/8, 0.01, 1>
    rotate -90*x
}

#local Hand =
union
{
    object { HalfHand rotate 15*y }
    object { HalfHand scale <-1, 1, 1> rotate -15*y }
    scale ClockRadius*0.7
    translate <0, -0.05, 0.01>
    
    texture
    {
        pigment { color rgb 0.05 }
    }
}

#local Clock =
union
{
    // Clock box
    difference
    {
        union
        {
            ProfiledBox (-0.5, 0.1, 0.5, ClockHeight, Wood, ProfileWood)
            
            sphere_sweep
            {
                linear_spline
                5,
                
                <-0.4,               0.2, 0>, ProfileRadius
                <-0.4, ClockHeight - 0.1, 0>, ProfileRadius
                < 0.4, ClockHeight - 0.1, 0>, ProfileRadius
                < 0.4,               0.2, 0>, ProfileRadius
                <-0.4,               0.2, 0>, ProfileRadius
        
                texture { ProfileWood }
            }
        }
        
        cylinder
        {
            <0.0, 0, -1.0>,
            <0.0, 0,  0.45>, ClockRadius
            
            translate (0.1 + (ClockHeight - 0.1) / 2) * y
        }
        
        cutaway_textures
    }
    
    // Clock
    
    union
    {
        box
        {
            <-ClockRadius, -ClockRadius, 0.02>, <ClockRadius, ClockRadius, 0.15>
    
            texture
            {
                pigment { color rgb 1 }
            }
            texture
            {
                pigment
                {
                    image_map
                    {
                        png "pend-clock.png"
                        once
                        interpolate 2
                    }
                    
                    translate <-0.5, -0.5, 0>
                    scale (ClockRadius * 2) * 0.92
                }
            }
        }
        
        // Hands
        object { Hand scale <1, 0.8, 1> rotate -ClockHourHand (CurrentTime)*z }
        object { Hand scale <0.8, 1.3, 1> rotate -ClockMinuteHand (CurrentTime)*z }
        
        translate (0.1 + (ClockHeight - 0.1) / 2) * y
    }
    
    
    // Glass

    intersection
    {
        sphere
        {
            #local P = (pow (ClockRadius, 2.0) - pow (GlassThickness, 2.0)) / (2.0 * GlassThickness);
            
            0, P + GlassThickness
            
            /* The formula below is the solution of the following system of equations:
             *  /
             *  |            2      2              2
             *  |  ( x - P )    + y    = ( T + P )
             * <
             *  |  x = 0
             *  |  y = R
             *  \
             *
             * where P is the unknown to be placed in the "translate" statement,
             * R is the clock radius and T is the glass thickness.  The sphere is
             * centered in (P, 0) and must pass through points (-T, 0), (0, R) and
             * (0, -R).  The y=0 axis is placed along the clock's surface.
             */
            
            translate z*P
        }
        
        cylinder
        {
            <0.0, 0, -1.0>,
            <0.0, 0,  0.15>, ClockRadius
        }
        
        translate (0.1 + (ClockHeight - 0.1) / 2) * y

        scale <1, 1, ThicknessScale>
        material { Glass }
    }
    
    translate (PedestalHeight + BoxHeight + PendulumBoxHeight)*y
}


#declare Columns =
union
{
    object 
    { 
        Column 
        scale <0.75, 1, 0.75> * 0.08 * ClockHeight 
        translate <0.4, 0, 0.1> 
    }
    
    object 
    { 
        Column 
        scale <-0.75, 1, 0.75> * 0.08 * ClockHeight 
        translate <-0.4, 0, 0.1> 
    }

    texture { Wood }
    
    translate (PedestalHeight + BoxHeight + PendulumBoxHeight + 0.1) * y
}


//////////////////////////////////////////////////////////////////
// Top


#local Cap =
union
{
    object
    {
        PyramidBase (1.0, 0.5, 0.05, 0.1)
        
        translate <0, 0, 0.25>
    }
    
    box
    {
        <-0.45, 0.1, 0.05>, <0.45, 0.2, 0.45>
    }

    object
    {
        PyramidBase (0.9, 0.4, 0.05, 0.1)
        
        translate <0, 0.2, 0.25>
    }
    
    object
    {
        Statue
        scale 1.2
        rotate 180*y
        translate <0, 0.3, 0.25>
    }

    texture { Wood }

    translate (PedestalHeight + BoxHeight + PendulumBoxHeight + ClockHeight) * y
}

//////////////////////////////////////////////////////////////////
// The complete object


#declare PendulumClock =
union
{
    object { Base }
    
    object { PendulumBox }
    
    object { Clock scale <1, 1, 0.67> translate 0.33/2*z }
    
    object { Columns }
    
    object { Cap }
}
