// Persistence of Vision Ray Tracer Scene Description File
// File: ?.pov
// Vers: 3.5
// Desc: Radiosity Scene Template
// Date: mm/dd/yy
// Auth: ?
//

#version 3.5;

#declare Radiosity=on;

#include "functions.inc"

global_settings {
  assumed_gamma 1.0
  //max_trace_level 25
  #if (Radiosity)
    radiosity {
      pretrace_start 0.08           // start pretrace at this size
      pretrace_end   0.04           // end pretrace at this size
      count 35                      // higher -> higher quality (1..1600) [35]
      nearest_count 5               // higher -> higher quality (1..10) [5]
      error_bound 1.8               // higher -> smoother, less accurate [1.8]
      recursion_limit 3             // how much interreflections are calculated (1..5+) [3]
      low_error_factor .5           // reduce error_bound during last pretrace step
      gray_threshold 0.0            // increase for weakening colors (0..1) [0]
      minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
      brightness 1                  // brightness of radiosity effects (0..1) [1]

      adc_bailout 0.01/2
      //normal on                   // take surface normals into account [off]
      //media on                    // take media into account [off]
      //save_file "file_name"       // save radiosity data
      //load_file "file_name"       // load saved radiosity data
      //always_sample off           // turn sampling in final trace off [on]
      //max_sample 1.0              // maximum brightness of samples
    }
  #end
}

#default {
  texture {
    pigment {rgb 1}
    #if (Radiosity)
      finish {
        ambient 0.0
        diffuse 0.6
        specular 0.3
      }
    #else
      finish {
        ambient 0.1
        diffuse 0.6
        specular 0.3
      }
    #end
  }
}

// ----------------------------------------

camera {
  right x*image_width/image_height
  /*
  location  <0,1.5,-4>
  look_at   <0,1,0>
  angle 39
  */
  /*
  location  <1,4,-4>
  look_at   <0,1.7,0>
  angle 32
  */
  location  <4.0, 8.5, -8.0>
  direction 1.5*z
  right     x*image_width/image_height
  look_at   <0.0, 5.0,  0.0>  
}

light_source {
  <30,20,-40>       // light's position
  color rgb <1, 1, 1>  // light's color
}

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

// ----------------------------------------

plane {
  y, 0
  texture {
    pigment { rgb <0.7,0.5,0.4> }
  }
}

// ----------------------------------------

// ----------------------------------------
// Checks if the provided position is free from other spheres.
#macro Is_Free_Position (Pos, rad)
#local res=true;
#local i = 0;
#while ((i < CW) & res)
  #local j = 0;
  #while ((j < Nb_Segments) & res)
    #if (vlength (SegPos[i][j] - Pos) > (rad + SegRad[i][j]))
      #local res = true;
    #else
//      #debug concat ("<", vstr(3, SegPos[i][j], ", ", 0,2), ">  -  <",
//	                  vstr(3, Pos, ", ", 0,2), "> -->  ", str(vlength (SegPos[i][j] - Pos), 0,2), "\n")
      #local res = false;      
    #end
    #local j = j + 1;
  #end
  #local i = i + 1;
#end
res
#end

// ----------------------------------------
// Returns a random position inside the object that is free from other spheres.
#macro Get_Free_Position (rad, obj_container, success, TheSeed)
#local Pmin=min_extent (obj_container);
#local Pmax=max_extent (obj_container);
#local delta=Pmax-Pmin;

#local position_ok=false;
#local try_count=0;
#while ((!position_ok) & (try_count < 500))
  #local Pos=Pmin + <rand(TheSeed),rand(TheSeed),rand(TheSeed)>*delta;
  #if (inside (obj_container, Pos))
    #if (Is_Free_Position (Pos, rad))
      #local position_ok=true;
    #end
  #end
  #local try_count = try_count + 1;
#end
#declare success = position_ok;
Pos
#end

// ----------------------------------------
// Returns a possible direction and curving for the worm so that it fits inside
// the object. This routine doesn't take into account the other spheres.
#macro Get_Possible_Conformation (scaling, shape, obj_container, start_pos, success, TheSeed, InitDir, RotateV)
#local Pmin=min_extent (obj_container);
#local Pmax=max_extent (obj_container);
#local delta=Pmin-Pmax;

#local direction_ok=false;
#local try_count=0;
#while ((!direction_ok) & (try_count < 500))
  
  // We choose an initial direction for the worm
  #local idir=vrotate (x, <rand(TheSeed),rand(TheSeed),rand(TheSeed)>*360);
  // Then we find a random vector that is perpendicular to dir to use as a rotation vector
  // between two segments.
  #local rot=10*rand(TheSeed)*
  vnormalize(vcross (idir, vrotate (x, <rand(TheSeed),rand(TheSeed),rand(TheSeed)>*360)));
  
  // Now we go through all the future segments to check if their centers are inside the object.
  #local direction_ok=true;
  #local dir = idir;
  #local Pos = start_pos;
  #local sc = 1;
  #while ((sc < Nb_Segments) & direction_ok)
    // We compute the position of the next segment.
    #local Pos = Pos + dir*shape[sc].v*scaling;
    #local dir=vrotate (dir, rot);
    
    #if (!inside (obj_container, Pos))
      #local direction_ok = false;
    #else
      #declare SegPos[CW][sc] = Pos;
      #declare SegRad[CW][sc] = shape[sc].u*scaling;
    #end
    #local sc = sc + 1;
  #end
  #local try_count = try_count + 1;
#end
#declare InitDir = idir;
#declare RotateV = rot;
#declare success = direction_ok;
#end

// ----------------------------------------
// Adds a worm in a free location.
#macro Make_Worm (scaling, shape, obj_container, TheSeed, ok, nb_p, nb_c, end_tests)

#local initial_direction = <0,0,0>;
#local rotation_vector = <0,0,0>;
#local success=false;

#local ok=false;
#local pos_attempts=0;
#while ((pos_attempts < end_tests) & !ok)
  // We take the first position for the worm.
  #declare SegPos[CW][0]=Get_Free_Position (scaling*shape[0].u, obj_container, success, TheSeed);
  #declare SegRad[CW][0]=scaling*shape[0].u;
  
  #local conf_attempts=0; 
  #while ((conf_attempts < end_tests) & !ok)
 
    Get_Possible_Conformation
      (scaling, shape, obj_container, SegPos[CW][0], success, TheSeed, initial_direction, rotation_vector)
  
    #if (success)
      // We will check all the spheres except the first one.
      #local ok=true;
      #local cs=1;
      #while ((cs < Nb_Segments) & ok)
	#if (!Is_Free_Position (SegPos[CW][cs], SegRad[CW][cs]))
	  #local ok = false;
	#end
	#local cs = cs + 1;
      #end
      // If it's all in free space, we display and eventually save it !
      #if (ok)
        #declare Last_Worm=union {
	  #local cs=0;
	  #while (cs < Nb_Segments)
	    sphere { SegPos[CW][cs], SegRad[CW][cs] }
	    #local cs = cs + 1;
	  #end
	}
	#declare CW = CW + 1;
      #end
    #end
    #local conf_attempts = conf_attempts + 1;
  #end
  #local pos_attempts = pos_attempts + 1;
#end
#declare nb_p=pos_attempts;
#declare nb_c=conf_attempts;
#end

// ----------------------------------------
#macro Save_All_Worms (filename)
#fopen Fstream filename write
#write (Fstream, concat ("#declare Max_Worms = ", str (CW, 0, 0), ";\n"))
#write (Fstream, concat ("#declare Nb_Segments = ", str (Nb_Segments, 0, 0), ";\n\n"))
#write (Fstream, concat ("#declare SegPos = array [Max_Worms][Nb_Segments] {"))
#local i = 0;
#while (i < CW)
  #local j = 0;
  #write (Fstream, "\n  {")
  #while (j < Nb_Segments)
    #write (Fstream, "<", vstr(3, SegPos[i][j], ", ", 0,4), ">")
    #if (j = Nb_Segments - 1)
      #write (Fstream, "}")
    #else
      #write (Fstream, ", ")
    #end
    #local j = j + 1;
  #end
  #if (i = CW - 1)
    #write (Fstream, "};\n\n")
  #else
    #write (Fstream, ",")
  #end
  #local i = i + 1;
#end
#write (Fstream, concat ("#declare SegRad = array [Max_Worms][Nb_Segments] {"))
#local i = 0;
#while (i < CW)
  #local j = 0;
  #write (Fstream, "\n  {")
  #while (j < Nb_Segments)
    #write (Fstream, str(SegRad[i][j], 0,4))
    #if (j = Nb_Segments - 1)
      #write (Fstream, "}")
    #else
      #write (Fstream, ", ")
    #end
    #local j = j + 1;
  #end
  #if (i = CW - 1)
    #write (Fstream, "};\n\n")
  #else
    #write (Fstream, ",")
  #end
  #local i = i + 1;
#end
#fclose Fstream
#end

// ----------------------------------------
#declare i = 0;

#macro Add_Worms (scaling, shape, obj_container, end_tests, TheSeed)
#local a = 0;
#local b = 0;

#declare continue = true;
#while ((i < Max_Worms) & continue)
  Make_Worm (scaling, shape, obj_container, TheSeed, continue, a, b, end_tests)
  #if (continue)
    #debug concat ("Worn nb ", str(i, 4, 0), "  (pos ", str(a,2,0), ", conf ", str (b,2,0), ") - (", str(scaling,0,3), ")\n")
    object { Last_Worm }
  #end
  #declare i = i + 1;
#end
#end

// ----------------------------------------
// Container
#declare cont=difference {
  sphere { <0,1,0>, 1 }
  sphere { <0,1,0>, 2 translate <0,-1.35,0> }
}

// ----------------------------------------
#include "main_droite_pov.inc"
#local Main=object {
  Main_droite
  scale 6
  translate <-0.3,-0.3,0.3>
  rotate <0,0,-90>    
  rotate <0,-90,0>
  translate <-1,6,-1>
}
  
#local Rad = 6.3;                                                 

// Equations definissant le trou du vase
#local Ray_Angle=45;

#local R=22/((2*pi)*cos(radians(Ray_Angle)));

#debug concat ("Rayon : ", str (R,0,2), "\n")

#declare fn_creneaux_ri = function(x,y,z) { -(1.3+x) }
#declare f_diam= function (x,y,z) { 1.6 + sin(1.5*pi*y/6.5)/1.5 }

//#declare cont=object { Main }

// ----------------------------------------
#declare Max_Worms=100000;
#declare Nb_Segments=12;

#declare SegPos = array [Max_Worms][Nb_Segments];
#declare SegRad = array [Max_Worms][Nb_Segments];

#declare BaseWorm = array [Nb_Segments]
{ <0.04,0.036>, <0.06,0.046>, <0.09,0.056>, <0.11,0.066>, <0.12,0.072>, <0.125,0.074>,
  <0.125,0.074>, <0.12,0.072>, <0.11,0.066>, <0.09,0.056>, <0.07,0.046>, <0.04,0.036> };

// current number of worms.
#declare CW=0;

#local MySeed = seed (0);

//object { cont pigment { rgbt <0,0,1,0.95> } }

#local level = 0;
#while (level < 60)

#declare fn_Env = function { max (f_sphere (x,y,z,Rad) , -f_sphere (x,y,z,Rad-0.7*pow (0.98, level)), -fn_creneaux_ri (f_diam(x,y,z)*sqrt(x*x+z*z)-R, y, R*acos(x/sqrt(x*x+z*z)))) } 

#declare ISO=isosurface {
  function { fn_Env (x, y, z) }
  contained_by { box { <-3.2, 3.5, -3.2>, <3.2, Rad+0.01, 3.2> } }
  all_intersections
  accuracy 0.001
  max_gradient 30
  pigment { rgb 1 }
}

#declare cont=difference {
  object { ISO }
  union {
    cylinder {
      <0,0,0>, <0,1.3, 0>, 0.23
      rotate <-10,0,0>
      translate <-0.95,5,0.3>
    }
    cylinder {
      <-0.9,0,0>, <1.1,0,0>, 0.23
      scale <1,3,1.5>
      rotate <15,4,0>
      translate <0,5.7,0.8>
    }
    cylinder {
      <-0.7,0,0>, <0,0,0>, 0.23
      scale <1,3,1.5>
      rotate <15,-10,0>
      translate <0,5.9,0.5>
    }
  }
}


  Add_Worms (pow (0.98, level), BaseWorm, cont, 15, MySeed)
  Save_All_Worms ("worm_pos_vase_local.inc")
#undef fn_Env
  #local level = level + 1;
#end
