#include "colors.inc"
#include "rand.inc"
#include "saturn_rings.inc"
#include "stars.inc"
#include "stones.inc"
#include "textures.inc"

// the oneill space habitat

#declare Hull_Texture = texture { Aluminum }
#declare Hull_Length = 30000;
#declare Wall_Thickness = 100;
#declare Hull_Radius = 3000;

// grid function for the glass panels
#declare MyGrid = function(x, y, z) {
  abs( mod( z, 1 )) > 0.9 |
  abs(mod( degrees(atan2(y, x)), 5)) > 4.9
}

#declare Glass_Texture = texture {
  function { MyGrid(x, y, z) }
  texture_map {
    [0 Glass]
    [1 Aluminum]
  }
  scale 100
}


#declare Turf_Texture =
texture {
  bozo
  texture_map {
    // this is water
    [0.30
      pigment { 
	rgbf <0.0, 0.0, 1.0, 0.9>
      }
      normal {
	ripples 0.75
	frequency 10
      }
      finish {
	reflection {0.3, 1 fresnel}
	conserve_energy
      } ]
    // this is vegetation
    [0.30
      pigment {
	bozo
	color_map {
	  [0.2  color DarkBrown]
	  [0.4  color ForestGreen]
	  [0.8  color DarkGreen ]
	}
      }
      normal { bumps }
      // scale down to counteract the global scale
      scale 0.1 ]
  }
  scale 300
}

#macro Make_Panel( Radius, Height, Thickness, Sweep )
#local Internal_Radius = Radius - Thickness;
intersection {
  difference {
    cylinder {
      <0, 0, 0>, <0, 0, Height>, Radius
      open
      hollow
    }
    cylinder {
      <0, 0, 0>, <0, 0, Height>, Internal_Radius
      open
      hollow
    }
  }

  box {
    <-Radius, 0, 0>
    <Radius, Radius, Height>
  }

  box {
    <-Radius, 0, 0>
    <Radius, -Radius, Height>
    rotate <0, 0, Sweep>
  }
}
#end

#declare Panel = object {
  Make_Panel( Hull_Radius, Hull_Length, Wall_Thickness, 60 ) 
}

#declare Turf = object {
  Make_Panel( Hull_Radius-Wall_Thickness, Hull_Length, 10, 60 ) 
}

#declare Reflector = object {
  Make_Panel( Hull_Radius, Hull_Length, 1, 60 )
  rotate <0, 0, -30>
  translate <-Hull_Radius, 0, 0>
  rotate <0, 45, 0>
  translate <Hull_Radius, 0, 0>
  rotate <0, 0, 30>
}

#declare Solid_Panels = union {
  union {
    object { Panel }
    object { Panel rotate <0, 0, 120> }
    object { Panel rotate <0, 0, 240> }
    texture { Hull_Texture }
  }
  union {
    object { Turf }
    object { Turf rotate <0, 0, 120> }
    object { Turf rotate <0, 0, 240> }
    texture { Turf_Texture }
  }  
} 

#declare Glass_Panels = union {
  object { Panel rotate <0, 0, 60>}
  object { Panel rotate <0, 0, 180> }
  object { Panel rotate <0, 0, 300> }
  texture { Glass_Texture }
}

#declare Reflector_Panels = union {
  object { Reflector rotate <0, 0, 60>}
  object { Reflector rotate <0, 0, 180> }
  object { Reflector rotate <0, 0, 300> }
  texture { Polished_Chrome }
}

#declare Plate = object {
  cylinder {
    <0, 0, 0>, <0, 0, Wall_Thickness>,
    Hull_Radius
  }
  texture { Hull_Texture }
}


#declare oneill = union {
  object { Solid_Panels }
  object { Glass_Panels }
  object { Reflector_Panels }
  object { Plate translate <0, 0, -Wall_Thickness> }
  object { Plate translate <0, 0, Hull_Length> }
  inverse
}

// the planet Saturn with rings and texture,
// as provided by Sigmund Kyrre Aas <as@stud.ntnu.no>  
// used with permission
#declare Saturn=union {
  sphere { 0, 60368   // 1 unit equals 1 km 
    scale <1,0.9,1>
    pigment { 
      image_map {
	jpeg "saturn.jpg"
	map_type 1
      }
    }
  }
  object {
    Saturn_rings
    scale 74500     // 1 unit equals 1 km
  }
}


//
// Macros 
//

// this is an awful hack
// and the Window parameter is ignored
#macro Make_Windows( Bottom_Left, Top_Right, Offset, Window )
#local W = ( Top_Right.x - Bottom_Left.x );
#local H = ( Top_Right.y - Bottom_Left.y );
#local Windows = union {
  #declare XCounter = Bottom_Left.x;
  #while ( XCounter < Top_Right.x - 2*Offset )
    #declare YCounter = Bottom_Left.y;
    #while ( YCounter < Top_Right.y - 2*Offset )
      union {
	difference {
	  box {
	    <0, 0.8, 0>,
	    <1.4, 3.2, -0.2>
	  }
	  box {
	    <0.2, 1, 0.1>,
	    <1.2, 3, -0.3>
	  }
	}
	box {
	  <0.2, 1, -0.0001>,
	  <1.2, 3, -0.1>
	  #if ( SRand( RdmA ) < 0 )
	    pigment { Black }
	  #else 
	    texture {
	      pigment { White }
	      finish { ambient 0.6 }
	    }
	  #end
	  texture { Glass }
	}
	translate <XCounter, YCounter, 0>
      }
      #declare YCounter = YCounter + 4;
    #end
    #declare XCounter = XCounter + 3;
  #end
}
#declare Real_Size = max_extent(Windows)-min_extent(Windows);
object {
  Windows
  translate <0.5*(W-Real_Size.x), 0.5*(H-Real_Size.y), Bottom_Left.z>
}
#end

#macro Make_Door( Width, Height )
difference {
  box {
    <0, 0, -0.1>, <Width, Height, 0.1>
  }
  #declare XCounter = 1/9;
  #while ( XCounter < 1 )
    #declare YCounter = 1/13;
    #while ( YCounter < 1 )
      box {
	<0, 0, 0>, <3*Width/9, 3*Height/13, -0.2>
	translate < XCounter*Width, YCounter*Height, 0>
      }
      #declare YCounter = YCounter + 4/13;
    #end
    #declare XCounter = XCounter + 4/9;
  #end
}
#end


//
// Textures
//
#declare White_Concrete = texture {
  pigment { rgb < 1, 1, .8>  }
  normal { bumps 0.1 scale .2 }
  finish {
    specular .1 
    roughness 1
  }
}

#declare Grey_Concrete = texture {
  pigment { rgb < .9, .9, .8> }
  normal { bumps .1 scale .2 }
  finish {
    specular .1 
    roughness 1
  }
}

#declare Concrete_Blocks = texture {
  pigment {
    cells
    pigment_map {
      [0 rgb < .9, .9, .8> ]
      [1 rgb < .8, .8, .7> ]
    }
  }
  finish {
    specular .1 
    roughness 1
  }
}

#declare Pink_Concrete = texture {
  pigment { rgb < 1, .95, .8>  }
  normal { bumps .1 scale .2 }
  finish {
    specular .1 
    roughness 1
  }
}

#declare Office_Glass_Texture = texture {
  function { MyGrid(x, y, z) }
  texture_map {
    [0 Bronze_Texture]
    [1 Aluminum]
  }
  scale 4
}

#declare Dome_Glass = texture {
    pigment { rgbf <1,1,1,0.5> }
    finish {
        ambient 0
        diffuse 0.5
        reflection 0.9
        phong 0.3
        phong_size 60
    }
}

//
// Parts of Buildings
//
#declare Steps = union {
  #declare Counter = 1;
  #while ( Counter < 8 )
    cylinder {
      <0, 0, -36>, <0, .5*Counter, -36>, 22 - 2*Counter
    }
    #declare Counter = Counter + 1;
  #end
}

#declare Decoration1 =  lathe {
  quadratic_spline
  3,
  <-1, -16>,
  <0, -9>,
  <4, 0>
}

#declare Decoration2 = difference {
  box {
    <-0.5, 0, 2>, <4, -4, -2>
  }
  cylinder {
    <4, -4, -2.5>, <4, -4, 2.5>, 2
  }
}

#declare Window1 = union {
  difference {
    box {
      <0, 0.8, 0>,
      <1.4, 3.2, -0.2>
    }
    box {
      <0.2, 1, 0.1>,
      <1.2, 3, -0.3>
    }
  }
  box {
    <0.2, 1, -0.0001>,
    <1.2, 3, -0.1>
    pigment { Black }
    texture { Glass }
  }
}

#declare Window2 = union {
  difference {
    cylinder {
      <0, 0, 0>, <0, 0, -0.5>, 2
    }
    cylinder {
      <0, 0, 0.1>, <0, 0, -0.6>, 1.8
    }
  }
  cylinder {
    <0, 0, -0.0001>, <0, 0, -0.09999>, 1.8
    pigment { Black }
    texture { Glass }
  }
  #declare Counter1 = 0;
  #while ( Counter1 < 180 )
    box {
      <-1.9, -0.05, -0.5>, <1.9, 0.05, -0.1>
      rotate <0, 0, Counter1>
    }
    #declare Counter1 = Counter1 + 45;
  #end
}

#declare Cylinder_Part = union {
  cylinder {
    <0, 0, 0>, <0, 8, 0>, 20
  }
  cylinder {
    <0, 8, 0>, <0, 40, 0>, 20
    texture { Office_Glass_Texture }
  }
  cylinder {
    <0, 40, 0>, <0, 48, 0>, 16
  }
  cylinder {
    <0, 48, 0>, <0, 92, 0>, 20
    texture { Office_Glass_Texture }
  }
  cylinder {
    <0, 92, 0>, <0, 96, 0>, 20
  }
  cylinder {
    <22, 0, 0>, <22, 92, 0>, 2
    rotate <0, -30, 0>
    texture { Glass }
  }
  cylinder {
    <22, 92, 0>, <22, 96, 0>, 2
    rotate <0, -30, 0>
    texture { Aluminum }
  }
  box {
    <0, 0, -3>, <50, 4, 3>
    texture { 
      gradient y
      texture_map {
	[0.1 Aluminum]
	[0.1 Glass]
      }
      scale 4
    }
    translate <10, 42, 0>
    rotate <0, 150, 0>
  }
}
    
#declare R = 3;
#declare P1 = <0, 0, 1.5>;
#declare P2 = vrotate( P1, <0, 10, 0> ); 
#declare P3 = <0, 0, 2>;
#declare P4 = vrotate( P3, <0, 20, 0> );
#declare P5 = vrotate( <0, 0, R>, <0, 10, 0> );
#declare Slice1 = prism {
  linear_spline
  -0.1, 0.1,
  5,
  0, <P2.x, P2.z>, <P3.x, P3.z>
  <-P2.x, P2.z>, 0
}
#declare Slice2 = prism {
  linear_spline
  -0.1, 0.1,
  5,
  <P3.x, P3.z>, <P2.x, P2.z>, 
  <P4.x, P4.z>, <P5.x, P5.z>,
  <P3.x, P3.z>
}

#declare Star = union {
  #declare Angle = 0;
  #while ( Angle < 360 )
    object {
      Slice1
      rotate <0, Angle, 0>
      texture {
	pigment { Sienna }
	normal{
	  crackle 0.6
	  scale 0.24
	}
	finish{phong 0.8 phong_size 200}
      }

    }
    object {
      Slice1
      rotate <0, Angle+20, 0>
      texture {
	pigment { DarkGreen }
	normal{
	  crackle 0.6
	  scale 0.24
	}
	finish{phong 0.8 phong_size 200}
      }
    }
    object {
      Slice2
      rotate <0, Angle, 0>
      texture {
	pigment { Thistle }
	normal{
	  crackle 0.6
	  scale 0.24
	}
	finish{phong 0.8 phong_size 200}
      }
    }
    object {
      Slice2
      rotate <0, Angle, 180>
      texture {
	pigment { Salmon }
	normal{
	  crackle 0.6
	  scale 0.24
	}
	finish{phong 0.8 phong_size 200}
      }
    }
    #declare Angle = Angle + 40;
  #end
}
  
#declare StarCircle = union {
  difference {
    cylinder {
      <0, -0.1, 0>, <0, 0.1, 0>, R
      texture {
	pigment { Copper }
	normal{
	  crackle 0.6
	  scale 0.24
	}
	finish{phong 0.8 phong_size 200}
      }
    }
    object {
      Star
      scale <1, 2, 1>
    }
  }
  object {
    Star
  }
}

#declare R1 = (R-0.1)*(3 - 2*sqrt(2) );
#declare Mosaic = union {
  difference {
    box { 
      <-R, -0.1, -R>, <R, 0.1, R>
      texture {
	pigment { Firebrick }
	normal{
	  crackle 0.6
	  scale 0.24
	}
	finish{phong 0.8 phong_size 200}
      }
    }
    cylinder {
      <0, -0.2, 0>, <0, 0.2, 0>, R+0.1
      pigment { White }
    }
//      #declare Angle = 45;
//      #while ( Angle < 360 )
//        cylinder {
//  	<0, -0.2, 0>, <0, 0.2, 0>, R1
//  	translate <0, 0, R+R1+0.1>
//  	rotate <0, Angle, 0>
//  	pigment { White }
//        }
//        #declare Angle = Angle + 90;
//      #end
  }
  difference {
    cylinder {
      <0, -0.1, 0>, <0, 0.1, 0>, R+0.1
      texture { Brass_Metal }
    }
    cylinder {
      <0, -0.2, 0>, <0, 0.2, 0>, R
      pigment { White }
    }
  }
  object {
    StarCircle 
  }
//    #declare Angle = 45;
//    #while ( Angle < 360 )
//      cylinder {
//        <0, -0.1, 0>, <0, 0.1, 0>, R1
//        //StarCircle
//        //scale <R1/R, 1, R1/R>
//        translate <0, 0, R+R1+0.1>
//        rotate <0, Angle, 0>
//        texture { Copper_Metal }
//      }
//      #declare Angle = Angle + 90;
//    #end
  difference {
    box {
      <-R-0.1, -0.1, -R-0.1>,
      <R+0.1, 0.1, R+0.1>
      texture { Brass_Metal }
    }
    box { 
      <-R, -0.2, -R>, <R, 0.2, R>
      pigment { White }
    }  
  }
}
  

//
// Lights
//

#declare Dome = union {
  #local R = 12;
  sphere {
    0, R
    hollow
    texture { Dome_Glass }
  }
  // struts
  union {
    #declare Counter1 = 0;
    #while ( Counter1 < 180 )
      torus {
	sin(radians(Counter1))*R+0.05, 0.1
	rotate <0, 0, 90>
	translate <cos(radians(Counter1))*R, 0, 0>
      }
      torus {
	sin(radians(Counter1))*R+0.05, 0.1
	rotate <90, 0, 0>
	translate <0, 0, cos(radians(Counter1))*R>

      }
      #declare Counter1 = Counter1 + 10;
    #end
    texture { Aluminum }
  }
  light_source {
    0, rgb 0.2
    fade_distance 25
    fade_power 1   
    looks_like {
      sphere {
	0, 1
	hollow
	pigment{ rgbt 1}
	interior{ 
	  media{ 
	    emission rgb 4
	    density{
	      spherical
	    }
	  }
	}
	finish { ambient 1 }
      }
    }
    translate <0, 0.8*R, 0>
  }
  clipped_by { 
    plane { 
      -y, -0.75*R
    }
  }
  translate <0, -0.75*R, 0>
}

#declare Arch = sphere_sweep {
  cubic_spline
  7,
  <-3, 9, 0>, .2,
  <-2, 4, 0>, .2,
  <-1, 1, 0>, .1,
  <0, 0, 0>, .1,
  <1, 1, 0>, .1,
  <2, 4, 0>, .2,
  <3, 9, 0>, .2
  rotate <180, 0, 0>
  translate <0, 4, 0>
}

#declare Street_Lamp = union {
  object {
    Arch
    rotate <180, 0, 0>
    scale 0.25
    translate <0, 5, 0>
  }
  object {
    Arch
    rotate <180, 90, 0>
    scale 0.25
    translate <0, 5, 0>
  }
  cylinder {
    <0, 0, 0>, <0, 3.9, 0>, 0.1
  }
  sphere {
    <0, 3.9, 0>, 0.1
  }
  difference {
    cylinder {
      <0, 5, 0>, <0, 5.05, 0>, 0.7
    }
    cylinder {
      <0, 4.99, 0>, <0, 5.06, 0>, 0.45
    }
  }    
  difference {
    sphere {
      <0, 5, 0>, 0.6
    }
    sphere {
      <0, 5, 0>, 0.55
    }
    box {
      <-1, 0, -1>, <1, 5, 1>
    }
  }
  light_source {
    0, rgb 0.2
    fade_distance 5
    fade_power 1   
    looks_like {
      sphere {
	0, 0.3
	hollow
	pigment{ rgbt 1}
	interior{ 
	  media{ 
	    emission rgb 4
	    density{
	      spherical
	    }
	  }
	}
	finish { ambient 1 }
      }
    }
    translate <0, 4.7, 0>
  }

  texture { Aluminum }
}


// 
// Buildings
//

// a stepped cylinder
#declare Cylindrical_Building =
difference {
  union {
    #declare Counter = 1;
    #declare Accumulator = 0;
    #while ( Counter > 0 )
      union {
	cylinder {
	  <0, 0, 0>, <0, 40, 0>, 40
	}
	cylinder {
	  <0, 40, 0>, <0, 39, 0>, 45
	}
	#declare Counter1 = 0;
	#while ( Counter1 < 360 ) 
	  box {
	    <-1, 0, -2>, <1, 40, 2>
	    translate <40, 0, 0>
	    rotate <0, Counter1, 0>
	  }
	  object {
	    Decoration2
	    translate <40, 40, 0>
	    rotate <0, Counter1, 0>
	  }
	  #declare Counter1 = Counter1 + 60;
	#end
	#declare Counter1 = 15;
	#while ( Counter1 < 360 )
	  union {
	    difference {
	      cylinder {
		<-0.5, 0, 0>, <0.5, 0, 0>, 3
	      }
	      cylinder {
		<-0.6, 0, 0>, <0.6, 0, 0>, 2.8
	      }
	    }
	    cylinder {
	      <0, 0, 0>, <0.05, 0, 0>, 2.8
	      texture { Brass_Metal }
	      texture { Glass }
	    }
	    translate <40, 30, 0>
	    rotate <0, Counter1, 0>
	  }
	  #declare Counter1 = Counter1 + 30;
	#end
	scale Counter
	translate <0, Accumulator, 0>
      }
      #declare Accumulator = Accumulator + 40 * Counter;
      #declare Counter = Counter - .25;
    #end
    object {
      Steps
      rotate <0, -60, 0>
    }
  }
  // the entrance
  box {
    <0, 0, 0>, <8, 16, 2>
    translate <-2, 3.5, -40>
    rotate <0, -60, 0>
  }
}




// a structure similar to a tv tower
#declare TV_Tower = union {
  cylinder {
    <0, 0, 0>, <0, 95, 0>, 10
  }
  sphere {
    <0, 120, 0>, 30
  }

  cylinder {
    <0, 119, 0>, <0, 121, 0>, 45
  }
  cylinder {
    <0, 38, 0>, <0, 40, 0>, 25
  }
}

// a long, square building with an apsis
#declare Long_Building = difference {
  union {
    // the apsis
    cylinder {
      <0, 0, 0>, <0, 24, 0>, 15
    }
    // the transparent dome on the roof
    object {
      Dome
      translate <0, 24, 0>
    }
    // the main rectangular part
    box {
      <-15, 0, 0>, <15, 24, 90>
    }
    // the base
    #declare Base = union {
      cylinder {
	<0, 0, 0>, <0, 1, 0>, 17
      }
      box {
	<-17, 0, 0>, <17, 1, 92>
      }
    }
    object {
      Base
    }
    // the ledges
    object {
      Base
      translate <0, 20, 0>
    }
  }
  // the slits
  #declare Count = -90;
  #while ( Count < 100 )
    object {
      box {
	<-0.5, 3, -0.5>,
	<0.5, 19, 0.5>
      }
      translate <0, 0, -15>
      rotate <0, Count, 0>
    }
    #declare Count = Count + 15;
  #end
}


// A tower, or spire
#declare Width = 40;
#declare Height = 120;
#local Width3 = Width/2;
#local Width2 = 1.05*Width3;
#declare Spire = union {
  difference {
    union {
      box {
	<-Width2, 0, -Width2>, <Width2, 3, Width2>
      }
      box {
	<-Width3, 0, -Width3>, <Width3, 0.8*Height, Width3>
      }
      prism {
	conic_sweep
	linear_spline
	0,  // height 1
	1,  // height 2
	5,  // the number of points making up the shape...
	<1, 1>, <1, -1>, <-1, -1>, <-1, 1>, <1, 1>
	rotate <180, 0, 0>
	scale <0.35*Width, 0.2*Height, 0.35*Width>
	translate <0, Height, 0>
      }
      // the steps
      union {
	#local Counter = 0;
	#while ( Counter < 10 )
	  box {
	    <-Width/10, 0, 0>, <Width/10, .3, -10>
	    scale <2 - Counter/10, 1, 1 - Counter/10>
	    translate <0, .3*Counter, -Width3>
	  }
	  #declare Counter = Counter + 1;
	#end
	texture { White_Concrete }
      }
    }
    // the entrance way
    box {
      <-Width/10, 3.00000001, -Width2>, <Width/10, 15, -0.9*Width2>
    }
    // the corner grooves
    #declare Groove = sphere_sweep {
      linear_spline
      2,
      <0, .15*Height, 0>, Width/10,
      <0, Height, 0>, Width/10
    }
    object {
      Groove
      translate <-Width2, 0, -Width2>
    }
    object {
      Groove
      translate <Width2, 0, -Width2>
    }
    object {
      Groove
      translate <-Width2, 0, Width2>
    }
    object {
      Groove
      translate <Width2, 0, Width2>
    }
    //texture { White_Concrete }
  }
  object {   
    Make_Door( 2, 6 )
    translate <-2, 3, -0.9*Width2-0.1>
    texture { Brass_Metal }
  }
  object {   
    Make_Door( 2, 6 )
    translate <0, 3, -0.9*Width2-0.1>
    texture { Brass_Metal }
  }
  #declare Mors = text {
    ttf "timrom.ttf" "MORS OMNIA AEQVAT", 1, 0
    scale 2.5
  }
  #declare DX = max_extent( Mors );
  object {
    Mors
    translate <-0.5*DX.x, 20, -Width3-0.1>
    pigment { Black }
  }
  // the windows
  #declare Windows = object {
    Make_Windows( <-20, 20, -20>, <20, 96, 20>, 5, Window1 )
  }
  #declare Rotation = 0;
  #while ( Rotation < 360 )
    object {
      Windows
      rotate <0, Rotation, 0>
    }
    #declare Rotation = Rotation + 90;
  #end
}  

// a spire and two wings enclosing a courtyard
#declare Main_Building = union {
  union {
    // the spire
    object {
      Spire
    }
    // the transept
    box {
      <-35, 0, -10>,
      <35, 16, 10>
    }
    box {
      <-35, 0, -11>,
      <35, 1, 11>
    }
    translate <0, 0, 70>
  }
  // the wings
  object {
    Long_Building
    translate <-45, 0, 0>
  }
  object {
    Long_Building
    translate <45, 0, 0>
  }
  // the base plate
  box {
    <-62, -1, -19>,
    <62, 0, 94>
    texture { Grey_Concrete }
  }
  // a mosaic
  object {
    Mosaic
    scale <4, 1, 4>
    translate <0, -0.099, 10>
  }
  // the lights
  #declare Counter1 = -10;
  #while ( Counter1 < 60 )
    object {
      Street_Lamp
      translate <15, 0, Counter1>
    }
    object {
      Street_Lamp
      translate <-15, 0, Counter1>
    }
    #declare Counter1 = Counter1 + 20;
  #end

  translate <0, 1, 0>
  texture { White_Concrete }
}


// another spire
#declare Spire2 = union {
  #local R = sqrt(2)*20;
  box {
    <-20, 0, -20>, <20, 100, 20>
  }
  // the dome
  intersection {
    box {
      <-20, 0, -20>, <20, 300, 20>
    }
    sphere {
      0, R
      scale <1, 1.61, 1>
      translate <0, 100, 0>
    }
  }
  // the copper roof
  union {
    sphere {
      <0, 0, 0>, R
    }
    // seams
    #declare Counter1 = 0;
    #while ( Counter1 < 180 )
      torus {
	R+0.05, 0.1
	rotate <90, Counter1, 0>
      }
      #declare Counter1 = Counter1 + 5;
    #end
    #declare Counter1 = 0;
    #while ( Counter1 < 180 )
      torus {
	sin(radians(Counter1))*R+0.05, 0.1
	//rotate <0, 0, 90>
	translate <0, cos(radians(Counter1))*R, 0>
      }
      #declare Counter1 = Counter1 + 5;
    #end
    scale <1, 1.61, 1> 
    translate <0, 100, 0>
    texture { 
      Copper_Texture
    }
    clipped_by {
      box {
	<-20, 0, -20>, <20, 300, 20>
      }
    }
  }
  #declare Roof_Rim = torus {
    20.05, 0.1
    clipped_by {
      box {
	<-21, -1, -21>, <21, 1, 0>
      }
    }
    rotate <90, 0, 0>
    scale <1, 1.61, 1>
    translate <0, 100, -20>
    texture { Copper_Texture }
  }
  // the spike on the roof
  cone {
    <0, 130, 0>, 2,
    <0, 160, 0>, 0
    texture { Aluminum }
  }
  box {
    <-25, 0, -10>, <25, 100, 10>
  }
  cylinder {
    <-25, 0, 0>, <25, 0, 0>, 10
    scale <1, 1.61, 1>
    translate <0, 100, 0>
  }
  box {
    <-10, 0, -25>, <10, 100, 25>
  }
  cylinder {
    <0, 0, -25>, <0, 0, 25>, 10
    scale <1, 1.61, 1>
    translate <0, 100, 0>
  }
  box {
    <-30, 0, -15>, <30, 50, 15>
  }
  box {
    <-15, 0, -30>, <15, 50, 30>
  }
  // the windows
  #declare Windows = union {
    Make_Windows( <-15, 6, -30>, <15, 50, -30>, 2, Window1 )
    Make_Windows( <-10, 52, -25>, <10, 105, -25>, 2, Window1 )
    Make_Windows( <-20, 52, -20>, <-10, 105, -20>, 2, Window1 )
    Make_Windows( <10, 52, -20>, <20, 105, -20>, 2, Window1 )
  }
  #declare Counter1 = 0;
  #while ( Counter1 < 360 )
    object { 
      Windows
      rotate <0, Counter1, 0>
    }
    object { 
      Roof_Rim
      rotate <0, Counter1, 0>
    }
    #declare Counter1 = Counter1 + 90;
  #end
  texture { White_Concrete }
}

// spire with base plate and lights
#declare Spire_Building = 
union {
  object {
    Spire2
    translate <0, 1, 0>
  }
  box {
    <-30, 0, -30>,
    <30, 1, 30>
    texture { Grey_Concrete }
  }
  light_source {
    <-28, 1, -28>, color White
    spotlight
    fade_distance 25
    fade_power 1   
    radius 15
    falloff 20
    tightness 10
    point_at <0, 120, 0>
  }
  light_source {
    <-28, 1, 28>, color White
    spotlight
    fade_distance 25
    fade_power 1   
    radius 15
    falloff 20
    tightness 10
    point_at <0, 120, 0>
  }
  light_source {
    <28, 1, -28>, color White
    spotlight
    fade_distance 25
    fade_power 1   
    radius 15
    falloff 20
    tightness 10
    point_at <0, 120, 0>
  }
  light_source {
    <28, 1, 28>, color White
    spotlight
    fade_distance 25
    fade_power 1   
    radius 15
    falloff 20
    tightness 10
    point_at <0, 120, 0>
  }
}

// a modern office building
// doesn't really fit in well
#declare Cylinder_Building = union {
  object {
    Cylinder_Part
    translate <30, 0, 0>
  }
  object {
    Cylinder_Part
    translate <30, 0, 0>
    rotate <0, 120, 0>
  }
  object {
    Cylinder_Part
    translate <30, 0, 0>
    rotate <0, 240, 0>
  }
}


// makes a single floor of the following building
#macro Make_Floor()
#local Counter1 = 0;
union {
  box {
    <-40, 0, -20>, <40, 4, 20>
  }
  cylinder {
    <-40, 0, 0>, <-40, 4, 0>, 20
  }
  cylinder {
    <40, 0, 0>, <40, 4, 0>, 20
  }
  // top ledge
  box {
    <-40.5, 3.5, -20.5>, <40.5, 4, 20.5>
  }
  cylinder {
    <-40, 3.5, 0>, <-40, 4, 0>, 20.5
  }
  cylinder {
    <40, 3.5, 0>, <40, 4, 0>, 20.5
  }
  #declare Counter1 = -40;
  #while ( Counter1 < 50 )
    object {
      Make_Windows( <-4, 0, -20>, <4, 4, -20>, 0, Window1 )
      translate <Counter1, -1, 0>
    }
    object {
      Make_Windows( <-4, 0, -20>, <4, 4, -20>, 0, Window1 )
      rotate <0, 180, 0>
      translate <Counter1, -1, 0>
    }
    #declare Counter1 = Counter1 + 20;
  #end
  #declare Counter1 = -30;
  #while ( Counter1 < 40 )
    union {
      box {
	<-5, 0, -25>, <5, 4, 25>
      }
      box {
	<-5.5, 3.5, -25.5>, <5.5, 4, 25.5>
      }
      object {
	Make_Windows( <-4, 0, -25>, <4, 4, -25>, 0, Window1 )
	translate <0, -1, 0>
      }
      object {
	Make_Windows( <-4, 0, -25>, <4, 4, -25>, 0, Window1 )
	translate <0, -1, 0>
	rotate <0, 180, 0>
      }
      translate <Counter1, 0, 0>
    }
    #declare Counter1 = Counter1 + 20;
  #end

}
#end

// a large building
#declare Tenement = union {
  
  #local Base = union {
    box {
      <-45, 0, -25>, <45, 4, 25>
    }
    cylinder {
      <-40, 0, 0>, <-40, 4, 0>, 25
    }
    cylinder {
      <40, 0, 0>, <40, 4, 0>, 25
    }
  }
  object {
    Base
    scale <1.1, 0.25, 1.1>
  }
  object {
    Base
    translate <0, 1, 0>
  }
  #declare Counter1 = 5;
  #while ( Counter1 < 61 )
    object {
      Make_Floor()
      translate <0, Counter1, 0>
    }
    #declare Counter1 = Counter1 + 4;
  #end
  object {
    Base
    translate <0, 61, 0>
  }

}

// LOD buildings
#declare LOD0 = box {
  <-20, 0, -15>,
  <20, 80, 15>
  texture { White_Concrete }
}

#declare LOD1 = union {
  box {
    <-20, 0, -15>,
    <20, 80, 15>
    texture { Pink_Concrete }
  }
  prism {
    conic_sweep
    linear_spline
    0, // height 1
    1, // height 2
    5, // the number of points making up the shape...
    <-20, -15>, <-20, 15>, <20, 15>, <20, -15>, <-20, -15>
    rotate <180, 0, 0>
    scale <1, 20, 1>
    translate <0, 100, 0>
    texture { Copper_Metal }
  }
}
 
#declare LOD2 = sphere {
  <0, 15, 0>, 20
  texture { Grey_Concrete }
}

#declare LOD3 = cylinder {
  <0, 0, 0>, <0, 70, 0>, 17
  texture { Concrete_Blocks }
}

#declare LOD4 = union {
  object { LOD3 }
  sphere {
    <0, 70, 0>, 17
    texture { Copper_Metal }
  }
}

#declare LOD5 = union {
  object { LOD0 }
  object { LOD0 rotate <0, 90, 0> }
}

#declare Buildings = array[6] {
  LOD0,
  LOD1,
  LOD2,
  LOD3,
  LOD4,
  LOD5
}

#macro Make_City() 
union {
  intersection {
    object { 
      Make_Panel( 2890, 600, 1, 12 )
      rotate <0, 0, -6>
      translate <0, 0, 300>
    }
    cylinder {
      <0, 0, 0>, <0, 1, 0>, 300
    }
    texture { Grey_Concrete }
  }    
  #local Angle = -6;
  #while ( Angle <= 6 )
    #local R = 300 - 25*abs(Angle);
    #local Z = -R;
    #while ( Z <= R )
      #local I = RRand(0, 5.99, RdmA);
      object {
	Buildings[I]
	translate <0, -2890, Z>
	rotate <0, 0, Angle+1>
      }
      #declare Z = Z + 50;
    #end
    #declare Angle = Angle + 2;
  #end
}
#end


//
// Transport
//

// maglev tubeway
#declare Tubeway = union {
  difference {
    cylinder {
      <0, 5, 0>, <0, 5, Hull_Length>, 1.5
    }
    cylinder {
      <0, 5, 0>, <0, 5, Hull_Length>, 1.4
    }
    texture { Glass }
  }
  #declare Counter = 0;
  #while ( Counter < Hull_Length )
    union {
      torus {
	2, .5
	rotate <90, 0, 0>
	translate <0, 5, Counter>

      }
      cylinder {
	<0, 0, 0>, <0, 3.5, 0>, .5
	translate <0, 0, Counter>
      }
      texture { Polished_Chrome }
    }
    #declare Counter = Counter + 30;
  #end
}

// Zeppelin
#declare Zeppelin = union {
  #declare R = 30;
  union {
    sphere {
      <0, 0, 0>, R
    }
    // struts
    #declare Counter1 = 0;
    #while ( Counter1 < 180 )
      torus {
	R+0.05, 0.1
	rotate <Counter1, 0, 0>
      }
      #declare Counter1 = Counter1 + 30;
    #end
    #declare Counter1 = 0;
    #while ( Counter1 < 180 )
      torus {
	sin(radians(Counter1))*R+0.05, 0.1
	rotate <0, 0, 90>
	translate <cos(radians(Counter1))*R, 0, 0>
      }
      #declare Counter1 = Counter1 + 10;
    #end
    scale <3, 1, 1>
  }
  // the fins
  #declare Counter1 = 0;
  #while ( Counter1 < 360 )
    union {
      difference {
	union {
	  cone {
	    <0, 0, 0>, 1,
	    <0, 10, 0>, 0.5
	  }
	  sphere {
	    <0, 11, 0>, 1.1
	  }
	  scale <3, 1, 1>
	}
	box {
	  <2.5, 9, -2>, <3.5, 13, 2>
	}
      }
      intersection {
	sphere {
	  <0, 11, 0>, 1.1
	  scale <3, 1, 1>
	}
	box {
	  <2.5, 9, -2>, <3.5, 13, 2>
	}
	texture { Polished_Brass }
      }
      // the propeller
      cylinder {
	<-0.1, 0, 0>,
	<0.1, 0, 0>, 3
	texture { 
	  // fake motion blur
	  pigment{ 
 	    onion
 	    pigment_map {
 	      [0 rgbf 1]	      
 	      [0.7 rgbf <0.71, 0.65, 0.26, 0.5>]
 	      [1 rgbf 1]	      
 	    }
 	    scale 3
 	  }

	  finish{ metallic }
	}
	translate <2.5, 11, 0>
      }
      scale 2
      rotate <Counter1, 0, 0>
      translate <82, 0, 0>
    }
    #declare Counter1 = Counter1 + 90;
  #end
  // gondola
  difference {
    union {
      cylinder {
	<-12, 0, 0>, <12, 0, 0>, 12
      }
      sphere {
	<-12, 0, 0>, 12
      }
      sphere {
	<12, 0, 0>, 12
      }
    }
    union {
      cylinder {
	<-12, 0, 0>, <12, 0, 0>, 11.9
      }
      sphere {
	<-12, 0, 0>, 11.9
      }
      sphere {
	<12, 0, 0>, 11.9
      }
    }
    #declare Counter1 = -11;
    #while ( Counter1 < 12 )
      box {
	<Counter1, -2, -12.5>,
	<Counter1+1.5, -4, 12.5>
      }
      box {
	<Counter1, -6, -12.5>,
	<Counter1+1.5, -8, 12.5>
      }
      #declare Counter1 = Counter1 + 2;
    #end
    #local Round_Windows = union {
      #declare Counter1 = 0;
      #while ( Counter1 <= 180 )
	box {
	  <-0.75, -2, 0>
	  <0.75, -4, 12.75>
	  rotate <0, Counter1, 0>
	}
	box {
	  <-0.75, -2, 0>
	  <0.75, -4, 12.75>
	  rotate <20, Counter1, 0>
	}
	#declare Counter1 = Counter1 + 10;
      #end
      translate <12, 0, 0>
    }
    object {
      Round_Windows
    }
    object {
      Round_Windows
      rotate <0, 180, 0>
    }
    
    hollow
    translate <0, -0.9*R, 0>
  }
  light_source {
    <0, -R-3, 0>
    color White 
    fade_distance 25
    fade_power 1   
  }

  texture { Aluminum }
}






//////////////////////////////
// the scene
//////////////////////////////
camera {
  location  <0, -2870, 0>
  // good view
  look_at   <0, -2845, 100>
}


light_source {
  <-100000, 100000, 0>, rgb 1
  parallel
  point_at 0
}

object {
  oneill
  interior {
    media {
      absorption rgb <1.5, 1, 1>*0.00001
    }
  }
  rotate <0, 0, -1.5>
  translate <0, 0, -1000>
}

object {
  Saturn
  rotate <0, 0, 60>
  translate <40000, 50000, 100000>

}

plane {
  z, 1100000
  texture { 
    Starfield1
  }
  // the dreaded hollow plane problem
  hollow
}


object {
  TV_Tower
  translate <0, -2890, 260>
  rotate <0, 0, -1.5>
  texture { White_Concrete }
  texture { Glass }
}


object {
  Tenement
  rotate <0, 90, 0>
  translate <0, -2890, 150>
  rotate <0, 0, -1.5>
  texture { Pink_Concrete }
}

object {
  Arch
  scale 25
  translate <0, -2890, 260>
  rotate <0, 0, 1.5>
  texture { Brass_Metal }
}

object {
  Main_Building
  translate <0, -2890, 100>
}

object {
  Spire_Building
  translate <0, -2890, 200>
  rotate <0, 0, 2.5>
}

object {
  Zeppelin
  rotate <0, 70, 0>
  translate <100, -2740, 420>
}

object {
  Tubeway
  translate <0, -2890, 0>
  rotate <0, 0, 1.5>
}

object {
  Tubeway
  translate <0, -2890, 0>
  rotate <0, 0, 241.5>
}

object {
  Tubeway
  translate <0, -2890, 0>
  rotate <0, 0, 121.5>
}

object {
  Make_City()
  translate <0, 0, 6000>
  rotate <0, 0, 120>
}

object {
  Make_City()
  translate <0, 0, 7000>
  rotate <0, 0, 240>
}

