/*
	Author:	 Philip Chan
	Last Updated:	 November 30, 2003
	Version:	POV-Ray version 3.5
	
	This include file contains three pigments and three macros for creating stars and starfields.
	
	Stars ( scaling, colours )
		Creates a sky_sphere with red, white and blue stars.
		The parameter:
			scaling:	scale value applied to pigments, recommend value of 1 for 1024x768
			colours:	vector indicating which colours to use: <red, white, blue>
	
	Bright_Star ( translation, z_rotate, tint_select )
		Creates a red, white or blue, "bright" star.	The recommended distance of the star from the
		camera is 1250 units for 1024x768.	The flares coming out of the star currently assumes that
		the vector from the camera to the stars is <delta, delta, a>, where abs(delta) << abs(a).
		The three parameters are as follows:
			translation:	location of the object
			z_rotate:	 rotation of the object about the z-axis (to spin the flares)
			tint_select:	float value, < 1/3 for red, > 2/3 for blue, otherwise white
	
	Bright_Starfield ( stars, box1, box2, randomize )
		Creates a starfield using the Bright_Star macro.	See notes under said macro.
		The four parameters are as follows:
			stars:	number of stars in the starfield
			box1, box2:	 two corners as in the standard POV box object within which the stars are placed.
			randomize:	0 results in only white stars, otherwise the colour of each star is random
*/

// Pigments for a Starfield
#declare NormalStars = pigment {
	leopard
	turbulence 0.5
	colour_map {
		[0.00 colour rgbt <0, 0, 0, 1>]
		[0.97 colour rgbt <0, 0, 0, 1>]
		[0.99 colour rgb <1, 1, 1>]
		[1.00 colour rgb <1, 1, 1>]
	}
	translate <41.76, 38.78, 1.438>
	rotate <190.44, 61.2, 331.92>
}

#declare BlueStars = pigment {
	leopard
	turbulence 0.5
	colour_map {
		[0.00 colour rgbt <0, 0, 0, 1>]
		[0.96 colour rgbt <0, 0, 0, 1>]
		[0.99 colour rgb <.8, .8, 1> ]
		[1.00 colour rgb <.8, .8, 1> ]
	}
	translate <4.77, 83.2, 48.1>
	rotate <18, 57.96, 260.28>
}

#declare RedStars = pigment {
	leopard
	turbulence 0.5
	colour_map {
		[0.00 colour rgbt <0, 0, 0, 1>]
		[0.96 colour rgbt <0, 0, 0, 1>]
		[0.99 colour rgb <1, .9, .9>]
		[1.00 colour rgb <1, .9, .9>]
	}
	translate <13.78, 3.889, 171.1>
	rotate <233.28, 331.56, 71.64>
}

// Starfield
#macro Stars ( scaling, colours )
	sky_sphere {
		#if ( colours.red )
			pigment { RedStars }
		#end
		#if ( colours.green )
			pigment { NormalStars }
		#end
		#if ( colours.blue )
			pigment { BlueStars }
		#end
		scale 0.0075*scaling
	}
#end

// Bright_Star for Starfield, recommended distance from camera approx. 1250
#macro Bright_Star ( translation, z_rotate, tint_select )
	#if ( tint_select < 1/3 )
		#local tint = <0.0, -0.45, -0.45>;
	#end
	#if ( tint_select > 2/3 )
		#local tint = <-0.5, -0.5, 0.0>;
	#end
	#ifndef ( tint )
		#local tint = <-0.2, -0.2, -0.2>;
	#end
	merge {
		sphere { <0.00, 0.00, 0.00>, 5.00
			pigment { colour rgbt <1.00, 1.00, 1.00, 1.00> }
				hollow
				interior {
					media {
						emission 1
						intervals 20
						density {
							spherical
							colour_map {
								[ 0.00 colour rgb <0.50, 0.50, 0.50>*0.11 + tint*0.11 ]
								[ 0.40 colour rgb <1.00, 1.00, 1.00>*0.20 + tint*0.2 ]
								[ 1.00 colour rgb <1.00, 1.00, 1.00>*0.20 + tint*0.2 ]
							}
						scale 5.00
					}
				}
		 }									
	}
	#local i = 0;
	#while ( i < 350 )
		cone { <-4.00, 0.00, 0.00>, 2.00, <-14.0, 0.00, 0.00>, 0.00
			scale <1.00, 1.00, 1.50>
			rotate z*i
		}						
		#local i = i + 60;
	#end
	pigment { colour rgbt <1.00, 1.00, 1.00, 1.00> }
		hollow
		interior {
			media {
				emission 1
				intervals 20
				density {
					spherical
					colour_map {
						[ 0.00 colour rgb <0.25, 0.25, 0.25>*0.1 + tint*0.25 ]
						[ 0.20 colour rgb <1.00, 1.00, 1.00>*0.1 + tint*0.1 ]
						[ 1.00 colour rgb <1.00, 1.00, 1.00>*0.1 ]
					}
					scale 14.00
				}
			}
		}
		rotate z*z_rotate
		translate translation
	}
#end

// random seed for Bright_Starfield macro
#declare Bright_Seed = seed(648);

// creates a starfield of bright stars
#macro Bright_Starfield ( stars, box1, box2, randomize )
	#local i = 0;
	#while ( i < stars )
		#if ( randomize = 0 )
			Bright_Star ( <rand(Bright_Seed)*(box1.x - box2.x) + box2.x,
				rand(Bright_Seed)*(box1.y - box2.y) + box2.y, rand(Bright_Seed)*(box1.z - box2.z) + box2.z>, rand(Bright_Seed)*60, 1/2 )
		#else
			Bright_Star ( <rand(Bright_Seed)*(box1.x - box2.x) + box2.x,
				rand(Bright_Seed)*(box1.y - box2.y) + box2.y, rand(Bright_Seed)*(box1.z - box2.z) + box2.z>, rand(Bright_Seed)*60, rand(Bright_Seed) )
		#end
		#local i = i + 1;
	#end
#end