
/**********************************************************************************
 Persistence of Vision Ray Tracer Scene Description File
 File name   : TdGdesert.pov
 Version     : 3.6
 Description : Desert landscape using isosurfaces
               IRTC entry for July-August competition
 Date        : August 2004
 Author      : Thomas A. M. de Groot
 E-mail      : t.degroot@inter.nl.net
 
 Copyright (C) 2004. All rights reserved.
**********************************************************************************/


#include "rand.inc"
#include "functions.inc"
#include "colors.inc"

#declare Radiosity = true;
#declare Saguaros = false; //put there your cacti or other objects

global_settings {
  #if (Radiosity)
  radiosity {
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.01           // end pretrace at this size
    count 35                      // higher -> higher quality (1..1600) [35]
    nearest_count 5               // higher -> higher quality (1..10) [5]
    error_bound 0.2               // higher -> smoother, less accurate [1.8]
    recursion_limit 1             // how much interreflections are calculated (1..5+) [3]
    low_error_factor .3           // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    brightness 0.3                  // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
    normal on                   // take surface normals into account [off]
    media off                    // take media into account [off]
    always_sample off           // turn sampling in final trace off [on]
    //max_sample 1.0              // maximum brightness of samples
  }
  #end
  noise_generator 2
  assumed_gamma 1.0

}

#declare Loc = 4;						// location of the camera and primary scaling
#declare Slope = true;			// true = slope & altitude; false = gradient texture
#declare SlopeValue = 0.6;	// slope <> altitude ratio

// ----------------------------------------
#declare P1gain = 0.1;	
#declare P1turb = <0.1, 0.5, 0.3>;			
#declare P1scale = <0.1, 0.1, 0.5>;	
#declare P1rotate = <0, 0, 0>;			
#declare P1angle = 45;

#declare P2gain = 0.5;	  
#declare P2turb = <0.5, 0.9, 0.9>;		
#declare P2scale = <0.1, 2.5, 0.1>;		// increasing y reduces breaking-up of P2 (?)
#declare P2rotate = <0, 90, 0>;				
#declare P2angle = -10;							// rotates from vertical (0) to horizontal (90); note: dents and wrinkles rotate in opposite ways!

#declare Iso = true;			// true for isosurface; false for heightfield. Watch out! may give strange results!
#declare Xscale = 75;			// increase size of landscape (not scale!) for X
#declare Zscale = 100;			// increase size of landscape (not scale!) for Z
// ----------------------------------------

// landscape functions:
#declare P1a=function {pattern {wrinkles scale P1scale rotate P1rotate }}
#declare P1b=function {pattern {wrinkles warp {turbulence P1turb octaves 3 omega 0.5 lambda 1} scale P1scale rotate P1rotate}}
#declare P1c=function {pattern {ripples frequency 0.1 warp {turbulence <0.5,0.3,0.5> octaves 3 omega 0.5 lambda 1} scale <0.1, 0.1, 0.1> }}

#declare P2a=function {pattern {dents scale P2scale rotate P2rotate}}
#declare P2b=function {pattern {dents warp {turbulence P2turb octaves 5 omega 0.5 lambda 2} scale P2scale rotate P2rotate}}

#declare P3=function {f_noise3d(x,y,z)*1}
#declare P4=function {f_hetero_mf(x,y,z, 0.8, 2, 1, 0, 0.9, 2)*1 } // H,lacunarity,octaves,offset,gain,ng
#declare P5=function {f_hetero_mf(x,y,z, 0.1, 2, 1, 0, 0.1, 2)*1 } // H,lacunarity,octaves,offset,gain,ng

// rotate y:
#declare P1ca=function {P1c(x*cos(radians(P1angle)) - z*sin(radians(P1angle)), y, x*sin(radians(P1angle)) + z*cos(radians(P1angle)) )}
// rotate z:
#declare P1aa=function {P1a(x*cos(radians(P1angle)) + y*sin(radians(P1angle)), (-x*sin(radians(P1angle)) + y*cos(radians(P1angle)))*2, z)*1}
// rotate z:
#declare P2aa=function {P2a(x*cos(radians(P2angle)) + y*sin(radians(P2angle)), (-x*sin(radians(P2angle)) + y*cos(radians(P2angle)))*2, z)*1}

#declare P=function {  P4(x, y, z)
										 //* P3(x*0.2, 0, z*0.5) // not used
                     //- P1a(x, y, z)*P1gain // not used
                     //+ P2(x, y, z)*P2gain  // not used
                     + P2aa(x, y, z)*P2gain
                     - P1aa(x, y, z)*P1gain
                    }

#declare Q=function { P1ca(x*3, y*3, z*3)
                     -P5(x, y, z)
                    }

//=========================================================================================================================================
//TEXTURES:

// A light colored sand texture (TdG 2001):
#declare TdG_LightSand = 
  texture {      
    pigment {
      agate
      agate_turb 0.3
      color_map {
               [ 0.0     rgbft <1.0, 0.816993, 0.479133, 0.0, 0.0> ]
               [ 0.049822  rgbft <0.939815, 0.699074, 0.518519, 0.0, 0.0> ]
               [ 0.096085  rgbft <0.991333, 0.730905, 0.383667, 0.0, 0.0> ]
               [ 0.149466  rgbft <0.990941, 0.863911, 0.446526, 0.0, 0.0> ]
               [ 0.41637  rgbft <0.925926, 0.739198, 0.407407, 0.0, 0.0> ]
               [ 1.0     rgbft <1.0, 0.971785, 0.729133, 0.0, 0.0> ]
      }
      turbulence 5.0
      lambda 6.0
      frequency 6.0
      ramp_wave
      warp {
        turbulence <5.0, 2.0, 1.0>
      }
    }      
    normal{
      granite , 0.2
      warp {
        turbulence <1.0, 5.0, 2.0>
      }
    }     
    finish {
      ambient 0.01
      specular 0.1
    }
  }

// A dark colored sand texture (TdG 2004):
#declare TdG_DarkSand = 
  texture {      
    pigment {
      agate
      agate_turb 0.3
      color_map {
               [ 0.0     rgbft <1.0, 0.816993, 0.479133, 0.0, 0.0> ]
               [ 0.049822  rgbft <0.939815, 0.699074, 0.518519, 0.0, 0.0>*0.1 ]
               [ 0.096085  rgbft <0.991333, 0.730905, 0.383667, 0.0, 0.0>*0.1 ]
               [ 0.149466  rgbft <0.990941, 0.863911, 0.446526, 0.0, 0.0>*0.1 ]
               [ 0.41637  rgbft <0.925926, 0.739198, 0.407407, 0.0, 0.0>*0.1 ]
               [ 1.0     rgbft <1.0, 0.971785, 0.729133, 0.0, 0.0> ]
      }
      turbulence 5.0
      lambda 6.0
      frequency 6.0
      ramp_wave
      warp {
        turbulence <5.0, 2.0, 1.0>
      }
    }      
    normal{
      granite , 0.2
      warp {
        turbulence <1.0, 5.0, 2.0>
      }
    }     
    finish {
      ambient 0.01
      specular 0.1
    }
  }

#declare CrackleRipple = 
texture {
  average
  texture_map {
     [0.3 pigment {
            color rgb <0.635,0.438,0.302>
          }               
          normal {
            ripples , 0.1
            bump_size 0.1
            turbulence 0.6
            scale  <0.005, 0.008, 0.5>
          }
     ]
     [1.0 pigment {
            color rgb <0.635,0.438,0.302>
          }               
          normal {
            crackle solid
            bump_size 2.0
            scale  <0.1, 0.1, 0.1>
          }
     ]
  }
  rotate -30.0 * y
}

#declare Sand1 =
texture {
 pigment {colour rgb <0.500,0.336,0.062>}
}

#declare Sand2 =
texture {TdG_DarkSand
 //pigment {colour rgb <0.500,0.400,0.300>}
}

#declare Sand3 =
texture {
 pigment {colour rgb <0.758,0.600,0.492>}
}

#declare Sand4 =
texture {TdG_LightSand
 //pigment {colour rgb <0.641,0.530,0.338>}
}


#declare Landtex =
texture {
  #if (Slope)
    slope {<0.0, SlopeValue, 0.0> , 0.0 , 0.5
      altitude <0.0, 1-SlopeValue, 0.0> , -2 , 1
    }
  #else
    gradient y
  #end
  texture_map {
    [0.00 Sand1 ]
    [0.10 CrackleRipple ]
    [0.25 Sand2 ]
    [0.50 CrackleRipple ]
    [0.75 Sand3 ]
    [0.85 CrackleRipple ]
    [1.00 Sand4 ]
  }
  rotate 10*z
}

#declare DuneSand =
texture {TdG_LightSand
 normal {
   ripples , 0.1
   bump_size 0.1
   turbulence 0.6
   scale  <0.02, 0.08, 0.5>
   rotate -45*y
 }
}

//=========================================================================================================================================
#if (Iso)
// the rocks:
isosurface {
  function {y- P(x, y, z)}
  //open  
  max_gradient 10
  accuracy 0.001
  contained_by {box {<0.5-Xscale*0.5, -2, 0.5-Zscale*0.5>, <0.5+Xscale*0.5, 2, 0.5+Zscale*0.5>}}    
  translate <-0.5, 0, -0.5>
  scale <Loc, Loc/2, Loc>
  texture {Landtex}
  rotate 6*z              // the rocky landscape is slightly tilted so that it disappears below the dunes to the left 
  //translate -Loc/2*x
}

// the dunes:
isosurface {
  function {y-Q(x, y, z)}
  max_gradient 10
  accuracy 0.001
  contained_by {box {<0.5-Xscale*0.5, -2, 0.5-Zscale*0.5>, <0.5+Xscale*0.5, 2, 0.5+Zscale*0.5>}}    
  translate <-0.5, 0, -0.5>
  scale <Loc, Loc/4, Loc>
  texture {DuneSand}
  translate 0.0*y
}

#else
height_field{
  function 1000, 1000 {P((x-0.5)*Xscale+0.5, 0, (y-0.5)*Zscale+0.5)}  
  translate <-0.5, 0, -0.5>
  scale <Loc*Xscale, 1, Loc*Zscale>
  pigment {Landtex}
  //translate Loc/2*x
}

#end

//=========================================================================================================================================
#if (Saguaros) // put here your own objects!
#include "Saguaro.inc"
#include "Saguaro2.inc"
object {Saguaro
  scale 0.2
  rotate -10*y
  translate <0.35, 0.25, -3.5>
}

object {Saguaro2
  scale 0.030
  rotate 95*y
  translate <0.15, 0.64, -9.6>
}
#end

//=========================================================================================================================================
// Mountain in the distance:

// put here your own heigthfield!
height_field {
  tga "SingleMountain.tga"
  texture {Landtex}
  translate <-0.5, 0, 0.5>
  scale <100, 5, 100>
  rotate 180*y
  translate <-70, 0, 300>
}

//=========================================================================================================================================
#declare SkyScale = 210;
#declare Intervals = 10;
#declare Scat = 1; 										// 1=isotropic; 2=Mie haze; 3=Mie murky; 4=Rayleigh; 5=Henyey-Greenstein
#declare E = 0.6;  										// eccentricity (for Scat=5); + scatters towards light; - from light
#declare ExtraHaze = 25;  	        	// less haze < > more haze
#declare ScatCol = <0.05, 0.05, 0.02>;

#declare ScatteringAir = 
   material { 
      texture {      
         pigment {
            color rgb <0.0, 0.0, 0.0>
         }
      }   
      interior {
         media { // DustyAir
            intervals Intervals
            samples 5,100
            variance 1.0/1000.0
            ratio 0.99
            scattering {
               Scat ,  rgb ScatCol/(SkyScale/ExtraHaze)
               #if (Scat=5)
                 eccentricity E
               #end
            }
         }
      }
   }
   
#local UpperCloudTex = 
   material {   
      texture {      
         pigment {
            bozo
            color_map {
               [ 0.00  rgbft <0.80, 0.80, 0.80, 0.0, 1.0> ]
               [ 0.50  rgbft <0.90, 0.90, 0.90, 0.0, 1.0> ]
               [ 0.70  rgbft <0.95, 0.95, 0.98, 0.0, 0.9> ]
               [ 0.90  rgbft <0.95, 0.95, 0.98, 0.0, 0.5> ]
               [ 1.00  rgbft <1.00, 1.00, 0.98, 0.0, 1.0> ]
            }
            turbulence 0.9
            omega 0.7
            ramp_wave
         }      
         finish {
            ambient 1.0
            diffuse 0.3
         }      
         scale <0.30, 1.00, 0.70>
         rotate 70.0*y
         translate <-1000.0, 0.0, 2000.0>
      }
   }

#declare MediaSphere =
  sphere {
    <0,0,0>,1
    hollow 
    no_shadow
    material {ScatteringAir}
    scale SkyScale
  }
//object {MediaSphere} // experiment with a media instead of fog! Very slow!


//sky_sphere
sphere {
  <0,0,0>,1
  hollow
  no_shadow
  texture {
    pigment {
      gradient y
      color_map {
               [ 0.0  rgb <0.8958, 0.953325, 1.000000> ]
               [ 0.3  rgb <0.4479, 0.724052, 0.916667> ]
               [ 1.0  rgb <0.0000, 0.358694, 0.604133> ]
      }
    }
    finish {ambient 0.1}
  }
  scale <SkyScale*2, SkyScale, SkyScale*2>
}

#declare UpperCloudLayer =
sphere { 
  <0,0,0>,1
  hollow
  material {UpperCloudTex rotate 0*y}
  scale <SkyScale*1.9, SkyScale*0.6, SkyScale*1.9>
}
object {UpperCloudLayer}
  
//=========================================================================================================================================
camera {
  //location  <0, 1, -1.1*Loc> 
  //location  <0, 1, -Loc*(Zscale/2)> 
  location  <0, 0.9, -10>
  look_at   <0, 0, 0> 
  right     x*image_width/image_height
  angle     54
}

light_source {<-(SkyScale*0.7),SkyScale*0.7,-(SkyScale*0.7)> colour rgb <1, 1, 0.7>*2.5}
//light_source {<100,100,-100> colour rgb 1*0.5 shadowless}

fog {
  fog_type   2
  distance   75
  color      rgbt <0.996, 0.975, 0.649, 0.3> 
  fog_offset 0.1
  fog_alt    2.5
  turbulence 0.8
}

