#include "functions.inc"

#declare SCENERY=true;
#declare SEA=true;
#declare SPLASH=true;                  
#declare SKY=true;

#declare GEOMETRY=true;
#declare CABLES=true;
#declare SMALL_CABLES=true;

#declare FOCAL=false;
#declare RAD=false;

// *****************************************************************************
// INCLUDE FILES 
// *****************************************************************************
//
#include "shapes.inc"
#include "splash.pov"
#include "car.pov"

// *****************************************************************************
// GLOBAL VARIABLES 
// *****************************************************************************
//
// Everything in metres for this project

// BRIDGE
// Z
#declare MAIN_SPAN = 1280.0;
#declare SIDE_SPAN = 343.0;  // 343
// X
#declare TOTAL_WIDTH = 27.0;
#declare ROAD_WIDTH = 19.0;
#declare SIDEWALK_WIDTH = 3.0;
// Y
#declare SPAN_HEIGHT = 67.0;  // gap between water and bottom of bridge
#declare SPAN_THICKNESS = 8.0;  // from bottom of bridge to road surface


#macro SpanHeight( Z )

 #if(abs(Z)<0.1)
  #local H=0.1;
 #else
  #local H=Z;
 #end
 #local T = (SPAN_HEIGHT + 10*cos(H*pi/(MAIN_SPAN/2+SIDE_SPAN)) - (10/abs(H/300)));
 
 #if(T<-100)
  #local T=-100;
 #end
 
 T
 
#end

#macro RoadAngle(Z)
 
 #local H1 = SpanHeight( Z-0.1 );
 #local H2 = SpanHeight( Z+0.1 );
 
 degrees( atan( (H1-H2) / 0.2 ) )
 

#end


#macro SpanHeight2( Z )
 (SPAN_HEIGHT)
#end


// TOWERS
// Z
#declare LEG_Z = 16.0;
// X
#declare LEG_X = 10.0;
// Y
#declare TOWER_HEIGHT = 227.0;


// CABLES
#declare MAIN_CABLE_DIAMETER = 0.92;             
                  
                  
// GLOBAL SETTINGS

global_settings { 
    
    assumed_gamma 1.3
    
    max_trace_level 10
    
    #if(RAD=true)
    radiosity {            
        pretrace_start 0.08
        pretrace_end   0.01
        count 400
        error_bound 0.03
        recursion_limit 2
        brightness 1
    }
    #end 
        
           
             
}



// *****************************************************************************
// CAMERAS 
// *****************************************************************************
//
#declare Camera1=  // view from sea level looking out along the length of the bridge
camera{
 location <-80,20,-950>
 look_at <150,240,-0>
 angle 55    // 50 is best
 
 up y sky y right x*4/3
  
#if(FOCAL=true)  
    focal_point < 0, TOWER_HEIGHT/2, MAIN_SPAN/2>    // pink sphere in focus
    aperture 2     // a nice compromise
    variance 1/128
    confidence 0.995
    blur_samples 500      // more samples, higher quality image
#end 
} 

#declare Camera2= // close up of the top of one of the towers
camera{
 location <-500,TOWER_HEIGHT*1.3,-1000>
 look_at <0,TOWER_HEIGHT*0.9,-MAIN_SPAN/2>
 angle 10
 
 up y sky y right x*4/3
}

#declare Camera2b= // aerial view zoomed in on the top of one of the towers
camera{
 location <-0,TOWER_HEIGHT*2,-MAIN_SPAN/2-100>
 look_at <0,TOWER_HEIGHT*1,-MAIN_SPAN/2>
 angle 20
 
 up y sky y right x*4/3
}


#declare Camera3= // distant view from a plane looking at the whole bridge
camera{   
 location <-2000,1300,-4600>  
   
   up y
   right x*1280/1024
   angle 15
   sky <0,1,0>
   look_at <0,0,-0>
}  

#declare Camera4= // directly ahead aerial view (for road alignment)
camera {
   perspective
   
    location <-6000,4000,-6000>  
   
    location <0,10000,-1>
   
   up y
   right x*4/3
   angle 20
   sky <0,1,0>
   look_at <0,0,0>
   
}

#declare Camera5= // end view
camera{ 
 location <-1000,20,-2500>
 look_at <0,120,-1000>
 angle 15
 
 up y sky y right x*4/3
}

#declare Camera6= // side view
camera{ 
 location <-2500,20,0>
 look_at <0,120,0>
 angle 45
 
 up y sky y right x*4/3
}

#declare Camera7=  // IRTC view
camera{
 location <-80,40,-250>
 look_at <50,40,50>
 angle 50    // 50 is best
 
 up y sky y right x*4/3
}

#declare Camera8=  // to check cars
camera{
 location <-80,40,-250>
 look_at <50,40,200>
 angle 25    // 50 is best
 
 up y sky y right x*4/3
}

#declare Camera9=  // to check end cables
camera{
 location <-80,120,600>
 look_at <0,60,1000>
 angle 15    // 50 is best
 
 up y sky y right x*4/3
}


camera{ Camera7 }


// *****************************************************************************
// LIGHTS 
// *****************************************************************************
//
light_source{ <-100000,50000,-90000> color rgb 1.5 }


light_source {
  <0,0.1,0>
  color rgb <0.3,0.3,0.5>
  parallel
  point_at <0,100,0>
}


// *****************************************************************************
// MATERIALS 
// *****************************************************************************
//
#declare Fio=
  finish{
   diffuse 0.8
   ambient 0.0
   specular 0.2
   roughness 0.01
  }

#declare Cio =  rgb <208,82,68>/256;


#declare Mio= 
material{
 texture{
  pigment{ color Cio  }
  finish{ Fio }
 }
}

#declare Mtruss=
material{

texture{
 pigment{
 gradient y
 color_map{
  [0.0 color Cio]
  [0.2 color Cio]
  [0.2 color rgbt 1]
  [1.0 color rgbt 1] 
 }
 }
 finish{ Fio }
 rotate z*45
}

texture{
 pigment{
 gradient y
 color_map{
  [0.0 color Cio]
  [0.2 color Cio]
  [0.2 color rgbt 1]
  [1.0 color rgbt 1] 
 }
 }
 finish{ Fio }
 rotate z*-45
}

texture{
 pigment{
 gradient y
 color_map{
  [0.0 color Cio]
  [0.2 color Cio]
  [0.2 color rgbt 1]
  [0.8 color rgbt 1] 
  [0.8 color Cio]
  [1.0 color Cio]
  }
 }
 finish{ Fio }
}



}

#declare Mconcrete=
material{
 texture{
  pigment{ color rgb <200,180,150>/256 }
  finish{
   diffuse 0.8
   ambient 0.0
   specular 0.2
   roughness 0.01
  }
 }
}


#declare Msea=
material{
 texture{
 pigment{ color rgbt <0,0.3,0.2,0.2> }
 finish{
  diffuse 0.2
  ambient 0.0
  specular 0.8
  roughness 0.003
  reflection {0.1 0.5 falloff 3 exponent 0.4}
  
 }

 normal{ agate 0.2 scale 25 }

 }

  interior {
    ior 1.3
    fade_distance 4
    fade_power 1001
  }

}


#declare Msplash=
material{
 texture{
 pigment{ color rgbt <0,0.3,0.2,0.8> }
 finish{
  diffuse 0.1
  ambient 0.0
  specular 0.3
  roughness 0.003
  reflection {0.01 0.15}
  
 }

 }

  interior {
    ior 1.3
    fade_distance 60
    fade_power 1001
  }

}

#declare Mroad=
material{
 texture{
 
  pigment{ gradient x
           color_map{
            [0.0 color rgb 1.0]
            [0.1 color rgb 1.0]
            [0.1 color rgb 0.2]
            [1.0 color rgb 0.2]
           }
         }
         
  finish{ diffuse 1
          specular 0.1
          roughness 0.1
        }       
  
  scale (TOTAL_WIDTH-8)/5.8
 
 }

 texture{
 
  pigment{ gradient z
           color_map{
            [0.00 color rgb 0.2]
            [0.75 color rgb 0.2]
            [0.75 color rgbt  <0,0,0,1>]
            [1.00 color rgbt  <0,0,0,1>]
           }
         }
         
  finish{ diffuse 1
          specular 0.4
          roughness 0.1
        }       
  
  scale 3
 
 }




}


// *****************************************************************************
// SCENE SET UP 
// *****************************************************************************
//
// the sky
sky_sphere {
    pigment {
      gradient y
      color_map {
        [ 0.5  color rgb <0.84,0.91,1.00> ]
        [ 0.7  color rgb <0.19,0.52,0.88> ]
      }
      scale 2
      translate -1
    }
  }

// clouds
#local Msky=
material{
     texture{
      pigment { colour rgbf <0.98,0.98,0.98,1> }
   
     } 
      
     
      interior {
       media
       {
        scattering { 3 rgb 0.004*<0.2,0.2,0.2> }
        density{
         bozo
         color_map {
          [0.00 color 0]
          [0.60 color 0]
          [0.80 color 0.4]
          [1.00 color 0.0]
           }
           // warp{ turbulence 0 } 
           
           turbulence 1.5
           lambda 3
           omega 0.6
           
           scale 20000
           scale <1,0.05,1>

           }
       }
      } 
   }

#if(SKY=true)
difference{
 sphere{ <0,-6400000,0> 6400000 + 7000 }
 sphere{ <0,-6400000,0> 6400000 + 5000 }
 

            hollow on
           material{ Msky } 
           
           
         }
#end



// the sea

#if(SEA=true)
isosurface{
   function{ y + 60 * cos(sqrt(x*x+z*z)/10) / (sqrt(x*x+z*z)/10+6) }
   max_gradient 1.3
   contained_by{ box{ <-150,-10,-300> <10000,10,MAIN_SPAN*8> } }
   material{ Msea }
 }  
#else
 plane{ y 0 pigment{ color rgb <0.1,0.3,0.5> } }
#end

#if(SPLASH=true)
object{
 Splash( <0,0,0> , 20000 , 30 )  // 50000 particles
 material{ Msplash }
}               
#end


#if(SCENERY=true)
// the land
height_field{ tga "hf.tga" 
              water_level 0.0001
              smooth
              
              translate <0,0,0>                // -0.35
              scale <10.343*1635,300,10.343*1213>  // 200
              translate <-9800,0,-7200>          // 140
              scale <1,1,-1>
              
              texture{ 
               pigment { 
                bozo
                color_map{
                    [0.0 rgb <255,255,255> / 255 ]
                    [0.1 rgb <255,190,170> / 255 ]
                    [0.6 rgb <255,168,130> / 255 ]
                    [1.0 rgb <200,100,60> / 255 ]
                   }
                 scale 20
                }
                              
               finish{ diffuse 0.6 ambient 0.2 }
              }
          
          
          
              
            /*  texture{
               pigment{ image_map{ jpeg "bottomDetail25.jpg" interpolate 2 once}  
                        rotate -90*x 
                        scale <1424*2.4384 , 1 , -1269*2.4384>
                        translate <-110,0,-3300> 
                        
                        }
                      finish{ diffuse 1.0 ambient 0.3 }
              }  */
              
              texture{
               pigment{ image_map{ jpeg "topDetail25.jpg" interpolate 2 once}  
                        rotate -90*x 
                        scale <642*2.4384 , 1 , -560*2.4384>
                        translate <-805,0,600> 
                        }
                      finish{ diffuse 1.0 ambient 0.3 }
                        
              }
              
              
              
              
              rotate y*8.5
              translate <-205,-50,10>   // x was -180
}


fog {
  fog_type   2
  distance   3000
  color      rgb <0.84,0.91,1.00>*0.95 // gray
  fog_offset 15
  fog_alt    50
  turbulence 0.9

}



#end              


// *****************************************************************************
// GEOMETRY 
// *****************************************************************************
//
// the road bit


#if(GEOMETRY=true)   


#local diagLen = sqrt( pow(SPAN_THICKNESS-2,2) + 8*8 );
#local diagAng = atan( 8 / (SPAN_THICKNESS-2) );

#macro MsideTruss()

union{
  
  box{ <0 , 0 , 0 >                <1 , 1 , 16> }
  box{ <0 , SPAN_THICKNESS , 0 >   <1 , SPAN_THICKNESS-1 , 16> }
  box{ <-1 , SPAN_THICKNESS-0.01 , 0 >   <0 , SPAN_THICKNESS+1.5 , 16> }
  
  
  box{ <0,0,0> <1,SPAN_THICKNESS,0.5> }
  box{ <0,0,8> <1,SPAN_THICKNESS,8.5> }
  
  box{ <0,0,-0.25> <1,diagLen,0.25> rotate  x*degrees(diagAng) translate <0,1,0> }
  box{ <0,0,-0.25> <1,diagLen,0.25> rotate -x*degrees(diagAng) translate <0,1,0> }
  
  difference{
   cylinder{ <0,1,0> <1,1,0> 1 }
   plane{ y 0.5 }
  }
  
  difference{
   cylinder{ <0,SPAN_THICKNESS-1,8> <1,SPAN_THICKNESS-1,8> 1 }
   plane{ -y (0.5-SPAN_THICKNESS) }
  }
  
 }
 
#end


#macro MspanSection()

union{

        // THE ORANGE TRUSS
        union{
         object{ MsideTruss() translate <-TOTAL_WIDTH/2,0,0> } // the left and right side trusses
         object{ MsideTruss() translate <-TOTAL_WIDTH/2,0,0> scale <-1,1,1> }
         
         material { Mio }
         
        } 
        
        union{
             
         box{ <-TOTAL_WIDTH/2+1 , 1 , 0> < TOTAL_WIDTH/2-1 , 2 , 1> material{ Mtruss} } // the x-brace
         
         box{ <-TOTAL_WIDTH/6 , 0 , 0> <-TOTAL_WIDTH/6-1 , 1 , 16> material{ Mtruss rotate y*90} } // the z-braces
         box{ < TOTAL_WIDTH/6 , 0 , 0> < TOTAL_WIDTH/6+1 , 1 , 16> material{ Mtruss rotate y*90} } // the z-braces
         
         box{ < TOTAL_WIDTH/6 , 1 , 8> < TOTAL_WIDTH/6+1 , SPAN_THICKNESS , 9> material{ Mtruss rotate x*90} } // the y-braces
         box{ <-TOTAL_WIDTH/6 , 1 , 8> <-TOTAL_WIDTH/6-1 , SPAN_THICKNESS , 9> material{ Mtruss rotate x*90} } // the y-braces
        
         
        } 
         
         
        // THE ROAD
        
        
        //top surface
        box{ <-TOTAL_WIDTH/2+4 , SPAN_THICKNESS +0.5     , 0 >
             < TOTAL_WIDTH/2-4 , SPAN_THICKNESS + 1 , 16 >
             material{ Mroad }
           }
        //bottom surface
        box{ <-TOTAL_WIDTH/2+1 , SPAN_THICKNESS     , 0 >
             < TOTAL_WIDTH/2-1 , SPAN_THICKNESS + 0.5 , 16 >
             pigment{ color rgb 0.2 }
           }
        //sidewalk left
        box{ <-TOTAL_WIDTH/2+1 , SPAN_THICKNESS + 1.5   , 0 >
             <-TOTAL_WIDTH/2+4 , SPAN_THICKNESS + 1.7 , 16 >
             pigment{ color rgb 0.2 }
           }
        //sidewalk right
        box{ < TOTAL_WIDTH/2-1 , SPAN_THICKNESS + 1.5   , 0 >
             < TOTAL_WIDTH/2-4 , SPAN_THICKNESS + 1.7 , 16 >
             pigment{ color rgb 0.2 }
           }


}
#end
 
        
#local Z=-MAIN_SPAN/2-SIDE_SPAN;
#while( Z<MAIN_SPAN/2+SIDE_SPAN+100)
    
 object{ MspanSection()
         
         #local H1 = SpanHeight(Z);
         #local H2 = SpanHeight(Z+16);
         
         #local D = sqrt( 16*16 + (H1-H2)*(H1-H2) );
         #local A = atan( (H1-H2) / 16 );
         
         scale <1,1,D/16*1.01>
         rotate degrees(A)*x
         
         translate <0,H1,Z>
         
       }
    
#local Z=Z+16;
#end


// the lampposts
#local OlampPost=
union{

union{
 cylinder{ <0,0,0> <0,8,0> 0.15 }
 difference{
  torus{ 0.5 0.15 rotate x*90 translate <0.5,8,0> }
  plane{ y 8 }
  plane{ -x (-0.5) }
 }
 difference{
  torus{ 0.5 0.15 rotate x*90 translate <0.5,8,0> }
  plane{ y 8 }
  plane{ -x (-0.5) }
  translate y*-0.4
 }
 Round_Box_Merge( <0.3,8.4-0.2,-0.35  > , <1.4 , 8.4+0.24,0.35> , 0.05 )
 
 material{ Mio }
}

 box{ <0.8,8.4-0.2,-0.20  >  <1.3 , 8.4-0.4,0.25> 
      pigment{ color rgbt <1,1,0,0.5> }
    }


}


union{
#local Z=-MAIN_SPAN/2-SIDE_SPAN;
#while( Z<MAIN_SPAN/2+SIDE_SPAN+100)
    
 object{ OlampPost                rotate x*RoadAngle(Z) translate <-TOTAL_WIDTH/2+2 , SpanHeight(Z) + SPAN_THICKNESS +1 , Z> }
 object{ OlampPost scale <-1,1,1> rotate x*RoadAngle(Z) translate < TOTAL_WIDTH/2-2 , SpanHeight(Z) + SPAN_THICKNESS +1 , Z> }
     
#local Z=Z+30;
#end
}




// *****************************************************************************
// THE TOWERS 
// *****************************************************************************
//


#local TOWER_UNIT = (TOWER_HEIGHT-SpanHeight(MAIN_SPAN/2)-SPAN_THICKNESS) / (8+7+6+5);

#macro MtowerSection(bit)
union{
  
#if(bit=8) // for the first tower segment, make the legs extend down to the sea
 #local Ystart = -TOWER_HEIGHT;
#else
 #local Ystart = 0;
#end

 box{ <-bit/2  ,Ystart,-bit/2>   < 5,bit*TOWER_UNIT,bit/2>   translate x*-TOTAL_WIDTH/2}
 box{ <-bit/2+1,Ystart,-bit/2-2> < 4,bit*TOWER_UNIT,bit/2+2> translate x*-TOTAL_WIDTH/2}
 box{ <-bit/2-1,Ystart,-bit/2+1> < 6,bit*TOWER_UNIT,bit/2-1> translate x*-TOTAL_WIDTH/2}

 box{ < bit/2  ,Ystart,-bit/2>   <-5,bit*TOWER_UNIT,bit/2>   translate x* TOTAL_WIDTH/2}
 box{ < bit/2-1,Ystart,-bit/2-2> <-4,bit*TOWER_UNIT,bit/2+2> translate x* TOTAL_WIDTH/2}
 box{ < bit/2+1,Ystart,-bit/2+1> <-6,bit*TOWER_UNIT,bit/2-1> translate x* TOTAL_WIDTH/2}


difference{
 box{ <-bit/2+1-TOTAL_WIDTH/2,0,-bit/2+1> <bit/2-1+TOTAL_WIDTH/2,bit*TOWER_UNIT,bit/2-1>}

#local N=0;
#while(N<5)
 
 box{ <5-TOTAL_WIDTH/2*(1-N/20),-0.01,-bit/2> <-5+TOTAL_WIDTH/2*(1-N/20),bit*TOWER_UNIT*(0.9-bit/40)*(1-(5-N)/12),bit/2>}
 

#local N=N+1;
#end

}

box{ <-TOTAL_WIDTH/2 , bit*TOWER_UNIT - 0.6 , -bit/2+1 >
     < TOTAL_WIDTH/2 , bit*TOWER_UNIT       , -bit/2+1-0.6>
   }
box{ <-TOTAL_WIDTH/2 , bit*TOWER_UNIT - 0.6 ,  bit/2-1 >
     < TOTAL_WIDTH/2 , bit*TOWER_UNIT       ,  bit/2-1+0.6>
   }

box{ <-TOTAL_WIDTH/2 , bit*TOWER_UNIT*(0.9-bit/40)+0.6   , -bit/2+1 >
     < TOTAL_WIDTH/2 , bit*TOWER_UNIT*(0.9-bit/40)       , -bit/2+1-0.6>
   } 
box{ <-TOTAL_WIDTH/2 , bit*TOWER_UNIT*(0.9-bit/40)+0.6   ,  bit/2-1 >
     < TOTAL_WIDTH/2 , bit*TOWER_UNIT*(0.9-bit/40)       ,  bit/2-1+0.6>
   }


 
#local s=0.2;

#local N=-5;
#while( N<=5 )

 box{ <-s,bit*TOWER_UNIT*(0.9-bit/40)+1,-s>
      < s,bit*TOWER_UNIT-1,              s>
      rotate y*45
      translate <N*(TOTAL_WIDTH/2-6)/5 , 0 , -bit/2+1>
    } 
 box{ <-s,bit*TOWER_UNIT*(0.9-bit/40)+1,-s>
      < s,bit*TOWER_UNIT-1,              s>
      rotate y*45
      translate <N*(TOTAL_WIDTH/2-6)/5 , 0 ,  bit/2-1>
    } 



#local N=N+1;
#end
 
 
  

 material{ Mio }

}
#end
   

// the cross bit

#local CROSS_HEIGHT = (SpanHeight(MAIN_SPAN/2) + SPAN_THICKNESS) / 2.5;
#local ANG = atan( CROSS_HEIGHT / (TOTAL_WIDTH-8) );
#local LEN = sqrt( CROSS_HEIGHT*CROSS_HEIGHT + (TOTAL_WIDTH-8)*(TOTAL_WIDTH-8));


#declare Ocross=
union{
 
 box{ <-TOTAL_WIDTH/2+1, 0 , -1.5 >
      < TOTAL_WIDTH/2-1, 3 ,  1.5 > }

 box{ <0  , -1.5,-1.5>
      <LEN,  1.5, 1.5>
      rotate z*degrees(ANG)
      translate <-TOTAL_WIDTH/2+4,1.5,0>
    }      
 
 box{ <0  , -1.5,-1.5>
      <LEN,  1.5, 1.5>
      rotate z*(180-degrees(ANG))
      translate <TOTAL_WIDTH/2-4,1.5,0>
    }      


}



// the two main towers
#declare Otower=
union{

// the cross bits below the road
  object{ Ocross translate <0,(SpanHeight(MAIN_SPAN/2) + SPAN_THICKNESS)-CROSS_HEIGHT,0>  }
  object{ Ocross translate <0,(SpanHeight(MAIN_SPAN/2) + SPAN_THICKNESS)-CROSS_HEIGHT*2,0>}
  

// the chunky orange base
 #local bit=11;
union{
 box{ <-bit/2  ,0,-bit/2>   < 5,CROSS_HEIGHT*0.5,bit/2>   translate x*-TOTAL_WIDTH/2}
 box{ <-bit/2+1,0,-bit/2-2> < 4,CROSS_HEIGHT*0.5,bit/2+2> translate x*-TOTAL_WIDTH/2}
 box{ <-bit/2-1,0,-bit/2+1> < 6,CROSS_HEIGHT*0.5,bit/2-1> translate x*-TOTAL_WIDTH/2}

 box{ < bit/2  ,0,-bit/2>   <-5,CROSS_HEIGHT*0.5,bit/2>   translate x* TOTAL_WIDTH/2}
 box{ < bit/2-1,0,-bit/2-2> <-4,CROSS_HEIGHT*0.5,bit/2+2> translate x* TOTAL_WIDTH/2}
 box{ < bit/2+1,0,-bit/2+1> <-6,CROSS_HEIGHT*0.5,bit/2-1> translate x* TOTAL_WIDTH/2}
 material{ Mio }
}

// the conrete foundation

union{

 Round_Box_Merge( <-TOTAL_WIDTH/2 , -4 , -6 >  ,  < TOTAL_WIDTH/2 , CROSS_HEIGHT*0.3 ,  6 > ,1 ) // centre bit

 Round_Box_Merge( <-TOTAL_WIDTH/2-10 , -4 , -10 > , < -TOTAL_WIDTH/2+6 , CROSS_HEIGHT*0.3 ,  10 >,1 ) // left bit
 Round_Box_Merge( < TOTAL_WIDTH/2+10 , -4 , -10 > , <  TOTAL_WIDTH/2-6 , CROSS_HEIGHT*0.3 ,  10 >,1 ) // right bit


 material{ Mconcrete }
}


// the bit above the road

 
 object{ MtowerSection(8) translate y*(SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS+TOWER_UNIT*0) }
 object{ MtowerSection(7) translate y*(SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS+TOWER_UNIT*8) }
 object{ MtowerSection(6) translate y*(SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS+TOWER_UNIT*15) }
 object{ MtowerSection(5) translate y*(SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS+TOWER_UNIT*21) }
 

// the circular walkways around the towers
union{

 cylinder{ <-TOTAL_WIDTH/2 , SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS     , 0>
           <-TOTAL_WIDTH/2 , SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS+1.5 , 0>
           8 }
 cylinder{ < TOTAL_WIDTH/2 , SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS     , 0>
           < TOTAL_WIDTH/2 , SpanHeight(MAIN_SPAN/2)+SPAN_THICKNESS+1.5 , 0>
           8 }

 material{ Mio }
}



 // the top caps to route the cables


union{
 #local bit=3; 
 box{ <-3/2,TOWER_HEIGHT,-bit/2>     <5,TOWER_HEIGHT+1,bit/2> translate x*(-TOTAL_WIDTH/2-1)}
 box{ <-3/2+1,TOWER_HEIGHT,-bit/2-2> <4,TOWER_HEIGHT+1,bit/2+2> translate x*(-TOTAL_WIDTH/2-1)}

 box{ < 3/2,TOWER_HEIGHT,-bit/2>     <-5,TOWER_HEIGHT+1,bit/2> translate x*(TOTAL_WIDTH/2+1)}
 box{ < 3/2-1,TOWER_HEIGHT,-bit/2-2> <-4,TOWER_HEIGHT+1,bit/2+2> translate x*(TOTAL_WIDTH/2+1)}

 #local bit=2; 
 box{ <-2/2,TOWER_HEIGHT+1,-bit/2>     <5,TOWER_HEIGHT+2,bit/2> translate x*(-TOTAL_WIDTH/2-1)}
 box{ <-2/2+1,TOWER_HEIGHT+1,-bit/2-2> <4,TOWER_HEIGHT+2,bit/2+2> translate x*(-TOTAL_WIDTH/2-1)}

 box{ < 2/2,TOWER_HEIGHT+1,-bit/2>     <-5,TOWER_HEIGHT+2,bit/2> translate x*(TOTAL_WIDTH/2+1)}
 box{ < 2/2-1,TOWER_HEIGHT+1,-bit/2-2> <-4,TOWER_HEIGHT+2,bit/2+2> translate x*(TOTAL_WIDTH/2+1)}


 cylinder{ <-TOTAL_WIDTH/2-0 , TOWER_HEIGHT+2 , 0> <-TOTAL_WIDTH/2+2 , TOWER_HEIGHT+2 , 0> 2 }
 cylinder{ <-TOTAL_WIDTH/2+2 , TOWER_HEIGHT+2 , 0> <-TOTAL_WIDTH/2+2.5 , TOWER_HEIGHT+2 , 0> 3 }
 cylinder{ <-TOTAL_WIDTH/2-0 , TOWER_HEIGHT+2 , 0> <-TOTAL_WIDTH/2-0.5 , TOWER_HEIGHT+2 , 0> 3 }
 
 
 
 cylinder{ < TOTAL_WIDTH/2+0 , TOWER_HEIGHT+2 , 0> < TOTAL_WIDTH/2-2 , TOWER_HEIGHT+2 , 0> 2 }
 cylinder{ < TOTAL_WIDTH/2-2 , TOWER_HEIGHT+2 , 0> < TOTAL_WIDTH/2-2.5 , TOWER_HEIGHT+2 , 0> 3 }
 cylinder{ < TOTAL_WIDTH/2-0 , TOWER_HEIGHT+2 , 0> < TOTAL_WIDTH/2+0.5 , TOWER_HEIGHT+2 , 0> 3 }
        
            
 
 material{ Mio }  
}  

}


object{ Otower translate  z*MAIN_SPAN/2 }
object{ Otower translate -z*MAIN_SPAN/2 }















   
// *****************************************************************************
// THE CABLES 
// *****************************************************************************
//

// the cables
#local CABLE_DROP = (TOWER_HEIGHT - SPAN_HEIGHT - SPAN_THICKNESS - 1.5);

 
 
// the top clips
#declare OtopClip=
union
{
  cylinder{ <0,-MAIN_CABLE_DIAMETER*(5/2-0.2),0>
            <0,-MAIN_CABLE_DIAMETER*(5/2+0.2),0>
            MAIN_CABLE_DIAMETER*0.7
          }
            
  torus{ MAIN_CABLE_DIAMETER/2 + MAIN_CABLE_DIAMETER/4
         MAIN_CABLE_DIAMETER/6
         rotate x*90
         translate <0,0,0>
       }
  
  cylinder{ <-0.3,0,0> <-0.3,-MAIN_CABLE_DIAMETER*(5/2-0.2),0> MAIN_CABLE_DIAMETER/6 }
  cylinder{ < 0.3,0,0> < 0.3,-MAIN_CABLE_DIAMETER*(5/2-0.2),0> MAIN_CABLE_DIAMETER/6 }
  
  
  material{ Mio }


} 
  
 
 
 
// the end sections (almost straight)   
#declare OendCables=
union{


sphere_sweep {
        cubic_spline
        SIDE_SPAN / 10 + 17
 
#local SZ = MAIN_SPAN/2 - 10;
#while(SZ < MAIN_SPAN/2 + SIDE_SPAN + 150)


  #local SY =  0.2*(CABLE_DROP) / ((exp(1)+exp(-1))/2-1) * (cosh( (MAIN_SPAN/2+SIDE_SPAN-SZ+400) / (SIDE_SPAN) )-1) + TOWER_HEIGHT - CABLE_DROP - 34.5;
    
  <0,SY,SZ> MAIN_CABLE_DIAMETER/2   
  
#local SZ=SZ+10;
#end                                        

        material{ Mio }
}    

// the smaller down cables
#if(SMALL_CABLES=true)
#local SZ = MAIN_SPAN/2;
#while(SZ < MAIN_SPAN/2 + SIDE_SPAN + 100)

  #local SY =  0.2*(CABLE_DROP) / ((exp(1)+exp(-1))/2-1) * (cosh( (MAIN_SPAN/2+SIDE_SPAN-SZ+400) / (SIDE_SPAN) )-1) + TOWER_HEIGHT - CABLE_DROP - 34.5;
  
  cylinder{ <-0.4,SY-MAIN_CABLE_DIAMETER*(5/2+0.2),SZ> <-0.4,SpanHeight(SZ)+SPAN_THICKNESS,SZ> MAIN_CABLE_DIAMETER/6 material{ Mio } }
  cylinder{ <0.4,SY-MAIN_CABLE_DIAMETER*(5/2+0.2),SZ> <0.4,SpanHeight(SZ)+SPAN_THICKNESS,SZ> MAIN_CABLE_DIAMETER/6 material{ Mio } }
  
  object{ OtopClip translate <0,SY,SZ> }
  

#local SZ=SZ+25;
#end                                    
#end


}

// object for one side

#declare Ocable1 = 

union{

// the BIG central span cables ;-)
// y=acosh(z/a)        


 
sphere_sweep {
        cubic_spline
        MAIN_SPAN / 10 + 2
 
#local SZ = -MAIN_SPAN/2 - 10;
#while(SZ < MAIN_SPAN/2 + 10)


  #local SY = (CABLE_DROP) / ((exp(1)+exp(-1))/2-1) * (cosh( SZ / (MAIN_SPAN/2) )-1) + TOWER_HEIGHT - CABLE_DROP + 4.5;
    
  <0,SY,SZ> MAIN_CABLE_DIAMETER/2   
  
#local SZ=SZ+10;
#end                                        

        material{ Mio }
}    
                    
                    
// the little cables in the central span
#if(SMALL_CABLES=true)

#local S1 = seed(51783);

#local SZ = -MAIN_SPAN/2;
#while(SZ < MAIN_SPAN/2)

  #local SY = (CABLE_DROP) / ((exp(1)+exp(-1))/2-1) * (cosh( SZ / (MAIN_SPAN/2) )-1) + TOWER_HEIGHT - CABLE_DROP + 4.5;

  #local CABLE_BOTTOM = SpanHeight(SZ) + SPAN_THICKNESS;
  #if(CABLE_BOTTOM < SpanHeight2(SZ) + SPAN_THICKNESS - 4 )
   #local CABLE_BOTTOM = SpanHeight2(SZ) + SPAN_THICKNESS - 4; 
  #end
  
  cylinder{ <0,0,0> <0,-(SY-MAIN_CABLE_DIAMETER*(5/2+0.2)-CABLE_BOTTOM),0> MAIN_CABLE_DIAMETER/6
            
            #if( SpanHeight(SZ)  < SpanHeight2(SZ)  - 6)
             rotate z*rand(S1)*15
             rotate y*rand(S1)*360
            #end
            
            translate <-0.4,SY-MAIN_CABLE_DIAMETER*(5/2+0.2),SZ>
            material{ Mio }
          }
  cylinder{ <0,0,0> <0,-(SY-MAIN_CABLE_DIAMETER*(5/2+0.2)-CABLE_BOTTOM),0> MAIN_CABLE_DIAMETER/6
            #if( SpanHeight(SZ)  < SpanHeight2(SZ) - 6)
             rotate z*rand(S1)*15
             rotate y*rand(S1)*360
            #end
            translate < 0.4,SY-MAIN_CABLE_DIAMETER*(5/2+0.2),SZ>
            material{ Mio }
          }
  
  
   
  object{ OtopClip translate <0,SY,SZ> }
  

#local SZ=SZ+25;
#end                                    
#end

// the big cables at either end (not quite    

  object{ OendCables }
  object{ OendCables scale <1,1,-1> }

}             

#if(CABLES=true)
 object{ Ocable1 translate  x*(TOTAL_WIDTH/2-1) }
 object{ Ocable1 translate -x*(TOTAL_WIDTH/2-1) }
#end

// *****************************************************************************


// *****************************************************************************
// SOME CARS
// *****************************************************************************
//
#local Ccol = array[10];

#local Ccol[0] = 0.5*<1,1,1>;
#local Ccol[1] = 0.5*<1,1,1>;
#local Ccol[2] = 0.5*<1,1,1>;
#local Ccol[3] = 0.1*<1,1,1>;
#local Ccol[4] = 0.1*<1,1,1>;
#local Ccol[5] = <0.6,0,0>;
#local Ccol[6] = 0.8*<1,1,1>;
#local Ccol[7] = <0,0,0.6>;
#local Ccol[8] = <0,0.3,0>;
#local Ccol[9] = 0.5*<1,1,1>;



union{  // the cars in the queue
#local R1 = seed( 8392 );
#local lane=0;
#while(lane<3)
    
 #local CZ=80 + rand(R1)*10;
 #local carMax = rand(R1) * 10 + 5;
 #local car=0;
 #while( car < carMax )
                             
   #local CX = (TOTAL_WIDTH-8)/6 * (lane+0.5) - (TOTAL_WIDTH/2-4);
   #local CY = SpanHeight( CZ );
   
   object{ Car( Ccol[ int( rand(R1)*10) ] ) 
           rotate 180 * y
           translate <0,SPAN_THICKNESS,0>
           rotate RoadAngle(CZ)*x
           translate <CX,CY+1.0,CZ>
          }
     
     
 #local CZ=CZ+10 + rand(R1)*15;   
 #local car=car+1;
 #end    
        
    
#local lane=lane+1;
#end
}

union{  // the cars skidding on the road
#local n=0;
#while( n < 20 )

 #local CZ=75 - rand(R1)*60;
 #local CX = rand(R1)*(TOTAL_WIDTH-10) - ((TOTAL_WIDTH-10)/2);
 #local CY = SpanHeight( CZ );
 
 object{ Car( Ccol[ int( rand(R1)*10) ] ) 
           rotate rand(R1) * 360 * y
           translate <0,SPAN_THICKNESS,0>
           rotate RoadAngle(CZ)*x
           translate <CX,CY+1.0,CZ>
          } 


#local n=n+1;
#end
}

union{  // + a few randoms
#local n=0;
#while( n < 10 )

 #local CZ = rand(R1)*40 - 20;
 #local CX = rand(R1)*(TOTAL_WIDTH+5) - ((TOTAL_WIDTH+5)/2);
 #local CY = rand(R1)*15 - 5;
 
 object{   Car( Ccol[ int( rand(R1)*10) ] ) 
           rotate <rand(R1),rand(R1),rand(R1)> * 360
           translate <CX,CY,CZ>
          } 


#local n=n+1;
#end
}


// *****************************************************************************
// SOME RANDOM BITS OF METAL
// *****************************************************************************
//
#local R1 = seed(1225);

#local n=0;
#while(n<30)


box{ <0,0,0> <1+rand(R1),rand(R1)*20+6,1> 
     
     #if( rand(R1) > 0.5 )
      material{ Mio }
     #else
      material{ Mtruss rotate z*90 }
     #end
     
     
     rotate z*(rand(R1)*180-90)
     rotate y*rand(R1)*360
     translate <rand(R1)*TOTAL_WIDTH - TOTAL_WIDTH/2,rand(R1)*35-10,rand(R1)*200-100>
     
     } // the x-brace

#local n=n+1;
#end         





// *****************************************************************************
// THE UFO
// *****************************************************************************
//

#local N=1;
#while(N<8)


union{

 sphere{ <0,0,0> 1 scale <20,5,20> }
 sphere{ <0,4,0> 6 }
 
 texture{
  pigment{ color rgbt <0.2,0.2,0.2,0.8> }
  finish{ reflection 0.5/8
          specular 0.8/8
          roughness 0.01 }
 }
  
 translate <1600+N*3,60,1600>
 
}

#local N=N+1;
#end


#end