#include "functions.inc"
#include "math.inc"

#declare PYRAMID_BASE = 756; // feet
#declare PYRAMID_HEIGHT = 481; // feet
#declare PYRAMID_BLOCK_WIDTH = 5; // feet
#declare PYRAMID_BLOCK_WIDTH_VARIANCE = 2; // feet
#declare PYRAMID_BLOCK_DEPTH = 10; // feet
#declare PYRAMID_BLOCK_DEPTH_VARIANCE = 1; // feet
#declare PYRAMID_BLOCK_EROSION = 0.2; // feet
#declare PYRAMID_BLOCK_EROSION_VARIANCE = 0.1; // feet
#declare PYRAMID_BLOCK_ROUGHNESS = 8;
#declare PYRAMID_BLOCK_TOLERANCE = 0.2/12; // feet

#declare SUN_BRIGHTNESS = <1.0,0.9,0.8>;
#declare SUN_DIRECTION = vnormalize(<0.4,0.8,-1.0>);

#declare AMBIENT = <0.40,0.42,0.44>;
#declare DIFFUSE = 1;

#declare INFINITY = 65536;

#declare CAMERA_FOCUS = <-PYRAMID_BASE*0.02,PYRAMID_HEIGHT*0.3,0>;
#declare CAMERA_DIRECTION = <PYRAMID_BASE*0.5,-PYRAMID_HEIGHT*0.2,-PYRAMID_BASE*0.65>;

#declare PYRAMID_COURSE_HEIGHT = array[203] // layer heights in the pyramid follow this pattern
{
    5.17014563997212, 4.33198209765582, 4.25257713048902, 3.88202061704392, 3.5467552001174,
    3.3526541692652,  3.45852745882094, 3.33500862100591, 3.21148978319088, 3.17619868667229,
    2.97327488169045, 2.62036391650464, 2.62918669063429, 2.59389559411571, 2.58507281998606,
    2.54978172346748, 2.47037675630067, 2.74388275431968, 3.36147694339484, 2.07335192046664,
    2.09982024285557, 3.04385707472762, 2.91151546278294, 2.85857881800507, 2.84975604387542,
    2.71741443193074, 2.62036391650464, 2.57625004585642, 2.47919953043032, 2.48802230455996,
    2.45273120804138, 2.3203895960967,  2.3203895960967,  2.31156682196706, 4.39374151656334,
    3.6261601672842,  3.37029971752449, 3.20266700906123, 2.99092042994974, 2.81446494735684,
    2.72623720606039, 2.47037675630067, 2.93798378517187, 3.6261601672842,  3.34383139513556,
    2.40861733739316, 3.14090759015371, 3.13208481602407, 2.75270552844932, 2.47037675630067,
    2.34685791848564, 2.3203895960967,  2.25863017718919, 2.26745295131883, 2.23216185480025,
    2.18804798415202, 1.98512417917018, 2.52331340107854, 2.53213617520819, 2.47919953043032,
    2.31156682196706, 2.15275688763344, 2.29392127370777, 2.25863017718919, 2.31156682196706,
    2.09099746872593, 3.08797094537584, 2.69976888367145, 2.85857881800507, 2.45273120804138,
    2.54095894933784, 2.27627572544848, 2.23216185480025, 2.68212333541216, 2.55860449759713,
    2.08217469459628, 2.24980740305954, 2.09099746872593, 2.03806082394806, 2.12628856524451,
    2.0468835980777,  2.05570637220735, 1.99394695329983, 2.45273120804138, 2.02041527568876,
    2.3203895960967,  1.93218753439231, 2.03806082394806, 2.00276972742947, 3.39676803991343,
    3.02621152646833, 2.92033823691258, 2.59389559411571, 2.30274404783741, 2.25863017718919,
    2.02923804981841, 2.29392127370777, 3.30854029861697, 3.37912249165414, 3.14090759015371,
    2.95562933343116, 2.4174401115228,  2.58507281998606, 2.3203895960967,  2.32921237022635,
    2.20569353241132, 2.18804798415202, 2.58507281998606, 2.34685791848564, 2.0468835980777,
    2.02041527568876, 2.12628856524451, 2.0468835980777,  2.02923804981841, 1.9498330826516,
    2.39097178913387, 1.97630140504054, 3.14090759015371, 2.80564217322719, 2.62918669063429,
    2.56742727172677, 2.29392127370777, 2.32921237022635, 2.1439341135038,  2.18804798415202,
    2.02923804981841, 1.99394695329983, 2.01159250155912, 1.9498330826516,  2.34685791848564,
    2.19687075828167, 2.0468835980777,  1.89689643787373, 2.01159250155912, 1.8704281154848,
    2.09099746872593, 1.96747863091089, 2.26745295131883, 2.24980740305954, 1.91454198613302,
    1.85278256722551, 2.03806082394806, 1.84395979309586, 2.74388275431968, 2.13511133937415,
    2.10864301698522, 1.9498330826516,  1.93218753439231, 1.91454198613302, 2.34685791848564,
    2.15275688763344, 2.02923804981841, 1.95865585678125, 1.92336476026267, 1.80866869657728,
    1.87925088961444, 1.8704281154848,  1.90571921200338, 1.81749147070692, 1.89689643787373,
    1.82631424483657, 2.09982024285557, 1.9498330826516,  2.27627572544848, 1.88807366374409,
    1.8704281154848,  1.81749147070692, 1.76455482592905, 1.80866869657728, 1.83513701896621,
    1.73808650354012, 1.8704281154848,  1.83513701896621, 1.73808650354012, 1.79102314831799,
    1.80866869657728, 1.79102314831799, 1.7733776000587,  1.81749147070692, 2.33803514435599,
    2.17922521002238, 2.05570637220735, 1.98512417917018, 1.95865585678125, 1.82631424483657,
    1.86160534135515, 1.73808650354012, 1.84395979309586, 1.80866869657728, 1.79984592244763,
    1.8704281154848,  1.7733776000587,  1.81749147070692, 1.82631424483657, 1.74690927766976,
    2.10864301698522, 2.07335192046664, 1.9498330826516,  1.90571921200338, 2.01159250155912,
    1.95865585678125, 1.98512417917018, 1.88807366374409
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Sky                                                                                                               //
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#declare SKY_TURBULENCE_SCALE = 100;

#declare SKY_HORIZON = <120,119,160>/255;
#declare SKY_LIGHT = <100,118,200>/255;
#declare SKY_DARK = <12,33,98>/255;

intersection {
    box { <2,2,2>*INFINITY,<-2,0,-2>*INFINITY }
    sphere { <0,0,0>,INFINITY inverse }
    texture {
        pigment {
            gradient y
            color_map {
                [0 rgb SKY_LIGHT]
                [1 rgb SKY_DARK]
            }
            scale SKY_TURBULENCE_SCALE
            warp { turbulence <1,1,1> octaves 3 lambda 2 omega 0.5 }
            scale INFINITY/SKY_TURBULENCE_SCALE
        }
        finish {
            ambient 0
            diffuse DIFFUSE
        }
    }
    translate <0,-INFINITY/SKY_TURBULENCE_SCALE,0>
    no_shadow
}

#if(CLOUDS)
    #declare CLOUD_ALTITUDE = 2000;
    #declare CLOUD_HEIGHT = 250;
    #declare CLOUD_RADIUS = INFINITY/8;
    cylinder {
        <0,CLOUD_ALTITUDE,0>,<0,CLOUD_ALTITUDE+CLOUD_HEIGHT,0>,CLOUD_RADIUS
        hollow
        texture { pigment { rgbt 1 } }
        interior {
            media {
                method 3
                samples 4,32
                intervals 1
                scattering { 1, 1 }
                density { // smooth cloud base
                    gradient y
                    colour_map {
                        [0 rgb 0]
                        [0.5 rgb 0.001]
                        [1 rgb 0]
                    }
                    scale CLOUD_HEIGHT
                    translate <0,CLOUD_ALTITUDE,0>
                }
                density { // disappear into distance
                    cylindrical
                    colour_map {
                        [0 rgb 0]
                        [0.1 rgb 1]
                    }
                    scale CLOUD_RADIUS
                }
                density { // clouds
                    granite
                    colour_map {
                        [0.53 rgb 0]
                        [0.7 rgb 1]
                    }
                    scale INFINITY/12
                }
                density { // cloud detail
                    granite
                    colour_map {
                        [0 rgb 0.5]
                        [0.4 rgb 1]
                    }
                    scale INFINITY/64
                }
            }
        }
    }
#end

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Haze                                                                                                              //
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#if(HAZE)
    #declare HAZE_HEIGHT = 1536+256;
    cylinder {
        <0,0,0>,<0,HAZE_HEIGHT,0>,INFINITY
        hollow
        texture { pigment { rgbt 1 } }
        interior {
            media {
                method 3
                samples 4,32
                intervals 1
                absorption 1
                //scattering { 1, 1 }
                density {
                    gradient y
                    colour_map {
                        [0.8 rgb 0.0000075*SKY_HORIZON]
                        [1 rgb 0]
                    }
                    scale HAZE_HEIGHT
                }
            }
        }
    }
#end


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Ground                                                                                                            //
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

isosurface {
    function { y-18*f_noise3d(x/400,0,z/400)+f_granite(x/100,0,z/100) }
    contained_by { box { <-INFINITY,-1,-INFINITY>, <INFINITY,21,INFINITY> } }
    max_gradient 1.6
    accuracy 0.1
    texture {
        pigment {
            color rgb <1,1,0.9>
        }
        finish {
            ambient AMBIENT
            diffuse DIFFUSE
        }
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Pyramid                                                                                                           //
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#declare f_pyramid = function(x,y,z) { y - PYRAMID_HEIGHT + max(2*PYRAMID_HEIGHT*abs(x)/PYRAMID_BASE, 2*PYRAMID_HEIGHT*abs(z)/PYRAMID_BASE) }

#macro LimestoneBlock( block_position, block_size, randomise, slope_intersect )
    #if( DEBUG = 1 )
        box {
            block_position-block_size/2+PYRAMID_BLOCK_TOLERANCE,
            block_position+block_size/2-PYRAMID_BLOCK_TOLERANCE
        }
    #else
        #local erosion = (PYRAMID_BLOCK_EROSION-PYRAMID_BLOCK_EROSION_VARIANCE) + randomise * PYRAMID_BLOCK_EROSION_VARIANCE * ( rand(stoneSeed) + 1 );
        #local size_x = block_size.x/2 - PYRAMID_BLOCK_TOLERANCE - erosion;
        #local size_y = block_size.y/2 - PYRAMID_BLOCK_TOLERANCE - erosion;
        #local size_z = block_size.z/2 - PYRAMID_BLOCK_TOLERANCE - erosion;
        #local center_x = block_position.x * (2 * rand(stoneSeed) - 1);
        #local center_y = block_position.y * (2 * rand(stoneSeed) - 1);
        #local center_z = block_position.z * (2 * rand(stoneSeed) - 1);
        #if( slope_intersect )
            #local position_x = block_position.x;
            #local position_y = block_position.y;
            #local position_z = block_position.z;
            isosurface {
                function {
                    max(f_rounded_box(x,y,z,erosion,size_x,size_y,size_z),f_pyramid(x+position_x,y+position_y,z+position_z))
                    +erosion*(f_granite((x-center_x)/PYRAMID_BLOCK_ROUGHNESS,(y-center_y)/PYRAMID_BLOCK_ROUGHNESS,(z-center_z)/PYRAMID_BLOCK_ROUGHNESS)-1)
                }
                contained_by { box { -block_size/2, block_size/2 } }
                max_gradient 1.7
                accuracy 0.1
                rotate <0,randomise * (2 * rand(stoneSeed) - 1)*atan2d(PYRAMID_BLOCK_TOLERANCE,block_size.z),0> // not 100% accurate, but good approximation
                translate block_position
            }
        #else
            isosurface {
                function {
                    f_rounded_box(x,y,z,erosion,size_x,size_y,size_z)
                    +erosion*(f_granite((x-center_x)/PYRAMID_BLOCK_ROUGHNESS,(y-center_y)/PYRAMID_BLOCK_ROUGHNESS,(z-center_z)/PYRAMID_BLOCK_ROUGHNESS)-1)
                }
                contained_by { box { -block_size/2, block_size/2 } }
                max_gradient 1.7
                accuracy 0.1
                rotate <0,randomise * (2 * rand(stoneSeed) - 1)*atan2d(PYRAMID_BLOCK_TOLERANCE,block_size.z),0> // not 100% accurate, but good approximation
                translate block_position
            }
        #end
    #end
#end

#if(PYRAMID)
    #declare f_layer_1 = function(x,y,z) { y-0.5*PYRAMID_HEIGHT - 24*f_noise3d(16*x/PYRAMID_BASE,0,16*z/PYRAMID_BASE) }
    #declare f_layer_2 = function(x,y,z) { y-0.75*PYRAMID_HEIGHT - 72*f_noise3d(8*x/PYRAMID_BASE,0,8*z/PYRAMID_BASE) }
    
    #declare stoneSeed = seed(736428);
    union {
        #declare max_layer_1 = 0;
        #declare min_layer_2 = 202;
        #declare max_layer_2 = 0;
    
        #declare row_inset = 2 * PYRAMID_HEIGHT / PYRAMID_BASE;
    
        // pyramid bottom
        #declare row_index = 0;
        #declare row_height = 0;
        #while( row_index < 100 )
            #declare vert_position = row_height + PYRAMID_COURSE_HEIGHT[ row_index ] * 0.5;
            
            #declare row_length = PYRAMID_BASE - 2 * row_height / row_inset - PYRAMID_COURSE_HEIGHT[ row_index ] * 2
                + 16 * (f_noise3d(0,vert_position/20,0));
            #declare depth_position = row_length / 2;
            
            #declare block_count = ceil(row_length / PYRAMID_BLOCK_WIDTH);
            #declare block_width = row_length / block_count;
            #declare block_prev_var_1 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_2 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_3 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_4 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_init_var_1 = block_prev_var_1;
            #declare block_init_var_2 = block_prev_var_2;
            #declare block_init_var_3 = block_prev_var_3;
            #declare block_init_var_4 = block_prev_var_4;
            
            #declare index = 1;
            #while( index < block_count )
            
                #declare horiz_position = (index+0.5) * block_width - row_length/2;
            
                
                
                #declare block_depth_var = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1);
                #if( index < block_count - 1 )
                    #declare block_width_var_1 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_2 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_3 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_4 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare this_block_depth_1 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_2 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_3 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_4 = PYRAMID_BLOCK_DEPTH;
                #else // last block
                    #declare block_width_var_1 = 0;
                    #declare block_width_var_2 = 0;
                    #declare block_width_var_3 = 0;
                    #declare block_width_var_4 = 0;
                    #declare this_block_depth_1 = block_width + block_init_var_4;
                    #declare this_block_depth_2 = block_width + block_init_var_1;
                    #declare this_block_depth_3 = block_width + block_init_var_2;
                    #declare this_block_depth_4 = block_width + block_init_var_3;
                #end
                #declare block_depth_var_1 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1);
                #declare block_depth_var_2 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1);
                #declare block_depth_var_3 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1);
                #declare block_depth_var_4 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1);
                #declare this_block_width_1 = block_width + block_width_var_1 - block_prev_var_1;
                #declare this_block_width_2 = block_width + block_width_var_2 - block_prev_var_2;
                #declare this_block_width_3 = block_width + block_width_var_3 - block_prev_var_3;
                #declare this_block_width_4 = block_width + block_width_var_4 - block_prev_var_4;
                
                // +ve z
                #if( f_layer_1(-horiz_position,vert_position,depth_position)<0 )
                    #declare block_position = <-(horiz_position + block_width_var_1 / 2 + block_prev_var_1 / 2),
                                               vert_position,
                                               depth_position-this_block_depth_1/2+block_depth_var_1/2>;
                    LimestoneBlock( block_position, <this_block_width_1,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_depth_1+block_depth_var_1>, 1, false )
                #end
                // +ve x
                #if( f_layer_1(depth_position,vert_position,horiz_position)<0 )
                    #declare block_position = <depth_position-this_block_depth_2/2+block_depth_var_2/2,
                                               vert_position,
                                               horiz_position + block_width_var_2 / 2 + block_prev_var_2 / 2>;
                    LimestoneBlock( block_position, <this_block_depth_2,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_width_2+block_depth_var_2>, 1, false )
                #end
                // -ve z
                #if( f_layer_1(horiz_position,vert_position,-depth_position)<0 )
                    #declare block_position = <horiz_position + block_width_var_3 / 2 + block_prev_var_3 / 2,
                                               vert_position,
                                               this_block_depth_3/2-depth_position-block_depth_var_3/2>;
                    LimestoneBlock( block_position, <this_block_width_3,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_depth_3+block_depth_var_3>, 1, false )
                #end
                // -ve x
                #if( f_layer_1(depth_position,vert_position,-horiz_position)<0 )
                    #declare block_position = <this_block_depth_4/2-depth_position-block_depth_var_4/2,
                                               vert_position,
                                               -(horiz_position + block_width_var_4 / 2 + block_prev_var_4 / 2)>;
                    LimestoneBlock( block_position, <this_block_depth_4,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_width_4+block_depth_var_4>, 1, false )
                    #declare max_layer_1 = max(row_index,max_layer_1);
                #end
                
                #declare block_prev_var_1 = block_width_var_1;
                #declare block_prev_var_2 = block_width_var_2;
                #declare block_prev_var_3 = block_width_var_3;
                #declare block_prev_var_4 = block_width_var_4;
                #declare index = index + 1;
            #end
            #declare row_height = row_height + PYRAMID_COURSE_HEIGHT[ row_index ];
            #declare row_index = row_index + 1;
        #end
        
        
        // pyramid middle
        #declare row_index = 0;
        #declare row_height = 0;
        #while( row_index < 170 )
            #declare vert_position = row_height + PYRAMID_COURSE_HEIGHT[ row_index ] * 0.5;
            
            #declare row_length = PYRAMID_BASE - 2 * row_height / row_inset - PYRAMID_COURSE_HEIGHT[ row_index ] * 2 - PYRAMID_BLOCK_DEPTH / 2;
            #declare depth_position = row_length / 2;
            
            #declare block_count = ceil(row_length / PYRAMID_BLOCK_WIDTH);
            #declare block_width = row_length / block_count;
            #declare block_prev_var_1 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_2 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_3 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_4 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_init_var_1 = block_prev_var_1;
            #declare block_init_var_2 = block_prev_var_2;
            #declare block_init_var_3 = block_prev_var_3;
            #declare block_init_var_4 = block_prev_var_4;
            
            #declare index = 1;
            #while( index < block_count )
            
                #declare horiz_position = (index+0.5) * block_width - row_length/2;
            
                
                
                #declare block_depth_var = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1);
                #if( index < block_count - 1 )
                    #declare block_width_var_1 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_2 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_3 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_4 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare this_block_depth_1 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_2 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_3 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_4 = PYRAMID_BLOCK_DEPTH;
                #else // last block
                    #declare block_width_var_1 = 0;
                    #declare block_width_var_2 = 0;
                    #declare block_width_var_3 = 0;
                    #declare block_width_var_4 = 0;
                    #declare this_block_depth_1 = block_width + block_init_var_4;
                    #declare this_block_depth_2 = block_width + block_init_var_1;
                    #declare this_block_depth_3 = block_width + block_init_var_2;
                    #declare this_block_depth_4 = block_width + block_init_var_3;
                #end
                #declare block_depth_var_1 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) * 0.7;
                #declare block_depth_var_2 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) * 0.7;
                #declare block_depth_var_3 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) * 0.7;
                #declare block_depth_var_4 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) * 0.7;
                #declare this_block_width_1 = block_width + block_width_var_1 - block_prev_var_1;
                #declare this_block_width_2 = block_width + block_width_var_2 - block_prev_var_2;
                #declare this_block_width_3 = block_width + block_width_var_3 - block_prev_var_3;
                #declare this_block_width_4 = block_width + block_width_var_4 - block_prev_var_4;
                
                
                
                #if( row_index >= 93 )
                    #declare block_position = <-(horiz_position + block_width_var_1 / 2 + block_prev_var_1 / 2),
                                               vert_position,
                                               depth_position-this_block_depth_1/2+block_depth_var_1/2>;
                    LimestoneBlock( block_position, <this_block_width_1,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_depth_1+block_depth_var_1>, 0.7, false )
                #end
                // +ve x
                #if( row_index >= 93 )
                    #declare block_position = <depth_position-this_block_depth_2/2+block_depth_var_2/2,
                                               vert_position,
                                               horiz_position + block_width_var_2 / 2 + block_prev_var_2 / 2>;
                    LimestoneBlock( block_position, <this_block_depth_2,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_width_2+block_depth_var_2>, 0.7, false )
                #end
                // -ve z
                #if( row_index >= 93 )
                    #declare block_position = <horiz_position + block_width_var_3 / 2 + block_prev_var_3 / 2,
                                               vert_position,
                                               this_block_depth_3/2-depth_position-block_depth_var_3/2>;
                    LimestoneBlock( block_position, <this_block_width_3,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_depth_3+block_depth_var_3>, 0.7, false )
                #end
                // -ve x
                #if( row_index >= 93 )
                    #declare block_position = <this_block_depth_4/2-depth_position-block_depth_var_4/2,
                                               vert_position,
                                               -(horiz_position + block_width_var_4 / 2 + block_prev_var_4 / 2)>;
                    LimestoneBlock( block_position, <this_block_depth_4,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_width_4+block_depth_var_4>, 0.7, false )
                    //#declare min_layer_2 = min(row_index,min_layer_2);
                    #declare max_layer_2 = max(row_index,max_layer_2);
                #end
                
                #declare block_prev_var_1 = block_width_var_1;
                #declare block_prev_var_2 = block_width_var_2;
                #declare block_prev_var_3 = block_width_var_3;
                #declare block_prev_var_4 = block_width_var_4;
                #declare index = index + 1;
            #end
            #declare row_height = row_height + PYRAMID_COURSE_HEIGHT[ row_index ];
            #declare row_index = row_index + 1;
        #end
        
        #declare P_granite_1 = 
            pigment {
                granite
                color_map {
                    [0.0 rgb <228,216,156>/255]
                    [0.25 rgb <255,243,185>/255]
                    [0.5 rgb <213,206,149>/255]
                    [0.75 rgb <223,210,150>/255]
                    [1.0 rgb <252,242,171>/255]
                }
                scale 0.1
            }
        #declare P_granite_2 = 
            pigment {
                granite
                color_map {
                    [0.0 rgb <242,221,172>/255]
                    [0.25 rgb <248,229,177>/255]
                    [0.5 rgb <231,212,169>/255]
                    [0.75 rgb <222,204,162>/255]
                    [1.0 rgb <228,205,155>/255]
                }
                scale 0.1
            }
        texture {
            pigment {
                granite
                pigment_map {
                    [0.25 P_granite_1]
                    [0.75 P_granite_2]
                }
            }
            finish {
                diffuse DIFFUSE
                ambient AMBIENT
            }
        }
    }
    
    union {
        // pyramid top
        #declare row_index = 0;
        #declare row_height = 0;
        #while( row_index < 202 )
            #declare vert_position = row_height + PYRAMID_COURSE_HEIGHT[ row_index ] * 0.5;
            
            #declare row_length = PYRAMID_BASE - 2 * row_height / row_inset - PYRAMID_COURSE_HEIGHT[ row_index ] * 2 * clip((159-row_index)/10,0,1)/*+PYRAMID_COURSE_HEIGHT[ row_index ] * 2*/;
            #declare depth_position = row_length / 2;
            
            #declare block_count = ceil(row_length / PYRAMID_BLOCK_WIDTH);
            #declare block_width = row_length / block_count;
            #declare block_prev_var_1 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_2 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_3 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_prev_var_4 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
            #declare block_init_var_1 = block_prev_var_1;
            #declare block_init_var_2 = block_prev_var_2;
            #declare block_init_var_3 = block_prev_var_3;
            #declare block_init_var_4 = block_prev_var_4;
            
            #declare index = 1;
            #while( index < block_count )
            
                #declare horiz_position = (index+0.5) * block_width - row_length/2;
            
                
                
                #declare block_depth_var = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1);
                #if( index < block_count - 1 )
                    #declare block_width_var_1 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_2 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_3 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare block_width_var_4 = PYRAMID_BLOCK_WIDTH_VARIANCE * (2 * rand(stoneSeed)-1);
                    #declare this_block_depth_1 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_2 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_3 = PYRAMID_BLOCK_DEPTH;
                    #declare this_block_depth_4 = PYRAMID_BLOCK_DEPTH;
                #else // last block
                    #declare block_width_var_1 = 0;
                    #declare block_width_var_2 = 0;
                    #declare block_width_var_3 = 0;
                    #declare block_width_var_4 = 0;
                    #declare this_block_depth_1 = block_width + block_init_var_4;
                    #declare this_block_depth_2 = block_width + block_init_var_1;
                    #declare this_block_depth_3 = block_width + block_init_var_2;
                    #declare this_block_depth_4 = block_width + block_init_var_3;
                #end
                #declare block_depth_var_1 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) / 2;
                #declare block_depth_var_2 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) / 2;
                #declare block_depth_var_3 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) / 2;
                #declare block_depth_var_4 = PYRAMID_BLOCK_DEPTH_VARIANCE * (2 * rand(stoneSeed)-1) / 2;
                #declare this_block_width_1 = block_width + block_width_var_1 - block_prev_var_1;
                #declare this_block_width_2 = block_width + block_width_var_2 - block_prev_var_2;
                #declare this_block_width_3 = block_width + block_width_var_3 - block_prev_var_3;
                #declare this_block_width_4 = block_width + block_width_var_4 - block_prev_var_4;
                
                
                
                #if( f_layer_2(-horiz_position,vert_position,depth_position)>=0 )
                    #declare block_position = <-(horiz_position + block_width_var_1 / 2 + block_prev_var_1 / 2),
                                               vert_position,
                                               depth_position-this_block_depth_1/2+block_depth_var_1/2>;
                    LimestoneBlock( block_position, <this_block_width_1,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_depth_1+block_depth_var_1>, 0.01+0.8*clip((169-row_index)/20,0,1), true )
                #end
                // +ve x
                #if( f_layer_2(depth_position,vert_position,horiz_position)>=0 )
                    #declare block_position = <depth_position-this_block_depth_2/2+block_depth_var_2/2,
                                               vert_position,
                                               horiz_position + block_width_var_2 / 2 + block_prev_var_2 / 2>;
                    LimestoneBlock( block_position, <this_block_depth_2,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_width_2+block_depth_var_2>, 0.01+0.8*clip((169-row_index)/20,0,1), true )
                #end
                // -ve z
                #if( f_layer_2(horiz_position,vert_position,-depth_position)>=0 )
                    #declare block_position = <horiz_position + block_width_var_3 / 2 + block_prev_var_3 / 2,
                                               vert_position,
                                               this_block_depth_3/2-depth_position-block_depth_var_3/2>;
                    LimestoneBlock( block_position, <this_block_width_3,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_depth_3+block_depth_var_3>, 0.01+0.8*clip((169-row_index)/20,0,1), true )
                #end
                // -ve x
                #if( f_layer_2(depth_position,vert_position,-horiz_position)>=0 )
                    #declare block_position = <this_block_depth_4/2-depth_position-block_depth_var_4/2,
                                               vert_position,
                                               -(horiz_position + block_width_var_4 / 2 + block_prev_var_4 / 2)>;
                    LimestoneBlock( block_position, <this_block_depth_4,PYRAMID_COURSE_HEIGHT[ row_index ],this_block_width_4+block_depth_var_4>, 0.01+0.8*clip((169-row_index)/20,0,1), true )
                    #declare min_layer_2 = min(row_index,min_layer_2);
                    //#declare max_layer_2 = max(row_index,max_layer_2);
                #end
                
                #declare block_prev_var_1 = block_width_var_1;
                #declare block_prev_var_2 = block_width_var_2;
                #declare block_prev_var_3 = block_width_var_3;
                #declare block_prev_var_4 = block_width_var_4;
                #declare index = index + 1;
            #end
            #declare row_height = row_height + PYRAMID_COURSE_HEIGHT[ row_index ];
            #declare row_index = row_index + 1;
        #end
        texture {
            pigment {
                granite
                color_map {
                    [0.0 rgb <248,242,189>/255]
                    [0.25 rgb <227,221,173>/255]
                    [0.5 rgb <240,236,183>/255]
                    [0.75 rgb <252,249,205>/255]
                    [1.0 rgb <253,247,194>/255]
                }
                scale 0.1
            }
            finish {
                diffuse DIFFUSE
                ambient AMBIENT
            }
        }
    }
    //#debug concat("max_layer_1 = ",str(max_layer_1,1,0),"\n")
    #debug concat("min_layer_2 = ",str(min_layer_2,1,0),"\n")
    #debug concat("max_layer_2 = ",str(max_layer_2,1,0),"\n")
#end
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Scene                                                                                                             //
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#declare v_up = -vnormalize(<CAMERA_DIRECTION.x,(-CAMERA_DIRECTION.x*CAMERA_DIRECTION.x-CAMERA_DIRECTION.z*CAMERA_DIRECTION.z)/CAMERA_DIRECTION.y,CAMERA_DIRECTION.z>);
#declare v_right = vnormalize(<-CAMERA_DIRECTION.z,0,CAMERA_DIRECTION.x>)*16/9;
#if( v_up.y < 0 )
    #declare v_up = -v_up;
    #declare v_right = -v_right;
#end
camera {
    perspective
    location CAMERA_DIRECTION+CAMERA_FOCUS
    up v_up
    right v_right
    direction -CAMERA_DIRECTION
    angle 90
}

light_source {
    INFINITY*0.9*SUN_DIRECTION
    rgb SUN_BRIGHTNESS
}