
// Glenn's POV-Ray Scene File - Radiator Maker
// created 12-Apr-06
// last update 13-Apr-06
// Copyright 2006 Glenn McCarter (glenn@mccarters.net). Some rights reserved.
// This work is licensed under the Creative Commons Attribution License
// To view a copy of this license, visit http://creativecommons.org/licenses/by/2.0/
//
// Makes a height_filed image_map to create an old-fashioned cast iron radiator
//

//#include "Colors.inc"
#include "GIncludeRH.inc"
//#include "GRuler.inc"


global_settings {
	adc_bailout 0.003922
	ambient_light <1.0,1.0,1.0>
	assumed_gamma 1.0
	hf_gray_16 off
	irid_wavelength <0.25,0.18,0.14>
	max_intersections 64
	max_trace_level 10
	number_of_waves 10
}

//===============================================
//             Variables
//===============================================

#declare RadLength = 3.0*Feet;
#declare RL = RadLength;
#declare RadHeight = RadLength/1.33333;
#declare RH = RadHeight;
//#declare WallHeight = WallLength/1.33333; // this allows standard camera to capture full radiator without distortion

// Image can be rendered twice.
// Once for the color map, once (in greyscale) for the height field
#declare    Color_On = 1; // turn on color version for image map
#declare       HF_On = 0; // turn on greyscale version for heightfield generation
#declare Material_On = 0; // turn on solid-color version for material map (DO NOT USE YET)


#declare RS = seed(70832); // randomizer

//===============================================
//             Cameras
//===============================================

#declare Closeup_Cam =
	camera {
		perspective
		direction <0.0,     1.0,  0.0>
		sky       <0.0,     0.0,  1.0>  // Use right handed-system
		up        <0.0,     0.0,  1.0>  // Where Z is up
		right     <1.33333, 0.0,  0.0>
//	angle 6
	angle 45
//	location  <0.01*Feet, -30*Feet, 0.01*Feet>
//	location  <1.01*Feet, -3*Feet, 3.01*Feet>
	location  <0.01*Feet, -3*Feet, 4.01*Feet>
	look_at <0,0,18>
//	location  <WallLength/2, -30*Feet, WallHeight/2>
//	look_at <WallLength/2, 0, WallHeight/2>
}


#declare CamRight = RL;
#declare CamUp = RH;

#declare Ortho_Cam =
	camera {
		orthographic

		sky       <0.0,     0.0,  1.0>  // Use right handed-system
//		up        <0.0,     0.0,  WallHeight>  // use for 4:3 walls
		up        <0.0,     0.0,  CamUp>  // use for full walls
		right     <CamRight, 0.0,  0.0>
//		up        <0.0,     0.0,  WallHeight/20>  // test small patch
//		right     <WallLength/20, 0.0,  0.0>
//		location <CamRight/2, -30*Feet,WallHeight/2> // use for 4:3 walls
//		look_at <CamRight/2, 0, WallHeight/2>
		location <CamRight/2, -30*Feet,CamUp/2> // use for full walls
		look_at <CamRight/2, 0, CamUp/2>
	}

camera {
//	Closeup_Cam // for testing only
	Ortho_Cam // for principal renders
  }


//===============================================
//             Textures
//===============================================

#declare PlainTex =
	texture {
		pigment { color rgb <0.95,.85,.84> }
	}

#declare GradientXHFTex =
	texture {
		pigment {
			gradient x
			color_map {
				[ 0.0 color rgb<0,0,0> ]
				[ 1.0 color rgb<1,1,1> ]
			}
		}
		finish { ambient 1.0 diffuse 0 }
	}


//===============================================
//             Lighting
//===============================================

#declare MyLight =
	light_source {
		<0,0,0>
		color rgb<1,1,1>
//		fade_distance 12*Feet
//		fade_power 1
	}

#if ( Color_On=1 )
	object { MyLight translate <20*Feet, -100*Feet, 20*Feet> } // render
#end

//===============================================
//             Environment
//===============================================

background { color rgb <0,0,0> }
//background { color rgb <0.93, 0.99, 0.4> } // bright yellow

//===============================================
//             Objects
//===============================================

#declare RedSph =
	sphere {
		<0,0,0>,.5
		texture { PlainTex }
		translate <0,0,0.5>
	}

#macro QtrTorXZ(Maj,Min)
	#local BL = Maj+Min+1;
	difference {
		torus { Maj,Min }
		box { <-BL,-BL,-BL>,<0,BL,BL> }
		box { <-BL,-BL,-BL>,<BL,BL,0> }
	} // end difference
#end

#macro QtrRadSecXZ(Maj,Min,Small)
	#local BL = Maj+Min+1;
	union {
		object { QtrTorXZ(Maj,Min) }
		object { QtrTorXZ(Maj+Min-Small/4,Small) }
	} // end union
#end

#macro StraightRadSecX(Height,Min,Small)
	union {
		cylinder { <0,0,0>,<0,0,Height>,Min }
		cylinder { <Min-Small/4,0,0>,<Min-Small/4,0,Height>,Small } 
	} // end union
#end

#macro RadLoop(Height,TopX,Maj,Min,Small)
	#local MMS = Maj+Min+Small;
	union {
		object { StraightRadSecX(TopX,Min,Small) rotate z*0 rotate y*90 translate <0,0,Min+Small> } // bottom
		object { QtrRadSecXZ(Maj,Min,Small) rotate y*90 translate <TopX,0,(Min+Small)+Maj> } // bottom

		object { StraightRadSecX(Height-Maj-Maj-Min-Min-Small,Min,Small) translate <TopX+Maj,0,Maj+Min+Small> }

		object { StraightRadSecX(TopX,Min,Small) rotate y*-90 translate <TopX,0,Height-Min-Small> } // top
		object { QtrRadSecXZ(Maj,Min,Small) translate <TopX,0,Height-Maj-Min-Small> }

	} // end union
#end

#macro RadiatorHF(Height,TopX,Maj,Min,Small,YScale,NumLoops,YSpacing)
	union {
		#local NL=0;
		#while (NL<NumLoops)
			object {
				RadLoop(Height,TopX,Maj,Min,Small)
				scale <1,YScale,1>
				translate <0,(Min)+NL*YSpacing,0>
			}
			#local NL=NL+1;
		#end
		cylinder { <0,Min,0>,<0,NL*YSpacing-Min,0>,Min*0.85 translate <0,0,Small+Min> }
		cylinder { <0,Min,0>,<0,NL*YSpacing-Min,0>,Min*0.85 translate <0,0,Height-Min-Small> }
		#if ( HF_On = 1 )
			GradientXHFTex(RSeed)
		#end
		#if ( Color_On = 1 )
//			texture { PlainTex }
			texture { GradientXHFTex scale <TopX+Maj+Min+Small+0.01,1,1> }
		#end

	} // end union
#end

#macro FullRad(Height,TopX,Maj,Min,Small,YScale,NumLoops,YSpacing)
	union {
		object { RadiatorHF(Height,TopX,Maj,Min,Small,YScale,NumLoops,YSpacing) }
		object { RadiatorHF(Height,TopX,Maj,Min,Small,YScale,NumLoops,YSpacing) scale <-1,1,1> }
		texture { PlainTex }
	} // end union
#end

//===============================================
//             Object Placement
//===============================================

//object { TestSphere scale 2 translate <0,-2,1> }

//object { QtrTorXZ(12,3) translate <0,0,4> texture { PlainTex } }
//object { QtrRadSecXZ(12,3,0.5) scale <1,0.6,1> translate <0,0,4> texture { PlainTex } }
//object { QtrRadSecXZ(12,3,0.5) scale <1,0.6,1> rotate z*-50 translate <12,8,4> texture { PlainTex } }
//object { StraightRadSecX(12,3,0.5) scale <1,0.6,1> translate <4,0,-8> texture { PlainTex } }
//object { RadLoop(24,2,3,1.8,0.25) scale <1,0.6,1> texture { PlainTex } }
//#macro RadiatorHF((Height,TopX,Maj,Min,Small,YScale,NumLoops,YSpacing)
//object { FullRad(24,1.2,2.2,1.8,0.18,0.54,6,3) rotate z*-40 }
object { RadiatorHF(24,1.2,2.2,1.8,0.18,0.54,12,3) rotate z*-90 }
