/*----------------------------------------------------------------------
 * keysock_44bsd.c -- 4.4BSD system dependencies for the PF_KEY socket.
 * 
 * Copyright 1995 by Bao Phan,  Randall Atkinson, & Dan McDonald,
 * All Rights Reserved.  All Rights have been assigned to the US
 * Naval Research Laboratory (NRL).  The NRL Copyright Notice and
 * License governs distribution and use of this software.
 *
 * Patents are pending on this technology.  NRL grants a license
 * to use this technology at no cost under the terms below with
 * the additional requirement that software, hardware, and 
 * documentation relating to use of this technology must include
 * the note that:
 *    	This product includes technology developed at and
 *	licensed from the Information Technology Division, 
 *	US Naval Research Laboratory.
 *
 ----------------------------------------------------------------------*/
/*----------------------------------------------------------------------
#	@(#)COPYRIGHT	1.1a (NRL) 17 August 1995

COPYRIGHT NOTICE

All of the documentation and software included in this software
distribution from the US Naval Research Laboratory (NRL) are
copyrighted by their respective developers.

This software and documentation were developed at NRL by various
people.  Those developers have each copyrighted the portions that they
developed at NRL and have assigned All Rights for those portions to
NRL.  Outside the USA, NRL also has copyright on the software
developed at NRL. The affected files all contain specific copyright
notices and those notices must be retained in any derived work.

NRL LICENSE

NRL grants permission for redistribution and use in source and binary
forms, with or without modification, of the software and documentation
created at NRL provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:

	This product includes software developed at the Information
	Technology Division, US Naval Research Laboratory.

4. Neither the name of the NRL nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THE SOFTWARE PROVIDED BY NRL IS PROVIDED BY NRL AND CONTRIBUTORS ``AS
IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO FINISHED SHALL NRL OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation
are those of the authors and should not be interpreted as representing
official policies, either expressed or implied, of the US Naval
Research Laboratory (NRL).

----------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/domain.h>
#include <sys/protosw.h>

#include <net/raw_cb.h>
#include <net/if.h>
#include <net/if_types.h>
#include <netkey/osdep_44bsd.h>

#include <netinet/in.h>
#include <netinet/in_var.h>

#ifdef INET6
#include <netinet6/in6.h>
#include <netinet6/ipv6.h>
#include <netinet6/in6_var.h>
#endif /* INET6 */

#include <netkey/key.h>
#include <netsec/ipsec.h>

extern struct keyso_cb keyso_cb;

/*
 * Definitions of protocols supported in the KEY domain.
 */

struct	sockaddr key_addr = { 2, PF_KEY, };
struct	sockproto key_proto = { PF_KEY, };

#define KEYREAPERINT 120

#define ROUNDUP(a) \
  ((a) > 0 ? (1 + (((a) - 1) | (sizeof(long) - 1))) : sizeof(long))

struct ifnet;
struct mbuf *m_devget __P((char *, int, int, struct ifnet *, void (*) __P((const void *, void *, size_t))));
int key_sendup __P((struct socket *, struct key_msghdr *, int));
int my_addr __P((SOCKADDR *));
int key_output __P((struct mbuf *, struct socket *));
int key_usrreq __P((struct socket *, int, struct mbuf *, struct mbuf *, struct mbuf *));

int key_sendup(s, km, flags)
struct socket *s;
struct key_msghdr *km;
int flags;
{
  struct mbuf *m;
  struct rawcb *rp = NULL;

  if (flags) {
    if (s && !(s->so_options & SO_USELOOPBACK)) {
      if (keyso_cb.any_count <= 1)
        return 0;
      rp = sotorawcb(s);
      rp->rcb_proto.sp_family = 0;
    }
    
    m = m_devget((caddr_t)km, km->key_msglen, 0, NULL, NULL);
    
    if (km->type)
      key_proto.sp_protocol = km->type;
    raw_input(m, &key_proto, &key_addr, &key_addr);
    if (rp)
      rp->rcb_proto.sp_family = PF_KEY;
  } else {
    MGETHDR(m, M_NOWAIT, MT_DATA);
    if (!m)
      return -1;
    
    m->m_len = m->m_pkthdr.len = 0;
    m->m_next = 0;
    m->m_nextpkt = 0;
    m->m_pkthdr.rcvif = 0;
    m_copyback(m, 0, km->key_msglen, (caddr_t)km);
    
    if (sbappendaddr(&(s->so_rcv), &key_addr, m, NULL)) {
      sorwakeup(s);
    } else {
      m_freem(m);
      return -1;
    }
  }

  return 0;
}

#ifdef notyet
/*----------------------------------------------------------------------
 * key_reaper():
 *      Scan key table and nuke unwanted entries
 ----------------------------------------------------------------------*/
void
key_reaper(whocares)
     void *whocares;
{
  DPRINTF(IDL_GROSS_FINISHED,("Entering key_reaper()\n"));

  timeout(key_reaper, NULL, KEYREAPERINT * HZ);
}
#endif /* notyet */

/*----------------------------------------------------------------------
 * key_init():
 *      Init routine for key socket and key engine
 ----------------------------------------------------------------------*/
void
key_init()
{
  DPRINTF(IDL_FINISHED,("Called key_init().\n"));
  if (key_inittables())
    panic("key_inittables failed!\n");
#ifdef notyet
  timeout(key_reaper, NULL, HZ);
#endif /* notyet */
  bzero((char *)&keyso_cb, sizeof(keyso_cb));
}

/*----------------------------------------------------------------------
 * my_addr():
 *      Determine if an address belongs to one of my configured interfaces.
 *      Currently handles only AF_INET and AF_INET6 addresses.
 ----------------------------------------------------------------------*/
int
my_addr(sa)
     SOCKADDR *sa;
{
  extern struct in6_ifaddr *in6_ifaddr;
  extern struct in_ifaddr *in_ifaddr;
  struct in6_ifaddr *i6a = 0;
  struct in_ifaddr *ia = 0;

  switch(sa->sa_family) {
#ifdef INET6
  case AF_INET6:
    for (i6a = in6_ifaddr; i6a; i6a = i6a->i6a_next) {
      if (IN6_ADDR_EQUAL(((struct sockaddr_in6 *)sa)->sin6_addr, 
			 i6a->i6a_addr.sin6_addr))
	return(1);
    }
    break;
#endif /* INET6 */
  case AF_INET:
    for (ia = in_ifaddr; ia; ia = ia->ia_next) {
      if (((struct sockaddr_in *)sa)->sin_addr.s_addr == 
	   ia->ia_addr.sin_addr.s_addr) 
	return(1);
    }
    break;
  }
  return(0);
}

/*----------------------------------------------------------------------
 * key_output():
 *      Process outbound pf_key message.
 ----------------------------------------------------------------------*/
int
key_output(m, so)
     struct mbuf *m;
     struct socket *so;
{
  struct key_msghdr *km = 0;
  int error = 0;
  int dstfamily = 0;

  DPRINTF(IDL_FINISHED,("key_output() got a message len=%d.\n", m->m_pkthdr.len));

#define senderr(e) \
  { error = (e); if (km) km->key_errno = error; goto flush; }

  if (m == 0 || ((m->m_len < sizeof(long)) && 
		 (m = m_pullup(m, sizeof(long))) == 0)) {
    DPRINTF(IDL_ERROR,("key_output can't pullup mbuf\n"));
    return (ENOBUFS);
  }
  if ((m->m_flags & M_PKTHDR) == 0)
    panic("key_output");

  DDO(IDL_FINISHED,dump_mbuf(m));  
  
  if (m->m_pkthdr.len < sizeof(struct key_msghdr)) {
    DPRINTF(IDL_CRITICAL,("keyoutput: Invalid m_pkthdr.len\n"));
    senderr(EINVAL);
  }

  KMALLOC(km, struct key_msghdr *, m->m_pkthdr.len); 
  if (!km) {
    DPRINTF(IDL_CRITICAL,("keyoutput: Can't malloc memory!\n"));
    senderr(ENOBUFS);
  }

  if (m->m_pkthdr.len != mtod(m, struct key_msghdr *)->key_msglen) {
    DPRINTF(IDL_CRITICAL,("keyoutput: Length mismatch!\n"));
    senderr(EINVAL);
  }

  m_copydata(m, 0, m->m_pkthdr.len, (caddr_t)km);

  km->key_errno = error = key_parse(&km, so, &dstfamily);
  DPRINTF(IDL_FINISHED, ("Back from key_parse\n"));
flush:
  if (km) {
    key_sendup(so, km, 1);
    KFREE(km);
  }
  return (error);
}


/*----------------------------------------------------------------------
 * key_usrreq():
 *      Handles PRU_* for pf_key sockets.
 ----------------------------------------------------------------------*/
int
key_usrreq(so, req, m, nam, control)
	struct socket *so;
	int req;
	struct mbuf *m, *nam, *control;
{
  register int error = 0;
  register struct rawcb *rp = sotorawcb(so);
  int s;

  DPRINTF(IDL_FINISHED,("Entering key_usrreq, req = %d.\n",req));

  if (req == PRU_ATTACH) {
    MALLOC(rp, struct rawcb *, sizeof(*rp), M_PCB, M_WAITOK);
    if ((so->so_pcb = (caddr_t)rp))
      bzero(so->so_pcb, sizeof(*rp));
  }

  if (req == PRU_DETACH && rp) {
    int af = rp->rcb_proto.sp_protocol;
    if (af == AF_INET)
      keyso_cb.ip4_count--;
#ifdef INET6
    else if (af == AF_INET6)
      keyso_cb.ip6_count--;
#endif /* INET6 */
    keyso_cb.any_count--;
  }
  s = splnet();
  error = raw_usrreq(so, req, m, nam, control);
  rp = sotorawcb(so);

  if (req == PRU_ATTACH && rp) {
    int af = rp->rcb_proto.sp_protocol;
    if (error) {
      free((caddr_t)rp, M_PCB);
      splx(s);
      return error;
    }
    if (af == AF_INET)
      keyso_cb.ip4_count++;
#ifdef INET6
    else if (af == AF_INET6)
      keyso_cb.ip6_count++;
#endif /* INET6 */
    keyso_cb.any_count++;
    rp->rcb_faddr = &key_addr;
    soisconnected(so);   /* Key socket, like routing socket, must be
			    connected. */

    /* Possibly set other needed flags/options at creation time in here. */
    so->so_options |= SO_USELOOPBACK; /* Like routing socket, we turn this */
                                      /* on by default                     */
  }
  splx(s);
  return error;
}

/*----------------------------------------------------------------------
 * key_cbinit():
 *      Control block init routine for key socket
 ----------------------------------------------------------------------*/
void
key_cbinit()
{
 /*
  *  This is equivalent to raw_init for the routing socket. 
  *  The key socket uses the same control block as the routing 
  *  socket.
  */
  DPRINTF(IDL_FINISHED,("Called key_cbinit().\n"));
}

/*
 * Protoswitch entry for pf_key 
 */

extern	struct domain keydomain;		/* or at least forward */

#ifdef NetBSD
#define CAST (void *)
#else /* NetBSD */
#define CAST
#endif /* NetBSD */

struct protosw keysw[] = {
{ SOCK_RAW,	&keydomain,	0,		PR_ATOMIC|PR_ADDR,
  CAST raw_input, CAST key_output, CAST raw_ctlinput,	0,
  key_usrreq,
  key_cbinit,	0,		0,		0,
  0,
}
};

struct domain keydomain =
    { PF_KEY, "key", key_init, 0, 0,
      keysw, &keysw[sizeof(keysw)/sizeof(keysw[0])] };
