/*
 * Copyright Johannes Sixt
 * This file is licensed under the GNU General Public License Version 2.
 * See the file COPYING in the toplevel directory of the source directory.
 */

#ifndef MAINWNDBASE_H
#define MAINWNDBASE_H

#include <tqlineedit.h>
#include <tqlayout.h>
#include <tqpushbutton.h>
#include <tqcstring.h>
#include "exprwnd.h"
#include "sys/types.h"			/* pid_t */

// forward declarations
class KDebugger;
class TTYWindow;
class UpdateUI;
class TDEToolBar;
class KStatusBar;
class TDEProcess;
class DebuggerDriver;

class WatchWindow : public TQWidget
{
    Q_OBJECT
public:
    WatchWindow(TQWidget* parent, const char* name, WFlags f = 0);
    ~WatchWindow();
    ExprWnd* watchVariables() { return &m_watchVariables; }
    TQString watchText() const { return m_watchEdit.text(); }
    int columnWidth(int i) const { return m_watchVariables.columnWidth(i); }
    void setColumnWidth(int i, int w) { m_watchVariables.setColumnWidth(i, w); }

protected:
    TQLineEdit m_watchEdit;
    TQPushButton m_watchAdd;
    TQPushButton m_watchDelete;
    ExprWnd m_watchVariables;
    TQVBoxLayout m_watchV;
    TQHBoxLayout m_watchH;

    virtual bool eventFilter(TQObject* ob, TQEvent* ev);
    virtual void dragEnterEvent(TQDragEnterEvent* event);
    virtual void dropEvent(TQDropEvent* event);

signals:
    void addWatch();
    void deleteWatch();
    void textDropped(const TQString& text);

protected slots:
    void slotWatchHighlighted();
};


class DebuggerMainWndBase
{
public:
    DebuggerMainWndBase();
    virtual ~DebuggerMainWndBase();

    /**
     * Sets the command to invoke the terminal that displays the program
     * output. If cmd is the empty string, the default is substituted.
     */
    void setTerminalCmd(const TQString& cmd);
    /**
     * Sets the command to invoke the debugger.
     */
    void setDebuggerCmdStr(const TQString& cmd);
    /**
     * Specifies the file where to write the transcript.
     */
    void setTranscript(const TQString& name);
    /**
     * Starts to debug the specified program using the specified language
     * driver.
     */
    bool debugProgram(const TQString& executable, TQString lang, TQWidget* parent);
    /**
     * Specifies the process to attach to after the program is loaded.
     */
    void setAttachPid(const TQString& pid);

    // the following are needed to handle program arguments
    void setCoreFile(const TQString& corefile);
    void setRemoteDevice(const TQString &remoteDevice);
    void overrideProgramArguments(const TQString& args);
    /** helper around KFileDialog */
    static TQString myGetFileName(TQString caption,
				 TQString dir, TQString filter,
				 TQWidget* parent);
    /** invokes the global options dialog */
    virtual void doGlobalOptions(TQWidget* parent);

protected:
    // settings
    virtual void saveSettings(TDEConfig*);
    virtual void restoreSettings(TDEConfig*);

    // override must return the integrated output window
    virtual TTYWindow* ttyWindow() = 0;

    // statusbar texts
    TQString m_statusActive;

    // output window
    TQString m_outputTermCmdStr;
    TQString m_outputTermKeepScript;
    TDEProcess* m_outputTermProc;
    int m_ttyLevel;
    virtual TQString createOutputWindow();	/* returns terminal name */
    void shutdownTermWindow();

    TQString m_lastDirectory;		/* the dir of the most recently opened file */

    TQString m_transcriptFile;		/* where gdb dialog is logged */

    bool m_popForeground;		/* whether main wnd raises when prog stops */
    int m_backTimeout;			/* when wnd goes back */
    int m_tabWidth;			/* tab width in characters (can be 0) */
    TQString m_sourceFilter;
    TQString m_headerFilter;

    // the debugger proper
    TQString m_debuggerCmdStr;
    KDebugger* m_debugger;
    void setupDebugger(TQWidget* parent,
		       ExprWnd* localVars,
		       ExprWnd* watchVars,
		       TQListBox* backtrace);
    DebuggerDriver* driverFromLang(TQString lang);
    /**
     * This function derives a driver name from the contents of the named
     * file.
     */
    TQString driverNameFromFile(const TQString& exe);

public:
    /*
     * Important! The following functions must be overridden in derived
     * classes and be declared as slots! Note: These must not be declared
     * virtual here since TQt signal mechanism fails miserably (because this
     * class will not be the left-most base class!).
     */
    void newStatusMsg(KStatusBar* statusbar);
    void slotDebuggerStarting();
};

#endif // MAINWNDBASE_H
