/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPFILEDIRVIEW_H
#define KFTPFILEDIRVIEW_H

#include "browser/locationnavigator.h"

#include <tqwidget.h>
#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqtimer.h>
#include <tqdatetime.h>
#include <tqsplitter.h>

#include <kurl.h>
#include <tdefileitem.h>

namespace KFTPSession {
  class Session;
  class Manager;
}

class TDEToolBar;
class KHistoryCombo;
class TDEAction;
class KLineEdit;

namespace KFTPEngine {
  class Thread;
  class Event;
}

namespace KFTPWidgets {

namespace Browser {

class DetailsView;
class ListView;
class TreeView;
class Actions;
class FilterWidget;

/**
 * @author Jernej Kos
 */
class View : public TQWidget
{
Q_OBJECT
friend class Actions;
friend class DetailsView;
friend class ListView;
friend class TreeView;
friend class KFTPSession::Manager;
friend class KFTPSession::Session;
public:
    /**
     * Class constructor.
     */
    View(TQWidget *parent, const char *name, KFTPEngine::Thread *client, KFTPSession::Session *session);
    
    /**
     * Class destructor.
     */
    ~View();
    
    /**
     * Returns the current URL.
     */
    const KURL &url() const;
    
    /**
     * Returns the current history elements.
     *
     * @param index Variable to save the current history position to
     * @return Current history element list
     */
    const TQValueList<LocationNavigator::Element> history(int &index) const;
    
    /**
     * Returns all currenty selected items.
     */
    const KFileItemList *selectedItems() const;
    
    /**
     * Returns all currently selected URLs.
     */
    KURL::List selectedURLs() const;
    
    /**
     * Returns all items in the current sort order.
     */
    const KFileItemList *items() const;
    
    /**
     * Open an URL. Note that if a remote URL is specified the session needs to
     * be connected to the specified host!
     *
     * @param url URL to open
     */
    void openUrl(const KURL &url);
    
    /**
     * Changes the visibility of tree widget.
     *
     * @param visible True to display the tree widget, false to hide it
     */
    void setTreeVisible(bool visible);
    
    /**
     * Changes the "show hidden files" setting.
     *
     * @param value True to enable showing hidden files, false otherwise
     */
    void setShowHidden(bool value);
    
    /**
     * Set the home URL.
     *
     * @param url URL to use as home URL
     */
    void setHomeUrl(const KURL &url);
    
    /**
     * Go one history hop back.
     */
    void goBack();
    
    /**
     * Go one history hop forward.
     */
    void goForward();
    
    /**
     * Go up in the directory structure.
     */
    void goUp();
    
    /**
     * Go the the predefined home URL.
     */
    void goHome();
    
    /**
     * Reload the current directory listing.
     */
    void reload();
    
    /**
     * Renames the provided source file to a new name.
     */
    void rename(const KURL &source, const TQString &name);
    
    /**
     * Returns the details view widget.
     */
    DetailsView *getDetailsView() const { return m_detailsView; }

    /**
     * Returns the tree view widget.
     */
    TreeView *getTreeView() const { return m_tree; }
    
    /**
     * Returns the status label widget.
     */
    TQLabel *getStatusLabel() const { return m_statusMsg; }
    
    /**
     * Returns the associated session.
     */
    KFTPSession::Session *getSession() const { return m_session; }
protected:
    /**
     * Initialize the widget.
     */
    void init();
    
    /**
     * Populate the toolbar.
     */
    void populateToolbar();
private:
    KFTPSession::Session *m_session;
    KFTPEngine::Thread *m_ftpClient;
    
    DetailsView *m_detailsView;
    TreeView *m_tree;
    
    Actions *m_actions;

    TDEToolBar *m_toolBarFirst;  /* Upper toolbar */
    TDEToolBar *m_toolBarSecond; /* Lower toolbar */
    TDEToolBar *m_searchToolBar;

    TQLabel *m_statusMsg;
    TQLabel *m_connDurationMsg;
    TQPushButton *m_sslIcon;
    TQSplitter *m_splitter;

    TQTimer *m_connTimer;
    TQTime m_connDuration;

    KHistoryCombo *m_historyCombo;
    FilterWidget *m_searchFilter;
public slots:
    void updateActions();
    void updateBookmarks();
private slots:
    void slotHistoryEraseClicked();
    void slotHistoryActivated(const TQString &text);
    void slotHistoryChanged();
    
    void slotDisplayCertInfo();
    void slotDurationUpdate();

    void slotEngineEvent(KFTPEngine::Event *event);

    void slotConfigUpdate();
};

}

}

#endif
