/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "logview.h"
#include "misc/kftpconfig.h"
#include "kftpapi.h"

#include <tdelocale.h>
#include <kstdaction.h>
#include <tdefiledialog.h>
#include <tdemessagebox.h>

#include <tqfile.h>
#include <tqtextstream.h>

namespace KFTPWidgets {

LogView::LogView(TQWidget *parent, const char *name)
 : KTextEdit(parent, name)
{
  setTextFormat(TQt::LogText);
  setMaxLogLines(200);
  setFont(KFTPCore::Config::logFont());
  
  // Override disabled background of KTextEdit
  unsetPalette();

  // Init actions
  m_saveToFileAction = KStdAction::saveAs(this, SLOT(slotSaveToFile()), KFTPAPI::getInstance()->mainWindow()->actionCollection(), "log_safeAs");
  m_clearLogAction = KStdAction::clear(this, SLOT(slotClearLog()), KFTPAPI::getInstance()->mainWindow()->actionCollection(), "log_clear");

  append(i18n("<b>KFTPGrabber</b> logger initialized.<br>"));
}


LogView::~LogView()
{
}

TQPopupMenu *LogView::createPopupMenu(const TQPoint &pos)
{
  TQPopupMenu *menu = KTextEdit::createPopupMenu(pos);
  
  m_saveToFileAction->plug(menu);
  menu->insertSeparator();
  m_clearLogAction->plug(menu);
  
  return menu;
}

void LogView::slotSaveToFile()
{
  TQString savePath = KFileDialog::getSaveFileName();
  
  if (!savePath.isEmpty()) {
    TQFile file(savePath);
    if (file.open(IO_WriteOnly)) {
      TQTextStream stream(&file);
      stream << text();
      file.close();
    } else {
      KMessageBox::error(0L, i18n("Unable to open file for writing."));
    }
  }
}

void LogView::slotClearLog()
{
  clear();
}

void LogView::append(const TQString &text)
{
  TQDateTime dt = TQDateTime::currentDateTime();
  TQTextEdit::append("[" + dt.toString("hh:mm:ss") + "] " + text);
  scrollToBottom();
}

void LogView::ftpLog(int type, const TQString &text)
{
  // Set the font if changed
  if (KFTPCore::Config::logFont() != font())
    setFont(KFTPCore::Config::logFont());

  TQString line;
  line = text.stripWhiteSpace();
  line.replace("<", "&lt;");
  line.replace(">", "&gt;");
  switch (type) {
    case 0: {
      TQString code = line.section(" ", 0, 0);
      TQString text = line.mid(line.find(' ')+1);
      append("<font color=" + KFTPCore::Config::logResponsesColor().name() + "><b>" + code + "</b> " + text + "</font><br>");
      break;
    }
    case 1: {
      if (line.left(4) == "PASS")
        line = "PASS (hidden)";

      append("<font color=" + KFTPCore::Config::logCommandsColor().name() + "><b>" + line + "</b></font><br>");
      break;
    }
    case 2: append("<font color=" + KFTPCore::Config::logMultilineColor().name() + ">" + line + "</font><br>"); break;
    case 3: append("<font color=" + KFTPCore::Config::logStatusColor().name() + "><b>*** " + line + "</b></font><br>"); break;
    case 4: append("<font color=" +  KFTPCore::Config::logErrorColor().name() + "><b>*** " + line + "</b></font><br>"); break;
  }
}

}

#include "logview.moc"
