/*
 *  Copyright (c) 2004 Boudewijn Rempt <boud@valdyas.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#ifndef KIS_SELECTION_MANAGER_
#define KIS_SELECTION_MANAGER_

#include "tqobject.h"
#include "tqptrlist.h"

#include "kis_image.h"
#include <koffice_export.h>

class TDEAction;
class KisView;
class KisDoc;
class KisClipboard;

/**
 * The selection manager is responsible selections
 * and the clipboard.
 */
class KRITACORE_EXPORT KisSelectionManager : public TQObject {

    Q_OBJECT
  

public:

    KisSelectionManager(KisView * parent, KisDoc * doc);
    virtual ~KisSelectionManager();

    void setup(TDEActionCollection * collection);

    void addSelectionAction(TDEAction * action);

public:
    /**
     * This function return if the selection should be displayed
     */
    bool displaySelection();
    
public slots:

    void updateGUI();
    void imgSelectionChanged(KisImageSP img);
    void clipboardDataChanged();

    void cut();
    void copy();
    KisLayerSP paste();
    void pasteNew();
    void cutToNewLayer();
    void selectAll();
    void deselect();
    void clear();
    void fillForegroundColor();
    void fillBackgroundColor();
    void fillPattern();
    void reselect();
    void invert();
    void copySelectionToNewLayer();
    void feather();
    void border();
    void expand();
    void contract();
    void smooth();
    void similar();
    void transform();
    void load();
    void save();
    void toggleDisplaySelection();

public:
    void grow (TQ_INT32 xradius, TQ_INT32 yradius);
    void shrink (TQ_INT32 xradius, TQ_INT32 yradius, bool edge_lock);
    void border(TQ_INT32 xradius, TQ_INT32 yradius);
    // the following functions are needed for the siox tool
    // they might be also usefull on its own
    void erode();
    void dilate();

private:
    void fill(const KisColor& color, bool fillWithPattern, const TQString& transactionText);
    
    void computeBorder (TQ_INT32  *circ, TQ_INT32  xradius, TQ_INT32  yradius);
    inline void rotatePointers (TQ_UINT8  **p, TQ_UINT32 n);
    void computeTransition (TQ_UINT8* transition, TQ_UINT8** buf, TQ_INT32 width);

    KisView * m_parent;
    KisDoc * m_doc;

    KisClipboard * m_clipboard;

    TDEAction *m_copy;
    TDEAction *m_cut;
    TDEAction *m_paste;
    TDEAction *m_pasteNew;
    TDEAction *m_cutToNewLayer;
    TDEAction *m_selectAll;
    TDEAction *m_deselect;
    TDEAction *m_clear;
    TDEAction *m_reselect;
    TDEAction *m_invert;
    TDEAction *m_toNewLayer;
    TDEAction *m_feather;
    TDEAction *m_border;
    TDEAction *m_expand;
    TDEAction *m_smooth;
    TDEAction *m_contract;
    TDEAction *m_similar;
    TDEAction *m_transform;
    TDEAction *m_load;
    TDEAction *m_save;
    TDEAction *m_fillForegroundColor;
    TDEAction *m_fillBackgroundColor;
    TDEAction *m_fillPattern;
    TDEToggleAction *m_toggleDisplaySelection;

    TQPtrList<TDEAction> m_pluginActions;

};

#endif // KIS_SELECTION_MANAGER_
