/* This file is part of the KDE project
   Copyright (C) 2005 Jaroslaw Staniek <js@iidea.pl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kexicsvwidgets.h"

#include <tqdir.h>
#include <tqlabel.h>
#include <tqlayout.h>

#include <tdelocale.h>
#include <klineedit.h>
#include <kdialogbase.h>
#include <kactivelabel.h>
#include <kiconloader.h>
#include <kmimetype.h>

#define KEXICSV_OTHER_DELIMITER_INDEX 4

KexiCSVDelimiterWidget::KexiCSVDelimiterWidget( bool lineEditOnBottom, TQWidget * parent )
 : TQWidget(parent, "KexiCSVDelimiterWidget")
 , m_availableDelimiters(KEXICSV_OTHER_DELIMITER_INDEX)

{
	TQBoxLayout *lyr = 
		lineEditOnBottom ? 
		(TQBoxLayout *)new TQVBoxLayout( this, 0, KDialogBase::spacingHint() )
		: (TQBoxLayout *)new TQHBoxLayout( this, 0, KDialogBase::spacingHint() );

	m_availableDelimiters[0]=KEXICSV_DEFAULT_FILE_DELIMITER;
	m_availableDelimiters[1]=";";
	m_availableDelimiters[2]="\t";
	m_availableDelimiters[3]=" ";

	m_combo = new KComboBox(this, "KexiCSVDelimiterComboBox");
	m_combo->insertItem( i18n("Comma \",\"") ); //<-- KEXICSV_DEFAULT_FILE_DELIMITER
	m_combo->insertItem( i18n( "Semicolon \";\"" ) );
	m_combo->insertItem( i18n( "Tabulator" ) );
	m_combo->insertItem( i18n( "Space \" \"" ) );
	m_combo->insertItem( i18n( "Other" ) );
	lyr->addWidget(m_combo);
	setFocusProxy(m_combo);

	m_delimiterEdit = new KLineEdit( this, "m_delimiterEdit" );
//  m_delimiterEdit->setSizePolicy( TQSizePolicy( (TQSizePolicy::SizeType)0, (TQSizePolicy::SizeType)0, 0, 0, m_delimiterEdit->sizePolicy().hasHeightForWidth() ) );
	m_delimiterEdit->setMaximumSize( TQSize( 30, 32767 ) );
	m_delimiterEdit->setMaxLength(1);
	lyr->addWidget( m_delimiterEdit );
	if (!lineEditOnBottom)
		lyr->addStretch(2);

	slotDelimiterChangedInternal(KEXICSV_DEFAULT_FILE_DELIMITER_INDEX); //this will init m_delimiter
	connect(m_combo, TQT_SIGNAL(activated(int)),
	  this, TQT_SLOT(slotDelimiterChanged(int)));
	connect(m_delimiterEdit, TQT_SIGNAL(returnPressed()),
	  this, TQT_SLOT(slotDelimiterLineEditReturnPressed()));
	connect(m_delimiterEdit, TQT_SIGNAL(textChanged( const TQString & )),
	  this, TQT_SLOT(slotDelimiterLineEditTextChanged( const TQString & ) ));
}

void KexiCSVDelimiterWidget::slotDelimiterChanged(int index)
{
	slotDelimiterChangedInternal(index);
	if (index==KEXICSV_OTHER_DELIMITER_INDEX)
		m_delimiterEdit->setFocus();
}

void KexiCSVDelimiterWidget::slotDelimiterChangedInternal(int index)
{
	bool changed = false;
	if (index > KEXICSV_OTHER_DELIMITER_INDEX)
		return;
	else if (index == KEXICSV_OTHER_DELIMITER_INDEX) {
		changed = m_delimiter != m_delimiterEdit->text();
		m_delimiter = m_delimiterEdit->text();
	}
	else {
		changed = m_delimiter != m_availableDelimiters[index];
		m_delimiter = m_availableDelimiters[index];
	}
	m_delimiterEdit->setEnabled(index == KEXICSV_OTHER_DELIMITER_INDEX);
	if (changed)
		emit delimiterChanged(m_delimiter);
}

void KexiCSVDelimiterWidget::slotDelimiterLineEditReturnPressed()
{
	if (m_combo->currentItem() != KEXICSV_OTHER_DELIMITER_INDEX)
		return;
	slotDelimiterChangedInternal(KEXICSV_OTHER_DELIMITER_INDEX);
}

void KexiCSVDelimiterWidget::slotDelimiterLineEditTextChanged( const TQString & )
{
	slotDelimiterChangedInternal(KEXICSV_OTHER_DELIMITER_INDEX);
}

void KexiCSVDelimiterWidget::setDelimiter(const TQString& delimiter)
{
	TQValueVector<TQString>::ConstIterator it = m_availableDelimiters.constBegin();
	int index = 0;
	for (; it != m_availableDelimiters.constEnd(); ++it, index++) {
		if (*it == delimiter) {
			m_combo->setCurrentItem(index);
			slotDelimiterChangedInternal(index);
			return;
		}
	}
	//else: set other (custom) delimiter
	m_delimiterEdit->setText(delimiter);
	m_combo->setCurrentItem(KEXICSV_OTHER_DELIMITER_INDEX);
	slotDelimiterChangedInternal(KEXICSV_OTHER_DELIMITER_INDEX);
}

//----------------------------------------------------

KexiCSVTextQuoteComboBox::KexiCSVTextQuoteComboBox( TQWidget * parent )
 : KComboBox(parent, "KexiCSVTextQuoteComboBox")
{
	insertItem( "\"" );
	insertItem( "'" );
	insertItem( i18n( "None" ) );
}

TQString KexiCSVTextQuoteComboBox::textQuote() const
{
	if (currentItem()==2)
		return TQString();
	return currentText();
}

void KexiCSVTextQuoteComboBox::setTextQuote(const TQString& textQuote)
{
	if (textQuote=="\"" || textQuote=="'")
		setCurrentText(textQuote);
	else if (textQuote.isEmpty())
		setCurrentText(i18n( "None" ));
}

//----------------------------------------------------

KexiCSVInfoLabel::KexiCSVInfoLabel( const TQString& labelText, TQWidget* parent )
 : TQWidget(parent, "KexiCSVInfoLabel")
{
	TQVBoxLayout *vbox = new TQVBoxLayout( this, 0, KDialogBase::spacingHint() );
	TQHBoxLayout *hbox = new TQHBoxLayout( this );
	vbox->addLayout(hbox);
	m_leftLabel = new TQLabel(labelText, this);
	m_leftLabel->setMinimumWidth(130);
	m_leftLabel->setSizePolicy(TQSizePolicy::Fixed, TQSizePolicy::Preferred);
	m_leftLabel->setAlignment(TQt::AlignVCenter | TQt::AlignLeft | TQt::WordBreak);
	hbox->addWidget(m_leftLabel);
	m_iconLbl = new TQLabel(this);
	m_iconLbl->setSizePolicy(TQSizePolicy::Fixed, TQSizePolicy::Preferred);
	m_iconLbl->setAlignment(TQt::AlignVCenter | TQt::AlignLeft);
	m_fnameLbl = new KActiveLabel(this);
	m_fnameLbl->setFocusPolicy(TQ_NoFocus);
	m_fnameLbl->setTextFormat(TQt::PlainText);
	m_fnameLbl->setSizePolicy(TQSizePolicy(TQSizePolicy::MinimumExpanding, TQSizePolicy::MinimumExpanding,1,0));
	m_fnameLbl->setLineWidth(1);
	m_fnameLbl->setFrameStyle(TQFrame::Box);
	m_fnameLbl->setAlignment(TQt::AlignVCenter | TQt::AlignLeft | TQt::WordBreak);
	hbox->addSpacing(5);
	hbox->addWidget(m_iconLbl);
	hbox->addWidget(m_fnameLbl, 1, TQt::AlignVCenter | TQt::AlignLeft | TQt::WordBreak);
	hbox->addSpacing(10);
	m_commentLbl = new KActiveLabel(this);
	m_commentLbl->setFocusPolicy(TQ_NoFocus);
	m_commentLbl->setTextFormat(TQt::PlainText);
	m_commentLbl->setSizePolicy(TQSizePolicy::MinimumExpanding, TQSizePolicy::MinimumExpanding);
	m_commentLbl->setLineWidth(1);
	m_commentLbl->setFrameStyle(TQFrame::Box);
	m_commentLbl->setAlignment(TQt::AlignVCenter | TQt::AlignLeft | TQt::WordBreak);
	hbox->addWidget(m_commentLbl, 0, TQt::AlignVCenter | TQt::AlignRight | TQt::WordBreak);

	m_separator = new TQFrame(this);
	m_separator->setFrameShape(TQFrame::HLine);
	m_separator->setFrameShadow(TQFrame::Sunken);
	vbox->addWidget(m_separator);
}

void KexiCSVInfoLabel::setFileName( const TQString& fileName )
{
	m_fnameLbl->setText( TQDir::convertSeparators(fileName) );
	if (!fileName.isEmpty()) {
		m_iconLbl->setPixmap( 
			KMimeType::pixmapForURL(KURL::fromPathOrURL(fileName), 0, TDEIcon::Desktop) );
	}
}

void KexiCSVInfoLabel::setLabelText( const TQString& text )
{
	m_fnameLbl->setText( text );
//	int lines = m_fnameLbl->lines();
//	m_fnameLbl->setFixedHeight( 
//		TQFontMetrics(m_fnameLbl->currentFont()).height() * lines );
}

void KexiCSVInfoLabel::setIcon(const TQString& iconName)
{
	m_iconLbl->setPixmap( DesktopIcon(iconName) );
}

void KexiCSVInfoLabel::setCommentText( const TQString& text )
{
	m_commentLbl->setText(text);
}

//----------------------------------------------------

TQStringList csvMimeTypes()
{
	TQStringList mimetypes;
	mimetypes << "text/x-csv" << "text/plain" << "all/allfiles";
	return mimetypes;
}

#include "kexicsvwidgets.moc"
