/*
 * Kivio - Visual Modelling and Flowcharting
 * Copyright (C) 2005 Peter Simonsson <psn@linux.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "stenciltexteditor.h"

#include <tqtoolbutton.h>
#include <tqfont.h>
#include <tqcolor.h>
#include <tqpopupmenu.h>

#include <kiconloader.h>
#include <ktextedit.h>
#include <tdefontcombo.h>
#include <kcolorbutton.h>
#include <kdebug.h>
#include <tdelocale.h>
#include <karrowbutton.h>

#include "kivio_stenciltexteditorui.h"

namespace Kivio {

StencilTextEditor::StencilTextEditor(const TQString& caption, TQWidget *parent, const char *name)
  : KDialogBase(parent, name, true, caption, KDialogBase::Ok|KDialogBase::Cancel)
{
  m_hAlign = -1;
  m_vAlign = -1;
  m_mainWidget = new StencilTextEditorUI(this);
  setMainWidget(m_mainWidget);

  m_mainWidget->m_boldButton->setIconSet(SmallIconSet("format-text-bold", 16));
  m_mainWidget->m_italicsButton->setIconSet(SmallIconSet("format-text-italic", 16));
  m_mainWidget->m_underLineButton->setIconSet(SmallIconSet("format-text-underline", 16));

  TQPopupMenu* menu = new TQPopupMenu(m_mainWidget->m_hAlignButton, "hAlignMenu");
  menu->setCheckable(true);
  menu->insertItem(SmallIconSet("format-text-direction-ltr", 16), i18n("Align Left"), TQt::AlignLeft);
  menu->insertItem(SmallIconSet("text_center", 16), i18n("Align Center"), TQt::AlignHCenter);
  menu->insertItem(SmallIconSet("format-text-direction-rtl", 16), i18n("Align Right"), TQt::AlignRight);
  m_mainWidget->m_hAlignButton->setPopup(menu);
  connect(menu, TQT_SIGNAL(activated(int)), this, TQT_SLOT(setHorizontalAlign(int)));
  connect(m_mainWidget->m_hAlignButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(showHAlignPopup()));

  menu = new TQPopupMenu(m_mainWidget->m_vAlignButton, "hAlignMenu");
  menu->setCheckable(true);
  menu->insertItem(SmallIconSet("align_top", 16), i18n("Align Top"), TQt::AlignTop);
  menu->insertItem(SmallIconSet("align_vcenter", 16), i18n("Align Vertical Center"), TQt::AlignVCenter);
  menu->insertItem(SmallIconSet("align_bottom", 16), i18n("Align Bottom"), TQt::AlignBottom);
  m_mainWidget->m_vAlignButton->setPopup(menu);
  connect(menu, TQT_SIGNAL(activated(int)), this, TQT_SLOT(setVerticalAlign(int)));
  connect(m_mainWidget->m_vAlignButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(showVAlignPopup()));

  connect(m_mainWidget->m_fontCombo, TQT_SIGNAL(activated(int)), this, TQT_SLOT(updateFormating()));
  connect(m_mainWidget->m_fontSizeCombo, TQT_SIGNAL(activated(int)), this, TQT_SLOT(updateFormating()));
  connect(m_mainWidget->m_boldButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(updateFormating()));
  connect(m_mainWidget->m_italicsButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(updateFormating()));
  connect(m_mainWidget->m_underLineButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(updateFormating()));
  connect(m_mainWidget->m_textColorButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(updateFormating()));

  m_mainWidget->m_textArea->setFocus();
}

StencilTextEditor::~StencilTextEditor()
{
}

void StencilTextEditor::setFont(const TQFont& font)
{
  m_mainWidget->m_textArea->setFont(font);
  m_mainWidget->m_fontCombo->setCurrentFont(font.family());
  m_mainWidget->m_fontSizeCombo->setCurrentText(TQString::number(font.pointSize()));
  m_mainWidget->m_boldButton->setOn(font.bold());
  m_mainWidget->m_italicsButton->setOn(font.italic());
  m_mainWidget->m_underLineButton->setOn(font.underline());
}

void StencilTextEditor::setFontColor(const TQColor& color)
{
  m_mainWidget->m_textArea->setPaletteForegroundColor(color);
  m_mainWidget->m_textColorButton->setColor(color);
}

void StencilTextEditor::setBackgroundColor(const TQColor& color)
{
  m_mainWidget->m_textArea->setPaletteBackgroundColor(color);
}

void StencilTextEditor::setHorizontalAlign(int flag)
{
  if(m_hAlign >= 0) {
    m_mainWidget->m_hAlignButton->popup()->setItemChecked(m_hAlign, false);
  }

  m_mainWidget->m_hAlignButton->popup()->setItemChecked(flag, true);
  m_mainWidget->m_hAlignButton->setIconSet(*(m_mainWidget->m_hAlignButton->popup()->iconSet(flag)));
  m_mainWidget->m_textArea->setAlignment(flag|verticalAlignment());
  m_hAlign = flag;
}

void StencilTextEditor::setVerticalAlign(int flag)
{
  if(m_vAlign >= 0) {
    m_mainWidget->m_vAlignButton->popup()->setItemChecked(m_vAlign, false);
  }

  m_mainWidget->m_vAlignButton->popup()->setItemChecked(flag, true);
  m_mainWidget->m_vAlignButton->setIconSet(*(m_mainWidget->m_vAlignButton->popup()->iconSet(flag)));
  m_mainWidget->m_textArea->setAlignment(flag|horizontalAlignment());
  m_vAlign = flag;
}

TQFont StencilTextEditor::font() const
{
  TQFont font;
  font.setFamily(m_mainWidget->m_fontCombo->currentFont());
  font.setPointSize(m_mainWidget->m_fontSizeCombo->currentText().toInt());
  font.setBold(m_mainWidget->m_boldButton->isOn());
  font.setItalic(m_mainWidget->m_italicsButton->isOn());
  font.setUnderline(m_mainWidget->m_underLineButton->isOn());

  return font;
}

TQColor StencilTextEditor::fontColor() const
{
  return m_mainWidget->m_textColorButton->color();
}

TQt::AlignmentFlags StencilTextEditor::horizontalAlignment() const
{
  return static_cast<TQt::AlignmentFlags>(m_hAlign);
}

TQt::AlignmentFlags StencilTextEditor::verticalAlignment() const
{
  return static_cast<TQt::AlignmentFlags>(m_vAlign);
}

void StencilTextEditor::setText(const TQString& text)
{
  m_mainWidget->m_textArea->setText(text);
}

TQString StencilTextEditor::text() const
{
  return m_mainWidget->m_textArea->text();
}

void StencilTextEditor::updateFormating()
{
  m_mainWidget->m_textArea->setFont(font());

  m_mainWidget->m_textArea->setPaletteForegroundColor(fontColor());


  m_mainWidget->m_textArea->selectAll(true);
  m_mainWidget->m_textArea->setAlignment(horizontalAlignment()|verticalAlignment());
  m_mainWidget->m_textArea->selectAll(false);

  m_mainWidget->m_textArea->setFocus();
}

void StencilTextEditor::showHAlignPopup()
{
  m_mainWidget->m_hAlignButton->openPopup();
}

void StencilTextEditor::showVAlignPopup()
{
  m_mainWidget->m_vAlignButton->openPopup();
}

}

#include "stenciltexteditor.moc"
