/*
  This file is part of the KDE KOffice project.

  Copyright (C) 2003 Cornelius Schumacher <schumacher@kde.org>
  Copyright (C) 2005 Fredrik Edemar <f_edemar@linux.se>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/
#ifndef KOFFICE_ICONSIDEPANEBASE_H
#define KOFFICE_ICONSIDEPANEBASE_H

#include <tdeactioncollection.h>
#include <tdelistbox.h>

#include <tqbuttongroup.h> 
#include <tqlayout.h>
#include <tqtooltip.h>
#include <tqvbox.h>
#include <tqwidgetstack.h>

#include "koshellsettings.h"


class Navigator;
class IconSidePane;

enum IconViewMode { LargeIcons = 48, NormalIcons = 32, SmallIcons = 22, ShowText = 3, ShowIcons = 5 };

/**
  A @see TQListBoxPixmap Square Box with an optional icon and a text
  underneath.
*/
class EntryItem : public TQListBoxItem
{
  public:
    EntryItem( Navigator *, int _id, const TQString &_text, const TQString & _pix );
    ~EntryItem();


    const TQPixmap *pixmap() const { return &mPixmap; }

    int id() { return mId; }

    void setNewText( const TQString &_text );

    void setHover( bool );
    void setPaintActive( bool );
    bool paintActive() const { return mPaintActive; }

    /**
      returns the width of this item.
    */
    virtual int width( const TQListBox * ) const;
    /**
      returns the height of this item.
    */
    virtual int height( const TQListBox * ) const;

  protected:
    void reloadPixmap();

    virtual void paint( TQPainter *p );

  private:
    TQPixmap mPixmap;
    TQString mPixmapName;
    int const mId;
    Navigator* navigator() const;
    bool mHasHover;
    bool mPaintActive;
};

/**
 * Tooltip that changes text depending on the item it is above.
 * Compliments of "Practical TQt" by Dalheimer, Petersen et al.
 */
class EntryItemToolTip : public TQToolTip
{
  public:
    EntryItemToolTip( TQListBox* parent )
      : TQToolTip( parent->viewport() ), mListBox( parent ) 
      {}
  protected:
    void maybeTip( const TQPoint& p ) {
      // We only show tooltips when there are no texts shown
      if ( KoShellSettings::sidePaneShowText() ) return;
      if ( !mListBox ) return;
      TQListBoxItem* item = mListBox->itemAt( p );
      if ( !item ) return;
      const TQRect itemRect = mListBox->itemRect( item );
      if ( !itemRect.isValid() ) return;

      const EntryItem *entryItem = static_cast<EntryItem*>( item );
      TQString tipStr = entryItem->text();
      tip( itemRect, tipStr );
    }
  private:
    TQListBox* mListBox;
};

/**
  Navigation pane showing all parts relevant to the user
*/
class Navigator : public TDEListBox
{
    Q_OBJECT
  
  public:
    Navigator(bool _selectable, TDEPopupMenu*, IconSidePane *, TQWidget *parent = 0, const char *name = 0 );

    int insertItem(const TQString &_text, const TQString &_pix);

    TQSize sizeHint() const;
    IconViewMode viewMode();
    bool showText();
    bool showIcons();
    void calculateMinWidth();
    bool leftMouseButtonPressed(){return mLeftMouseButtonPressed;}
    int minWidth() { return mMinWidth; }
    void resetWidth() { mMinWidth = 0; }

  signals:
    void itemSelected( int );
    void updateAllWidgets();

  protected:
    void resizeEvent( TQResizeEvent * );
    void enterEvent( TQEvent* );
    void mouseReleaseEvent(TQMouseEvent *e);
    void mousePressEvent(TQMouseEvent *e);

    void setHoverItem( TQListBoxItem*, bool );
    void setPaintActiveItem( TQListBoxItem*, bool );

  protected slots:
    void slotExecuted( TQListBoxItem * );
    void slotMouseOn( TQListBoxItem *item );
    void slotMouseOff();
    void slotShowRMBMenu( TQListBoxItem *, const TQPoint& );

  private:
    IconSidePane *mSidePane;
    int mMinWidth;
    TQListBoxItem *executedItem;
    bool mLeftMouseButtonPressed;
    TDEPopupMenu *mPopupMenu;
    bool mSelectable;
    TQListBoxItem* mMouseOn;
};

class IconSidePane :public TQVBox
{
    Q_OBJECT
  
  public:
    IconSidePane( TQWidget *parent, const char *name = 0 );
    ~IconSidePane();

    void setActionCollection( TDEActionCollection *actionCollection );
    TDEActionCollection *actionCollection() const;

    int insertItem(const TQString & _pix, const TQString &_text);
    int insertItem(int _grp, const TQString & _pix, const TQString &_text);
    int insertGroup(const TQString &_text, bool _selectable, TQObject *_obj = 0L, const char *_slot = 0L);
    void renameItem( int _grp, int _id, const TQString & _text );
    void removeItem( int _grp, int _id );
    void selectGroup(int);
    Navigator *group(int);

    IconViewMode sizeIntToEnum(int size) const;
    IconViewMode viewMode() { return mViewMode; }
    void setViewMode(int choice){mViewMode = sizeIntToEnum(choice);}
    bool showText() { return mShowText; }
    void toogleText(){mShowText=!mShowText;}
    bool showIcons() { return mShowIcons; }
    void toogleIcons(){mShowIcons=!mShowIcons;}
    TQButtonGroup *buttonGroup() { return m_buttongroup; }
    int minWidth();
    void resetWidth();

  public slots:
    void itemSelected(int);
    void updateAllWidgets();

  private slots:
    void buttonClicked();

  private:
    TQWidgetStack *mWidgetstack;
    TQValueList<int> mWidgetStackIds;
    Navigator *mCurrentNavigator;
    TQButtonGroup *m_buttongroup;
    TDEActionCollection *mActionCollection;
    TDEPopupMenu *mPopupMenu;

    IconViewMode mViewMode;
    bool mShowIcons;
    bool mShowText;
};


#endif
