/* KPilot
**
** Copyright (C) 2003 Reinhold Kainhofer <reinhold@kainhofer.com>
**
** This file defines a dialog box that lets the
** user select a set of databases (e.g. which databases
** should be ignored  when doing a backup)
*/

/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program in a file called COPYING; if not, write to
** the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
** MA 02110-1301, USA.
*/

/*
** Bug reports and questions can be sent to kde-pim@kde.org
*/

#include "options.h"

#include <tqlistview.h>
#include <tqpushbutton.h>
#include <tdelistview.h>
#include <tdemessagebox.h>
#include <kpushbutton.h>
#include <klineedit.h>

#include "dbSelection_base.h"
#include "dbSelectionDialog.moc"


KPilotDBSelectionDialog::KPilotDBSelectionDialog(TQStringList &selectedDBs, TQStringList &deviceDBs,
		TQStringList &addedDBs, TQWidget *w, const char *n) :
	KDialogBase(w, n, true, TQString(), KDialogBase::Ok | KDialogBase::Cancel,
		KDialogBase::Ok, false),
	fSelectedDBs(selectedDBs),
	fAddedDBs(addedDBs),
	fDeviceDBs(deviceDBs)
{
	FUNCTIONSETUP;

	fSelectionWidget = new KPilotDBSelectionWidget(this);
	setMainWidget(fSelectionWidget);

	// Fill the encodings list
	TQStringList items(deviceDBs);
	for ( TQStringList::Iterator it = fAddedDBs.begin(); it != fAddedDBs.end(); ++it ) {
		if (items.contains(*it)==0) items << (*it);
	}
	for ( TQStringList::Iterator it = fSelectedDBs.begin(); it != fSelectedDBs.end(); ++it ) {
		if (items.contains(*it)==0) items << (*it);
	}
	items.sort();

	for ( TQStringList::Iterator it = items.begin(); it != items.end(); ++it ) {
		TQCheckListItem*checkitem=new TQCheckListItem(fSelectionWidget->fDatabaseList,
			*it, TQCheckListItem::CheckBox);
		if (fSelectedDBs.contains(*it)) checkitem->setOn(true);
	}

	connect(fSelectionWidget->fNameEdit, TQ_SIGNAL(textChanged( const TQString & )),
		this, TQ_SLOT(slotTextChanged( const TQString &)));
	connect(fSelectionWidget->fAddButton, TQ_SIGNAL(clicked()),
		this, TQ_SLOT(addDB()));
	connect(fSelectionWidget->fRemoveButton, TQ_SIGNAL(clicked()),
		this, TQ_SLOT(removeDB()));
}

KPilotDBSelectionDialog::~KPilotDBSelectionDialog()
{
	FUNCTIONSETUP;
}

void KPilotDBSelectionDialog::addDB()
{
	FUNCTIONSETUP;
	TQString dbname(fSelectionWidget->fNameEdit->text());
	if (!dbname.isEmpty())
	{
		fSelectionWidget->fNameEdit->clear();
		new TQCheckListItem(fSelectionWidget->fDatabaseList, dbname,
			TQCheckListItem::CheckBox);
		fAddedDBs << dbname;
	}
}

void KPilotDBSelectionDialog::removeDB()
{
	FUNCTIONSETUP;
	TQListViewItem*item(fSelectionWidget->fDatabaseList->selectedItem());
	if (item)
	{
		TQString dbname=item->text(0);
		if (fDeviceDBs.contains(dbname))
		{
			KMessageBox::error(this, i18n("This is a database that exists on the device. It was not added manually, so it can not removed from the list."), i18n("Database on Device"));
		}
		else
		{
			fSelectedDBs.remove(dbname);
			fAddedDBs.remove(dbname);
			KPILOT_DELETE(item);
		}
	}
	else
	{
		KMessageBox::information(this, i18n("You need to select a database to delete in the list."),i18n("No Database Selected"), CSL1("NoDBSelected"));
	}
}

TQStringList KPilotDBSelectionDialog::getSelectedDBs()
{
	fSelectedDBs.clear();

	//  update the list of selected databases
	TQListViewItemIterator it( fSelectionWidget->fDatabaseList );
	while ( it.current() ) {
		TQCheckListItem *item = dynamic_cast<TQCheckListItem*>(it.current());
		++it;

		if ( item && item->isOn() )
			fSelectedDBs << item->text();
	}

	return fSelectedDBs;
}

void KPilotDBSelectionDialog::slotTextChanged( const TQString& dbname)
{
	FUNCTIONSETUP;
	fSelectionWidget->fAddButton->setDisabled(dbname.isEmpty());
}
