/***************************************************************************
*   Copyright (C) 2004 by                                                 *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#ifndef PROPERTYLISTVIEW_H
#define PROPERTYLISTVIEW_H

#include <tdelistview.h>

#include "datablocks/element.h"
#include "datablocks/ingredientproperty.h"
#include "datablocks/constraintlist.h"

class RecipeDB;
class TDEPopupMenu;

class PropertyCheckListItem : public TQCheckListItem
{
public:
	PropertyCheckListItem( TQListView* klv, const IngredientProperty &property );
	PropertyCheckListItem( TQListViewItem* it, const IngredientProperty &property );

	~PropertyCheckListItem( void )
	{}
	virtual TQString text( int column ) const;

	IngredientProperty property() const
	{
		return m_property;
	}

protected:
	IngredientProperty m_property;

};

class HidePropertyCheckListItem : public PropertyCheckListItem
{
public:
	HidePropertyCheckListItem( TQListView* klv, const IngredientProperty &property, bool enable = false );
	HidePropertyCheckListItem( TQListViewItem* it, const IngredientProperty &property, bool enable = false );

protected:
	virtual void stateChange( bool on );

private:
	bool m_holdSettings;
};


class ConstraintsListItem: public TQCheckListItem
{
public:
	ConstraintsListItem( TQListView* klv, const IngredientProperty &pty ) : TQCheckListItem( klv, TQString::null, TQCheckListItem::CheckBox )
	{
		// Initialize the constraint data with the the property data
		ctStored = new Constraint();
		ctStored->id = pty.id;
		ctStored->name = pty.name;
		ctStored->perUnit = pty.perUnit;
		ctStored->units = pty.units;
		ctStored->max = 0;
		ctStored->min = 0;
	}

	~ConstraintsListItem( void )
	{
		delete ctStored;
	}

private:
	Constraint *ctStored;

public:
	void setConstraint( const Constraint &constraint )
	{
		delete ctStored;
		ctStored = new Constraint( constraint );

		setOn( ctStored->enabled );
	}
	double maxVal()
	{
		return ctStored->max;
	}
	double minVal()
	{
		return ctStored->min;
	}
	int propertyId()
	{
		return ctStored->id;
	}
	void setMax( double maxValue )
	{
		ctStored->max = maxValue;
		setText( 3, TQString::number( maxValue ) );
	}
	void setMin( double minValue )
	{
		ctStored->min = minValue;
		setText( 2, TQString::number( minValue ) );
	}
	virtual TQString text( int column ) const
	{
		switch ( column ) {
		case 1:
			return ( ctStored->name );
		case 2:
			return ( TQString::number( ctStored->min ) );
		case 3:
			return ( TQString::number( ctStored->max ) );
		case 4:
			return ( TQString::number( ctStored->id ) );
		default:
			return ( TQString::null );
		}
	}
};


class PropertyListView : public TDEListView
{
	Q_OBJECT

public:
	PropertyListView( TQWidget *parent, RecipeDB * );

public slots:
	void reload( void );

protected:
	RecipeDB *database;
	bool m_loading;

protected slots:
	virtual void removeProperty( int id ) = 0;
	virtual void createProperty( const IngredientProperty &property ) = 0;
};



class StdPropertyListView : public PropertyListView
{
	Q_OBJECT

public:
	StdPropertyListView( TQWidget *parent, RecipeDB *, bool editable = false );

protected:
	virtual void removeProperty( int id );
	virtual void createProperty( const IngredientProperty &property );

private slots:
	void showPopup( TDEListView *, TQListViewItem *, const TQPoint & );

	void createNew();
	void remove
		();
	void rename();

	void modProperty( TQListViewItem* i );
	void saveProperty( TQListViewItem* i );

private:
	bool checkBounds( const TQString &name );

	TDEPopupMenu *kpop;
};



class PropertyConstraintListView : public PropertyListView
{
public:
	PropertyConstraintListView( TQWidget *parent, RecipeDB * );

protected:
	virtual void removeProperty( int id );
	virtual void createProperty( const IngredientProperty &property );
};

class CheckPropertyListView : public StdPropertyListView
{
	Q_OBJECT

public:
	CheckPropertyListView( TQWidget *parent, RecipeDB *, bool editable = false );

protected:
	virtual void createProperty( const IngredientProperty &property );

private:
	bool checkBounds( const TQString &name );
};

#endif //PROPERTYLISTVIEW_H
