/***************************************************************************
*   Copyright (C) 2004 by                                                 *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "borderdialog.h"

#include <ntqpushbutton.h>
#include <ntqgroupbox.h>
#include <ntqvbox.h>
#include <ntqlabel.h>
#include <ntqspinbox.h>
#include <ntqlayout.h>
#include <ntqtooltip.h>
#include <ntqwhatsthis.h>

#include <kdebug.h>
#include <tdehtml_part.h>
#include <tdehtmlview.h>
#include <tdelistbox.h>
#include <tdelocale.h>

#include "datablocks/kreborder.h"

BorderDialog::BorderDialog( const KreBorder &border, TQWidget* parent, const char* name )
		: KDialogBase( parent, name, true, TQString::null,
		    KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok )
{
	TQVBox *page = makeVBoxMainWidget();

	borderGroupBox = new TQGroupBox( page, "borderGroupBox" );
	borderGroupBox->setColumnLayout( 0, TQt::Vertical );
	borderGroupBox->layout() ->setSpacing( 6 );
	borderGroupBox->layout() ->setMargin( 11 );
	borderGroupBoxLayout = new TQVBoxLayout( borderGroupBox->layout() );
	borderGroupBoxLayout->setAlignment( TQt::AlignTop );

	layout4 = new TQHBoxLayout( 0, 0, 6, "layout4" );

	layout3 = new TQVBoxLayout( 0, 0, 6, "layout3" );

	styleLabel = new TQLabel( borderGroupBox, "styleLabel" );
	layout3->addWidget( styleLabel );

	styleListBox = new TDEListBox( borderGroupBox, "styleListBox" );
	layout3->addWidget( styleListBox );
	layout4->addLayout( layout3 );

	layout2 = new TQVBoxLayout( 0, 0, 6, "layout2" );

	colorLabel = new TQLabel( borderGroupBox, "colorLabel" );
	layout2->addWidget( colorLabel );

	TQHBox *color_hbox = new TQHBox( borderGroupBox );
	color_hbox->setSizePolicy( TQSizePolicy::MinimumExpanding, TQSizePolicy::MinimumExpanding );
	hsSelector = new KHSSelector( color_hbox );
	hsSelector->setMinimumSize( 140, 70 );
	connect( hsSelector, TQ_SIGNAL( valueChanged( int, int ) ), TQ_SLOT( slotHSChanged( int, int ) ) );

	valuePal = new KValueSelector( color_hbox );
	valuePal->setMinimumSize( 26, 70 );
	connect( valuePal, TQ_SIGNAL( valueChanged( int ) ), TQ_SLOT( slotVChanged( int ) ) );

	layout2->addWidget( color_hbox );
	layout4->addLayout( layout2 );

	layout1 = new TQVBoxLayout( 0, 0, 6, "layout1" );

	widthLabel = new TQLabel( borderGroupBox, "widthLabel" );
	layout1->addWidget( widthLabel );

	widthSpinBox = new TQSpinBox( borderGroupBox, "widthSpinBox" );
	widthSpinBox->setMinValue( 1 );
	layout1->addWidget( widthSpinBox );

	widthListBox = new TDEListBox( borderGroupBox, "widthListBox" );
	layout1->addWidget( widthListBox );
	layout4->addLayout( layout1 );
	borderGroupBoxLayout->addLayout( layout4 );

	borderPreview = new TDEHTMLPart( borderGroupBox );
	borderPreview->view() ->setSizePolicy( TQSizePolicy::Preferred, TQSizePolicy::Fixed );
	borderGroupBoxLayout->addWidget( borderPreview->view() );

	languageChange();

	connect( widthSpinBox, TQ_SIGNAL( valueChanged( int ) ), TQ_SLOT( updatePreview() ) );
	connect( widthListBox, TQ_SIGNAL( highlighted( int ) ), TQ_SLOT( updateSpinBox( int ) ) );
	connect( styleListBox, TQ_SIGNAL( highlighted( int ) ), TQ_SLOT( updatePreview() ) );

	initListBoxs();
	loadBorder( border );

	clearWState( WState_Polished );
}

BorderDialog::~BorderDialog()
{}

void BorderDialog::languageChange()
{
	borderGroupBox->setTitle( i18n( "Requested Border" ) );
	styleLabel->setText( i18n( "Style:" ) );
	colorLabel->setText( i18n( "Color:" ) );
	widthLabel->setText( i18n( "Width:" ) );
}

KreBorder BorderDialog::border() const
{
	int width = widthSpinBox->value();

	TQString style;
	switch ( styleListBox->currentItem() ) {
	case 0:
		style = "none";
		break;
	case 1:
		style = "dotted";
		break;
	case 2:
		style = "dashed";
		break;
	case 3:
		style = "solid";
		break;
	case 4:
		style = "double";
		break;
	case 5:
		style = "groove";
		break;
	case 6:
		style = "ridge";
		break;
	case 7:
		style = "inset";
		break;
	case 8:
		style = "outset";
		break;
	}

	return KreBorder( width, style, selColor );
}

void BorderDialog::loadBorder( const KreBorder &border )
{
	widthSpinBox->setValue( border.width );
	widthListBox->setCurrentItem( border.width - 1 );

	if ( border.style == "none" )
		styleListBox->setCurrentItem( 0 );
	else if ( border.style == "dotted" )
		styleListBox->setCurrentItem( 1 );
	else if ( border.style == "dashed" )
		styleListBox->setCurrentItem( 2 );
	else if ( border.style == "solid" )
		styleListBox->setCurrentItem( 3 );
	else if ( border.style == "double" )
		styleListBox->setCurrentItem( 4 );
	else if ( border.style == "groove" )
		styleListBox->setCurrentItem( 5 );
	else if ( border.style == "ridge" )
		styleListBox->setCurrentItem( 6 );
	else if ( border.style == "inset" )
		styleListBox->setCurrentItem( 7 );
	else if ( border.style == "outset" )
		styleListBox->setCurrentItem( 8 );

	setColor( border.color );

	updatePreview();
}

void BorderDialog::initListBoxs()
{
	styleListBox->insertItem( i18n( "None" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Dotted" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Dashed" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Solid" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Double" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Groove" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Ridge" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Inset" ) );
	styleListBox->insertItem( i18n( "See http://krecipes.sourceforge.net/bordertypes.png for an example", "Outset" ) );

	widthListBox->insertItem( "1" );
	widthListBox->insertItem( "2" );
	widthListBox->insertItem( "3" );
	widthListBox->insertItem( "4" );
	widthListBox->insertItem( "5" );
	widthListBox->insertItem( "6" );
	widthListBox->insertItem( "7" );
}

void BorderDialog::updatePreview()
{
	KreBorder b( border() );

	TQString html_str = TQString( "<html><body><div style=\"vertical-align: middle; border: %1px %2 %3;\"><center><h1>%4</h1></center></div></body></html>" ).arg( b.width ).arg( b.style ).arg( b.color.name() ).arg( i18n( "Border Preview" ) );

	borderPreview->begin();
	borderPreview->write( html_str );
	borderPreview->end();
	borderPreview->show();
}

void BorderDialog::updateSpinBox( int index )
{
	widthSpinBox->setValue( index + 1 );
}

void BorderDialog::slotHSChanged( int h, int s )
{
	int _h, _s, v;
	selColor.hsv( &_h, &_s, &v );
	if ( v < 1 )
		v = 1;

	KColor col;
	col.setHsv( h, s, v );

	setColor( col );
	updatePreview();
}

void BorderDialog::slotVChanged( int v )
{
	int h, s, _v;
	selColor.hsv( &h, &s, &_v );

	KColor col;
	col.setHsv( h, s, v );

	setColor( col );
	updatePreview();
}

void BorderDialog::setColor( const KColor &color )
{
	if ( color == selColor )
		return ;

	selColor = color;

	int h, s, v;
	color.hsv( &h, &s, &v );
	hsSelector->setValues( h, s );
	valuePal->setHue( h );
	valuePal->setSaturation( s );
	valuePal->setValue( v );
	valuePal->updateContents();
	valuePal->repaint( FALSE );
}

#include "borderdialog.moc"
