/***************************************************************************
*   Copyright (C) 2006 by                                                 *
*   Jason Kivlighn (jkivlighn@gmail.com)                                  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "klomanager.h"

#include <kdebug.h>

#include <ntqdom.h>
#include <ntqfile.h>
#include <ntqstringlist.h>

KLOManager::KLOManager()
{
}

KLOManager::~KLOManager()
{

}

TQStringList KLOManager::classes()
{
	TQStringList classesList;
	classesList << "title" << "instructions" << "yield" << "prep_time" << "photo" << "authors" <<
		"categories" << "header" << "ingredients" << "properties" << "ratings";
	return classesList;
}

void KLOManager::processDocument( const TQDomDocument &doc )
{
	TQDomElement layout = doc.documentElement();

	if ( layout.tagName() != "krecipes-layout" ) {
		kdDebug() << "This file does not appear to be a valid Krecipes layout file." << endl;
		return ;
	}

	TQDomNodeList l = layout.childNodes();
	for ( unsigned int i = 0 ; i < l.count(); i++ ) {
		TQDomElement el = l.item( i ).toElement();
		TQString tagName = el.tagName();
		TQDomNodeList subList = el.childNodes();
		/*if ( !*/beginObject( tagName )/* ) {*/; //###: just a thought....
		for ( unsigned int j = 0 ; j < subList.count(); j++ ) {
			TQDomElement subEl = subList.item( j ).toElement();
			TQString subTagName = subEl.tagName();

			if ( subTagName == "background-color" )
				loadBackgroundColor( tagName, getColorAttribute(el,subTagName) );
			else if ( subTagName == "font" )
				loadFont( tagName, getFontAttribute(el,subTagName) );
			else if ( subTagName == "text-color" )
				loadTextColor( tagName, getColorAttribute(el,subTagName) );
			else if ( subTagName == "visible" )
				loadVisibility( tagName, getBoolAttribute(el,subTagName) );
			else if ( subTagName == "alignment" )
				loadAlignment( tagName, getIntAttribute(el,subTagName) );
			else if ( subTagName == "border" )
				loadBorder( tagName, getBorderAttribute(el,subTagName) );
			else if ( subTagName == "columns" )
				loadColumns( tagName, getIntAttribute(el,subTagName) );
			else
				kdDebug() << "Warning: Unknown tag within <" << tagName << ">: " << subTagName << endl;
		}
		endObject();
	}
}

TQDomElement KLOManager::getLayoutAttribute( const TQDomElement &object, const TQString &attribute ) const
{
	TQDomNodeList l = object.childNodes();
	for ( unsigned i = 0; i < l.count(); i++ ) {
		TQDomElement el = l.item( i ).toElement();

		if ( el.tagName() == attribute )
			return el;
	}

	kdDebug() << "Warning: Requested attribute \"" << attribute << "\" not found." << endl;
	return TQDomElement();
}

bool KLOManager::getBoolAttribute( const TQDomElement &object, const TQString &attribute, bool defaultValue ) const
{
	TQDomElement result = getLayoutAttribute( object, attribute );
	if ( result.isNull() ) {
		return defaultValue;
	}
	else {
		return result.text() == "true";
	}
}

TQColor KLOManager::getColorAttribute( const TQDomElement &object, const TQString &attribute, const TQColor &defaultValue ) const
{
	TQDomElement result = getLayoutAttribute( object, attribute );
	if ( result.isNull() ) {
		return defaultValue;
	}
	else {
		return TQColor(result.text());
	}
}

TQString KLOManager::getTextAttribute( const TQDomElement &object, const TQString &attribute, const TQString &defaultValue ) const
{
	TQDomElement result = getLayoutAttribute( object, attribute );
	if ( result.isNull() ) {
		return defaultValue;
	}
	else {
		return result.text();
	}
}

int KLOManager::getIntAttribute( const TQDomElement &object, const TQString &attribute, int defaultValue ) const
{
	TQDomElement result = getLayoutAttribute( object, attribute );
	if ( result.isNull() ) {
		return defaultValue;
	}
	else {
		return result.text().toInt();
	}
}

KreBorder KLOManager::getBorderAttribute( const TQDomElement &object, const TQString &attribute, const KreBorder &defaultValue ) const
{
	TQDomElement result = getLayoutAttribute( object, attribute );
	if ( result.isNull() ) {
		return defaultValue;
	}
	else {
		return KreBorder( result.attribute( "width" ).toInt(), result.attribute( "style" ), TQColor(result.attribute( "color" )) );
	}
}

TQFont KLOManager::getFontAttribute( const TQDomElement &object, const TQString &attribute, const TQFont &defaultValue ) const
{
	TQDomElement result = getLayoutAttribute( object, attribute );
	if ( result.isNull() ) {
		return defaultValue;
	}
	else {
		TQFont font;
		font.fromString(result.text());
		return font;
	}
}

TQString KLOManager::alignmentAsCSS( int alignment )
{
	TQString text;

	if ( alignment & TQt::AlignLeft )
		text += "text-align: left;\n";
	if ( alignment & TQt::AlignRight )
		text += "text-align: right;\n";
	if ( alignment & TQt::AlignHCenter )
		text += "text-align: center;\n";
	if ( alignment & TQt::AlignTop )
		text += "vertical-align: top;\n";
	if ( alignment & TQt::AlignBottom )
		text += "vertical-align: bottom;\n";
	if ( alignment & TQt::AlignVCenter )
		text += "vertical-align: middle;\n";

	return text;
}

TQString KLOManager::borderAsCSS( const KreBorder &border )
{
	return TQString( "border: %1px %2 %3;\n" ).arg(border.width).arg(border.style).arg(border.color.name());
}

TQString KLOManager::bgColorAsCSS( const TQColor &color )
{
	return TQString( "background-color: %1;\n" ).arg( color.name() );
}

TQString KLOManager::fontAsCSS( const TQFont &font )
{
	TQString text;

	text += TQString( "font-family: %1;\n" ).arg( font.family() );
	text += TQString( "font-weight: %1;\n" ).arg( font.weight() );
	text += TQString( "font-size: %1pt;\n" ).arg( font.pointSize() );
	if ( font.underline() )
		text += "text-decoration: underline;\n";
	if ( font.strikeOut() )
		text += "text-decoration: line-through;\n";
	if ( font.bold() )
		text += "font-weight: bold;\n";
	if ( font.italic() )
		text += "font-style: italic;\n";

	return text;
}

TQString KLOManager::textColorAsCSS( const TQColor &color )
{
	return TQString( "color: %1;\n" ).arg( color.name() );
}

TQString KLOManager::visibilityAsCSS( bool visible )
{
	if ( visible )
		return "visibility: visible;\n";
	else
		return "visibility: hidden;\n";
}
