/*
	actions.h - Actions
	Copyright (C) 2005  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __ACTIONS_H__
#define __ACTIONS_H__

#include <tdeapplication.h>

// default external commands
const TQString
	DEFAULT_LOCK_SCREEN_COMMAND("xscreensaver-command -lock"),
// TODO: 2.0: logout command for GNOME (gnome-session-save --kill?)
	DEFAULT_LOGOUT_COMMAND(""),
	DEFAULT_REBOOT_COMMAND("/sbin/reboot"),
	DEFAULT_SHUT_DOWN_COMMAND("/sbin/poweroff");

#define ks_actions Action::getInstance()

class TQPixmap;

/** @short An action manager. */
class Action: public TQObject
{
	Q_OBJECT
public:

	/**
	 * Methods. Don't change these values!
	 */
	enum Method {
		Method_TDE = 0, /**< Standard KDE/TDM method. (default) */
		Method_DefaultCommand = 1, /**< Default external command. (e.g. /sbin/reboot) */
		Method_UserCommand = 2 /**< User command. Any command that can be executed by @c KRun::run. */
	};

	/**
	 * Actions.
	 */
	enum Type {
		Nothing = 0, /**< Nothing. */
		ShutDown = 1, /**< System shut down. */
		Reboot = 2, /**< System reboot. */
		LockScreen = 3, /**< Screen lock. */
		Logout = 4, /**< End current session. */
		Extras = 5 /**< Extras. Execute an item selected from the "Extras..." menu. */
	};

	/** Destructor. */
	virtual ~Action();

	/**
	 * Converts @p action value to the configuration group name.
	 */
	TQString actionToConfigGroup(const Type action) const;

	/**
	 * Returns @c true if the current action is active.
	 */
	inline bool active() const { return _active; }

	/**
	 * Activates the current action if @p yes is @c true.
	 */
	inline void setActive(const bool yes) { _active = yes; }

	/** Returns the currently selected action. */
	inline Type current() const { return _current; }

	/**
	 * Sets current action to @p action.
	 */
	inline void setCurrent(const Type action) { _current = action; }

	/**
	 * Ends current session.
	 * This function is a wrapper for the @c kapp->requestShutDown().
	 * @param type A shut down type
	 * @param action An action to execute
	 * @return @c true if successful; otherwise @c false
	 */
	bool endSession(const TDEApplication::ShutdownType type, const Type action);

	/**
	 * Executes an action.
	 * @param action An action to execute
	 * @param stopTimer If @c true, the current active action and its timer are stopped
	 * @return @c true if successful; otherwise @c false
	 */
	bool exec(const Type action, const bool stopTimer = true);

	/**
	 * Executes an action.
	 * Displays confirmation message before execution.
	 * @param action An action to execute
	 * @param delay A selected delay
	 */
	bool execConfirm(const Type action, const TQString &delay = TQString::null);

	/**
	 * Executes the current action without confirmation.
	 */
	bool execCurrent();

	/**
	 * Returns the current action name as text.
	 */
	TQString getCurrentName() const;

	/**
	 * Returns an icon for @p action.
	 */
	TQPixmap getIcon(const Type action) const;

	/**
	 * Returns an icon name for @p action.
	 */
	TQString getIconName(const Type action) const;

	/**
	 * Initializes and return singleton instance of the @c Action class.
	 */
	inline static Action *getInstance()
	{
		if (!_instance)
			_instance = new Action();

		return _instance;
	}

	/**
	 * Reads method for action from the configuration file.
	 * @param action An action
	 * @retval method A method to use
	 * @retval command A command to run
	 * @return A default command
	 */
	TQString getMethod(
		const Type action,
		Method &method,
		TQString &command
	);

	/**
	 * Changes the action method.
	 * @param group A configuration group (in @b kshutdownrc file)
	 * @param method A new method
	 * @param command A new command
	 */
	void setMethod(const TQString &group, const Method method, const TQString &command) const;

	/**
	 * Returns an action name as text.
	 * @param action An action
	 */
	TQString getName(const Type action) const;

	/** Returns @c true if @p action is @b not disabled by KIOSK. */
	bool isEnabled(const Type action);

	/** Returns @c true if test mode is enabled. */
	inline bool testMode() const { return _testMode; }

	/**
	 * Enables/disables the <b>test mode</b>.
	 * @param yes @c true to enable test mode
	 */
	inline void setTestMode(const bool yes) { _testMode = yes; }

	/**
	 * Executes the current action,
	 * and displays an error message if action failed.
	 */
	void totalExec();

	/**
	 * Returns @c true if application/system is shutting down.
	 */
	inline bool totalExit() const { return _totalExit; }

	/**
	 * Sets <b>total exit</b> info to @p yes.
	 */
	inline void setTotalExit(const bool yes) { _totalExit = yes; }
private:
	static Action *_instance;
	bool
		_active,
		_testMode,
		_totalExit;
	TQMap<Type, TQPixmap> *iconCache;
	Type _current;
	Action();
	bool lockScreen();
public slots:
	void slotLockScreen();
	void slotLogout();
	void slotReboot();
	void slotShutDown();
};

#endif // __ACTIONS_H__
