/***************************************************************************
 *   Copyright (C) 2006-2007 Nicolas Hadacek <hadacek@kde.org>             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef PURL_GUI_H
#define PURL_GUI_H

#include <tqvgroupbox.h>
#include <klineedit.h>
#include <tdelocale.h>
#include <kurllabel.h>

#include "common/global/purl.h"
#include "editlistbox.h"

namespace PURL
{
//-----------------------------------------------------------------------------
extern bool hasMimetype(FileType type);
extern TQPixmap icon(FileType type);
extern Directory getExistingDirectory(const TQString &startDir, TQWidget *widget, const TQString &caption);
extern Url getOpenUrl(const TQString &startDir, const TQString &filter, TQWidget *widget,
                      const TQString &caption);
extern UrlList getOpenUrls(const TQString &startDir, const TQString &filter, TQWidget *widget,
                          const TQString &caption);
enum SaveAction { NoSaveAction, AskOverwrite, CancelIfExists };
extern Url getSaveUrl(const TQString &startDir, const TQString &filter, TQWidget *widget,
                      const TQString &caption, SaveAction action);

//-----------------------------------------------------------------------------
class Label : public KURLLabel
{
Q_OBJECT
  
public:
  Label(const TQString &url, const TQString &text, TQWidget *parent = 0, const char *name = 0);

private slots:
  void urlClickedSlot();
};

//-----------------------------------------------------------------------------
class BaseWidget : public TQWidget
{
Q_OBJECT
  
public:
  BaseWidget(TQWidget *parent = 0, const char *name = 0);
  BaseWidget(const TQString &defaultDir, TQWidget *parent = 0, const char *name = 0);
  KLineEdit *lineEdit() { return _edit; }

signals:
  void changed();

protected slots:
  virtual void buttonClicked() = 0;

protected:
  TQString    _defaultDir;
  KLineEdit *_edit;

  void init();
};

//-----------------------------------------------------------------------------
class DirectoryWidget : public BaseWidget
{
Q_OBJECT
  
public:
  DirectoryWidget(TQWidget *parent = 0, const char *name = 0) : BaseWidget(parent, name) {}
  DirectoryWidget(const TQString &defaultDir, TQWidget *parent = 0, const char *name = 0) : BaseWidget(defaultDir, parent, name) {}
  void setDirectory(const Directory &dir) { _edit->setText(dir.path()); }
  Directory directory() const { return _edit->text(); }

protected slots:
  virtual void buttonClicked();
};

//-----------------------------------------------------------------------------
class DirectoriesWidget : public TQVGroupBox
{
Q_OBJECT
  
public:
  DirectoriesWidget(const TQString &title, TQWidget *parent = 0, const char *name = 0);
  DirectoriesWidget(const TQString &title, const TQString &defaultDir, TQWidget *parent = 0, const char *name = 0);
  void setDirectories(const TQStringList &dirs) { _editListBox->setTexts(dirs); }
  TQStringList directories() const { return _editListBox->texts(); }

signals:
  void changed();

private:
  EditListBox *_editListBox;
  void init(const TQString &defaultDir);
};

//-----------------------------------------------------------------------------
class UrlWidget : public BaseWidget
{
Q_OBJECT
  
public:
  UrlWidget(const TQString &filter, TQWidget *parent = 0, const char *name = 0)
    : BaseWidget(parent, name), _filter(filter) {}
  UrlWidget(const TQString &defaultDir, const TQString &filter, TQWidget *parent = 0, const char *name = 0)
    : BaseWidget(defaultDir, parent, name), _filter(filter) {}
  Url url() const { return PURL::Url::fromPathOrUrl(_edit->text()); }
  void setUrl(const Url &url) { _edit->setText(url.filepath()); }

protected slots:
  virtual void buttonClicked();

private:
  TQString _filter;
};

} // namespace

#endif
