#!/usr/bin/env python


import sys, string
from PyTQt.tqt import *

#
## Constructs an analog clock widget that uses an internal TQTimer.
#
def TQMIN( x, y ):
  if y > x:
    return y
  return x

#
## Constructs an analog clock widget that uses an internal TQTimer.
#

class AnalogClock( TQWidget ):
    def __init__( self, *args ):
        TQWidget.__init__(*(self,) + args)
        self.time = TQTime.currentTime()     # get current time
        internalTimer = TQTimer( self )      # create internal timer
        self.connect( internalTimer, SIGNAL("timeout()"), self.timeout )
        internalTimer.start( 5000 )         # emit signal every 5 seconds

#
## The TQTimer::timeout() signal is received by this slot.
#

    def timeout( self ):
        new_time = TQTime.currentTime()                # get the current time
        if new_time.minute() != self.time.minute():    # minute has changed
            self.update()

#
## The clock is painted using a 1000x1000 square coordinate system.
#
    def paintEvent( self, tqe ):            # paint clock
        if not self.isVisible():                     # is is invisible
            return
        self.time = TQTime.currentTime()             # save current time

        pts = TQPointArray()
        paint = TQPainter( self )
        paint.setBrush( self.foregroundColor() )    # fill with foreground color

        cp = TQPoint( self.rect().center() )         # widget center point
        d = TQMIN( self.width(), self.height() )     # we want a circular clock

        matrix = TQWMatrix()                         # setup transformation matrix
        matrix.translate( cp.x(), cp.y() )          # origin at widget center
        matrix.scale( d / 1000.0, d / 1000.0 )      # scale coordinate system

        h_angle = 30 * ( self.time.hour() % 12 - 3 ) + self.time.minute() / 2
        matrix.rotate( h_angle )                    # rotate to draw hour hand
        paint.setWorldMatrix( matrix )
        pts.setPoints( [ -20,0, 0,-20, 300,0, 0,20 ] )
        paint.drawPolygon( pts )                    # draw hour hand
        matrix.rotate( -h_angle )                   # rotate back to zero

        m_angle = ( self.time.minute() - 15 ) * 6
        matrix.rotate( m_angle )                    # rotate to draw minute hand
        paint.setWorldMatrix( matrix )
        pts.setPoints( [ -10,0, 0,-10, 400,0, 0,10 ] )
        paint.drawPolygon( pts )                    # draw minute hand
        matrix.rotate( -m_angle )                   # rotate back to zero

        for i in range( 0, 12 ):                     # draw hour lines
            paint.setWorldMatrix( matrix )
            paint.drawLine( 450,0, 500,0 )
            matrix.rotate( 30 )


class DigitalClock( TQLCDNumber ):
    def __init__( self, *args ): 
        TQLCDNumber.__init__(*(self,) + args)
        self.showingColon = 0
        self.setFrameStyle(TQFrame.Panel | TQFrame.Raised)
        self.setLineWidth( 2 )
        self.showTime()
        self.normalTimer = self.startTimer( 500 )
        self.showDateTimer = -1

    def timerEvent( self, e ):
        if e.timerId() == self.showDateTimer:
            self.stopDate()
        else:
            if self.showDateTimer == -1:
                self.showTime()
  
    def mousePressEvent( self, e ):
        if e.button() == TQt.LeftButton:
            self.showDate()

    def showDate( self ):
        if self.showDateTimer != -1:
            return
        d = TQDate.currentDate()
        self.display('%2d %2d' % (d.month(), d.day()))
        self.showDateTimer = self.startTimer(2000)

    def stopDate( self ):
        self.killTimer(self.showDateTimer)
        self.showDateTimer = -1
        self.showTime()

    def showTime( self ):
        self.showingColon = not self.showingColon
        s = list(str(TQTime.currentTime().toString())[:5]) #.left(5)
        if not self.showingColon:
            s[2] = ' '
        if s[0] == '0':
            s[0] = ' '
        s = ''.join(s)
        self.display( s )

    def TQMIN( x, y ):
        if y > x:
            return y
        return x

TRUE  = 1
FALSE = 0
MOVIEFILENAME = "trolltech.gif"

#
# WidgetView contains lots of TQt widgets.
#

class WidgetView ( TQWidget ):
    def __init__( self, *args ):
         TQWidget.__init__(*(self,) + args)
         
         # Set the window caption/title
         self.setCaption( "TQt Widgets Demo Application" )

         # Install an application-global event filter
         tqApp.installEventFilter( self )

         # Create a layout to position the widgets
         self.topLayout = TQVBoxLayout( self, 10 )

         # Create a grid layout to hold most of the widgets
         self.grid = TQGridLayout( 6, 3 )

         # This layout will get all of the stretch
         self.topLayout.addLayout( self.grid, 10 )

         # Create a menubar
         self.menubar = TQMenuBar( self )
         self.menubar.setSeparator( TQMenuBar.InWindowsStyle )
         
         # Create an easter egg
         TQToolTip.add( self.menubar, TQRect( 0, 0, 2, 2 ), "easter egg" )

         self.popup = TQPopupMenu()
         self.id = self.popup.insertItem( "&New" )
         self.popup.setItemEnabled( self.id, FALSE )
         self.id = self.popup.insertItem( "&Open" )
         self.popup.setItemEnabled( self.id, FALSE )
         self.popup.insertSeparator()
         self.popup.insertItem( "&Quit", tqApp, SLOT("quit()"), TQt.CTRL+TQt.Key_Q )

         self.menubar.insertItem( "&File", self.popup )

         # Must tell the layout about a menubar in a widget
         self.topLayout.setMenuBar( self.menubar )

         # Create an analog and a digital clock
         self.aclock = AnalogClock( self )
         self.aclock.resize( 50, 50 )
         self.dclock = DigitalClock( self )
         self.dclock.setMaximumWidth( 200 )
         self.grid.addWidget( self.aclock, 0, 2 )
         self.grid.addWidget( self.dclock, 1, 2 )

         # Give the dclock widget a blue palette
         col = TQColor()
         col.setRgb( 0xaa, 0xbe, 0xff )
         self.dclock.setPalette( TQPalette( col ) )

         # make tool tips for both of them
         TQToolTip.add( self.aclock, "custom widget: analog clock" )
         TQToolTip.add( self.dclock, "custom widget: digital clock" )

         # Create a push button.
         self.pb = TQPushButton( self, "button1" )	# create button 1
         self.pb.setText( "Push button 1" )
         self.pb.setFixedHeight( self.pb.sizeHint().height() )
         self.grid.addWidget( self.pb, 0, 0, TQt.AlignVCenter )
         self.connect( self.pb, SIGNAL("clicked()"), self.button1Clicked )
         TQToolTip.add( self.pb, "push button 1" )
         self.pm = TQPixmap()
         self.pix = self.pm.load( "tqt.png" )		# load pixmap for button 2
         if not self.pix:
             TQMessageBox.information( None, "TQt Widgets Example",
                                      "Could not load the file \"tqt.png\", which\n"
                                      "contains an icon used...\n\n"
                                      "The text \"line 42\" will be substituted.",
                                      TQMessageBox.Ok + TQMessageBox.Default )

         # Create a label containing a TQMovie
         self.movielabel = TQLabel( self, "label0" )
         self.movie = TQMovie( MOVIEFILENAME )
         self.movie.connectStatus( self.movieStatus )
         self.movie.connectUpdate( self.movieUpdate )
         self.movielabel.setFrameStyle( TQFrame.Box | TQFrame.Plain )
         self.movielabel.setMovie( self.movie )
         self.movielabel.setMargin( 0 )
         self.movielabel.setFixedSize( 128 + self.movielabel.frameWidth() * 2,
                                        64 + self.movielabel.frameWidth() * 2 )
         self.grid.addWidget( self.movielabel, 0, 1, TQt.AlignCenter )
         TQToolTip.add( self.movielabel, "movie" )

         # Create a group of check boxes
         self.bg = TQButtonGroup( self, "checkGroup" )
         self.bg.setTitle( "Check Boxes" )
         self.grid.addWidget( self.bg, 1, 0 )

         # Create a layout for the check boxes
         self.vbox = TQVBoxLayout(self.bg, 10)

         self.vbox.addSpacing( self.bg.fontMetrics().height() )

         self.cb = list(range(3))
         self.cb[0] = TQCheckBox( self.bg )
         self.cb[0].setText( "Read" )
         self.vbox.addWidget( self.cb[0] )
         self.cb[0].setMinimumSize( self.cb[0].sizeHint() )
         self.cb[1] = TQCheckBox( self.bg )
         self.cb[1].setText( "Write" )
         self.vbox.addWidget( self.cb[1] )
         self.cb[1].setMinimumSize( self.cb[1].sizeHint() )
         self.cb[2] = TQCheckBox( self.bg )
         self.cb[2].setText( "Execute" )
         self.cb[2].setMinimumSize( self.cb[2].sizeHint() )
         self.vbox.addWidget( self.cb[2] )
         self.bg.setMinimumSize( self.bg.childrenRect().size() )
         self.vbox.activate()

         self.connect( self.bg, SIGNAL("clicked(int)"), self.checkBoxClicked )
         
         TQToolTip.add( self.cb[0], "check box 1" )
         TQToolTip.add( self.cb[1], "check box 2" )
         TQToolTip.add( self.cb[2], "check box 3" )

         # Create a group of radio buttons
         self.bg = TQButtonGroup( self, "radioGroup" )
         self.bg.setTitle( "Radio buttons" )

         self.grid.addWidget( self.bg, 1, 1 )

         # Create a layout for the radio buttons
         self.vbox = TQVBoxLayout( self.bg, 10 )

         self.vbox.addSpacing( self.bg.fontMetrics().height() )
         self.rb = TQRadioButton( self.bg )
         self.rb.setText( "&AM" )
         self.rb.setChecked( TRUE )
         self.vbox.addWidget( self.rb )
         self.rb.setMinimumSize( self.rb.sizeHint() )
         TQToolTip.add( self.rb, "radio button 1" )
         self.rb = TQRadioButton( self.bg )
         self.rb.setText( "&FM" )
         self.vbox.addWidget( self.rb )
         self.rb.setMinimumSize( self.rb.sizeHint() )
         TQToolTip.add( self.rb, "radio button 2" )
         self.rb = TQRadioButton( self.bg )
         self.rb.setText( "&Short Wave" )
         self.vbox.addWidget( self.rb )
         self.rb.setMinimumSize( self.rb.sizeHint() )
         self.vbox.activate()

         self.connect( self.bg, SIGNAL("clicked(int)"), self.radioButtonClicked )
         TQToolTip.add( self.rb, "radio button 3" )

         # Create a list box
         self.lb = TQListBox( self, "listBox" )
         for i in range( 0, 100, 1 ): # fill list box
             txt = TQString()
             txt = "line %d" % i
             if i == 42 and self.pix:
                 self.lb.insertItem( self.pm )
             else:
                 self.lb.insertItem( txt )
    
         self.grid.addMultiCellWidget( self.lb, 2, 4, 0, 0 )
         self.connect( self.lb, SIGNAL("selected(int)"), self.listBoxItemSelected )
         TQToolTip.add( self.lb, "list box" )

         self.vbox = TQVBoxLayout( 8 )
         self.grid.addLayout( self.vbox, 2, 1 )

         # Create a slider
         self.sb = TQSlider( 0, 300, 1, 100, TQSlider.Horizontal, self, "Slider" )
         self.sb.setTickmarks( TQSlider.Below )
         self.sb.setTickInterval( 10 )
         self.sb.setFocusPolicy( TQWidget.TabFocus )
         self.sb.setFixedHeight( self.sb.sizeHint().height() )
         self.vbox.addWidget( self.sb )

         self.connect( self.sb, SIGNAL("valueChanged(int)"), self.sliderValueChanged )
         TQToolTip.add( self.sb, "slider" )

         # Create a combo box
         self.combo = TQComboBox( FALSE, self, "comboBox" )
         self.combo.insertItem( "darkBlue" )
         self.combo.insertItem( "darkRed" )
         self.combo.insertItem( "darkGreen" )
         self.combo.insertItem( "blue" )
         self.combo.insertItem( "red" )
         self.combo.setFixedHeight( self.combo.sizeHint().height() )
         self.vbox.addWidget( self.combo )
         self.connect( self.combo, SIGNAL("activated(int)"), self.comboBoxItemActivated )
         TQToolTip.add( self.combo, "read-only combo box" )

         # Create an editable combo box
         self.edCombo = TQComboBox( TRUE, self, "edComboBox" )
         self.edCombo.insertItem( "Permutable" )
         self.edCombo.insertItem( "Malleable" )
         self.edCombo.insertItem( "Adaptable" )
         self.edCombo.insertItem( "Alterable" )
         self.edCombo.insertItem( "Inconstant" )
         self.edCombo.setFixedHeight( self.edCombo.sizeHint().height() )
         self.vbox.addWidget( self.edCombo )
         self.connect( self.edCombo, SIGNAL("activated(const TQString &)"), self.edComboBoxItemActivated)
         TQToolTip.add( self.edCombo, "editable combo box" )

         self.edCombo.setAutoCompletion( TRUE )
         
         self.vbox.addStretch( 1 )

         self.vbox = TQVBoxLayout( 8 )
         self.grid.addLayout( self.vbox, 2, 2 )

         # Create a spin box
         self.spin = TQSpinBox( 0, 10, 1, self, "spin" )
         self.spin.setSuffix( " mm" )
         self.spin.setSpecialValueText( "Auto" )
         self.spin.setMinimumSize( self.spin.sizeHint() )
         self.connect( self.spin, SIGNAL( "valueChanged(const TQString &)" ), self.spinBoxValueChanged )
         TQToolTip.add( self.spin, "spin box" )
         self.vbox.addWidget( self.spin )
    
         self.vbox.addStretch( 1 )

         # Create a multi line edit
         self.mle = TQMultiLineEdit( self, "multiLineEdit" )

         self.grid.addMultiCellWidget( self.mle, 3, 3, 1, 2 )
         self.mle.setMinimumHeight( self.mle.fontMetrics().height() * 3 )
         self.mle.setText("This is a TQMultiLineEdit widget,\n"
                          "useful for small multi-line\n"
                          "input fields.")
         TQToolTip.add( self.mle, "multi line editor" )

         # Create a single line edit
         self.le = TQLineEdit( self, "lineEdit" )
         self.grid.addMultiCellWidget( self.le, 4, 4, 1, 2 )
         self.le.setFixedHeight( self.le.sizeHint().height() )
         self.connect( self.le, SIGNAL("textChanged(const TQString &)"), self.lineEditTextChanged )
         TQToolTip.add( self.le, "single line editor" )

         # Create a horizontal line (sort of TQFrame) above the message line
         self.separator = TQFrame( self, "separatorLine" )
         self.separator.setFrameStyle( TQFrame.HLine | TQFrame.Sunken )
         self.separator.setFixedHeight( self.separator.sizeHint().height() )
         self.grid.addMultiCellWidget( self.separator, 5, 5, 0, 2 )
         TQToolTip.add( self.separator, "tool tips on a separator! wow!" )

         self.grid.setRowStretch( 0, 0 )
         self.grid.setRowStretch( 1, 0 )
         self.grid.setRowStretch( 2, 0 )
         self.grid.setRowStretch( 3, 1 )
         self.grid.setRowStretch( 4, 1 )
         self.grid.setRowStretch( 5, 0 )

         self.grid.setColStretch( 0, 1 )
         self.grid.setColStretch( 1, 1 )
         self.grid.setColStretch( 2, 1 )

         # Create an label and a message in a plain widget
         # The message is updated when buttons are clicked etc.

         self.hbox = TQHBoxLayout()
         self.topLayout.addLayout( self.hbox )
         self.msgLabel = TQLabel( self, "msgLabel" )
         self.msgLabel.setText( "Message:" )
         self.msgLabel.setAlignment( TQt.AlignHCenter | TQt.AlignVCenter )
         self.msgLabel.setFixedSize( self.msgLabel.sizeHint() )
         self.hbox.addWidget( self.msgLabel )
         TQToolTip.add( self.msgLabel, "label 1" )

         self.msg = TQLabel( self, "message" )
         self.msg.setFrameStyle( TQFrame.Panel | TQFrame.Sunken )
         self.msg.setAlignment( TQt.AlignCenter )
         self.msg.setFont( TQFont( "times", 12, TQFont.Bold ) )
         self.msg.setText( "Message" )
         self.msg.setFixedHeight( self.msg.sizeHint().height() )
         self.msg.setText( "" )
         self.hbox.addWidget( self.msg, 5 )
         TQToolTip.add( self.msg, "label 2" )

         self.topLayout.activate()

    def setStatus(self, text):
        self.msg.setText( text )

    def movieUpdate( self, r ):
        # Uncomment this to test animated icons on your window manager
        self.setIcon( self.movie.framePixmap() )
        
    def movieStatus( self, s ):
        if s == TQMovie.SourceEmpty or s == TQMovie.UnrecognizedFormat:
            pm = TQPixmap('tt-logo.png')
            self.movielabel.setPixmap(pm)
            self.movielabel.setFixedSize(pm.size())
        else:
            if ( self.movielabel.movie() ):      # for flicker-free animation:
                self.movielabel.setBackgroundMode( TQWidget.NoBackground )
   
    def button1Clicked( self ):
        self.msg.setText( "The first push button was clicked" )

    def checkBoxClicked( self, id ):
        txt = "Check box %s clicked : " % str(id)
        chk = ["-","-","-"]
        if self.cb[0].isChecked():
            chk[0] = "r"
        if self.cb[1].isChecked():
            chk[1] = "w"
        if self.cb[2].isChecked():
            chk[2] = "x"
        txt = txt + str(chk[0]+chk[1]+chk[2])
        self.msg.setText( txt )
        
    def edComboBoxItemActivated( self, text):
        self.msg.setText( "Editable Combo Box set to %s" % text )

    def radioButtonClicked( self, id ):
        self.msg.setText( "Radio button #%d clicked" % id )

    def listBoxItemSelected( self, index ):
        self.msg.setText( "List box item %d selected" % index )

    def sliderValueChanged( self, value ):
        self.msg.setText( "Movie set to %d%% of normal speed" % value )
        self.movie.setSpeed( value )

    def comboBoxItemActivated( self, index ):
        self.msg.setText( "Combo box item %d activated" % index )
        p = TQApplication.palette()
        if index == 0:
            p.setColor( TQColorGroup.Highlight, TQt.darkBlue )
        elif index == 1:
            p.setColor( TQColorGroup.Highlight, TQt.darkRed )
        elif index == 2:
            p.setColor( TQColorGroup.Highlight, TQt.darkGreen )
        elif index == 3:
            p.setColor( TQColorGroup.Highlight, TQt.blue )
        elif index == 4:
            p.setColor( TQColorGroup.Highlight, TQt.red )
        TQApplication.setPalette( p, TRUE )

    def lineEditTextChanged( self, newText ):
        self.msg.setText("Line edit text: " + str(newText))

    def spinBoxValueChanged( self, valueText ):
        self.msg.setText("Spin box value: " + str(valueText))

    # All application events are passed throught this event filter.
    # We're using it to display some information about a clicked
    # widget (right mouse button + CTRL).
    #def eventFilter( self, event ):
    #    identify_now = TRUE
    #    if event.type() == Event_MouseButtonPress and identify_now:
    #        e = TQMouseEvent( event )
    #        if (e.button() == TQt.RightButton) and (e.state() & TQt.ControlButton) != 0:
    #            txt = TQString( "The clicked widget is a\n" )
    #            txt = txt + TQObect.className()
    #            txt = txt + "\nThe widget's name is\n"
    #            if TQObject.name():
    #                txt = txt + TQObject.name()
    #            else:
    #                txt = txt + "<no name>"
    #            identify_now = FALSE		# don't do it in message box
    #            TQMessageBox.message( "Identify Widget", txt, 0, TQObject )
    #            identify_now = TRUE;		# allow it again
    #    return FALSE				# don't eat event

################################################################################################

#TQApplication.setColourSpec( TQApplication.CustomColor )
a = TQApplication( sys.argv )

w = WidgetView()
a.setMainWidget( w )
w.show()
a.exec_loop()
