/*
 *  Copyright (C) 2002-2003, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include <tqfile.h>
#include <tdeaction.h>
#include <kdebug.h>
#include <kstdaction.h>

#include "xmlactionclient.h"
#include "xmlactionclient.moc"

namespace KJSEmbed {

//
// XML Tags and Attributes
//

TQString tag_header("header");
TQString tag_action("action");
TQString tag_type( "type" );
TQString tag_label( "label" );
TQString tag_icons( "icons" );
TQString tag_shortcut( "shortcut" );
TQString tag_name( "name" );
TQString tag_group( "group" );
TQString tag_text( "text" );
TQString tag_statustext( "statustext" );
TQString tag_whatsthis( "whatsthis" );
TQString tag_script( "script" );
TQString tag_data( "data" );
TQString tag_item( "item" );

TQString attr_type( "type" );
TQString attr_src( "src" );
TQString attr_exclusive( "exclusive" );

TQString type_include( "include" );
TQString type_debug( "debug" );

//
// Default Runner
//

bool XMLActionRunner::run( XMLActionClient *client, const XMLActionClient::XMLActionScript &s )
{
//    kdWarning() << "Runner:run called, type=" << s.type << " text=" << s.text << " src=" << s.src << endl;

    if ( s.type == type_include ) {
	kdDebug(80001) << "IncludeAction: " << s.src << endl;
	return client->load( s.src );
    }
    else if ( s.type == type_debug ) {
	kdDebug(80001) << "DebugAction: " << s.text << endl;
	return true;
    }

    return false;
}

//
// Main Client Class
//

XMLActionClient::XMLActionClient( TQObject *parent, const char *name )
    : TQObject( parent, name ? name : "XMLActionClient" ),
      ac(0), actrun(0)
{
}

XMLActionClient::~XMLActionClient()
{
}

bool XMLActionClient::load( const TQString &filename )
{
    XMLActionHandler h( this );
    return load( &h, filename );
}

bool XMLActionClient::load( XMLActionHandler *hand, const TQString &filename )
{
    TQFile f( filename );
    TQXmlInputSource src( &f );

    TQXmlSimpleReader reader;
    reader.setContentHandler( hand );
    bool ok = reader.parse( src );
    if ( !ok ) {
	kdWarning() << "Loading actionset " << filename << " failed, " << hand->errorString() << endl;
    }

    return ok;
}

bool XMLActionClient::bind( const TQString &name, const XMLActionScript &s )
{
//    kdWarning() << "Runner:bind called, name=" << name << " type=" << s.type
//		<< " text=" << s.text << " src=" << s.src << endl;

    scripts[name] = s;
    return true;
}

bool XMLActionClient::bind( TDEAction *act, const XMLActionScript &s )
{
    if ( !act )
	return false;

//    kdWarning() << "Runner:bind called, action=" << act->name() << " type=" << s.type
//		<< " text=" << s.text << " src=" << s.src << endl;

    connect( act, TQT_SIGNAL( activated() ), this, TQT_SLOT( action_activated() ) );
    return bind( act->name(), s );
}

bool XMLActionClient::run( const TQString &name )
{
    if ( scripts.contains( name ) )
	return run( scripts[name] );
    else
	return false;
}

bool XMLActionClient::run( const XMLActionScript &s )
{
//    kdWarning() << "Client:run called, type=" << s.type << " text=" << s.text << " src=" << s.src << endl;

    if ( actrun )
	return actrun->run( this, s );
    else
	return false;
}

void XMLActionClient::action_activated()
{
    const TQObject *sender = TQObject::sender();
    if ( !sender )
	return;

    run( sender->name() );
}

//
// SAX Document Handler
//

XMLActionHandler::XMLActionHandler( XMLActionClient *client )
    : TQXmlDefaultHandler(), actclient( client )
{
}

bool XMLActionHandler::characters( const TQString &chars )
{
    cdata = cdata + chars;
    return true;
}


bool XMLActionHandler::startElement( const TQString &, const TQString &, const TQString &qn,
				     const TQXmlAttributes &attrs )
{
    cdata = TQString::null;

    if ( qn == tag_script ) {
	ad.script.type = attrs.value( attr_type );
	ad.script.src = attrs.value( attr_src );
    }
    else if ( qn == tag_group ) {
	TQString ex = attrs.value( attr_exclusive );
	if ( ex == TQString("true") )
	    ad.exclusive = true;
    }
    else if ( qn == tag_action )
	inAction = true;

    return true;
}

bool XMLActionHandler::endElement( const TQString &, const TQString &, const TQString &qn )
{
    if ( qn == tag_action ) {
	defineAction();
	inAction = false;
    }
    else if ( qn == tag_type ) {
	ad.type = cdata;
	cdata = TQString::null;
    }
    else if ( qn == tag_label ) {
	ad.text = cdata;
	cdata = TQString::null;
    }
    else if ( qn == tag_text ) {
	// Nothing
    }
    else if ( qn == tag_icons ) {
	ad.icons = cdata;
	cdata = TQString::null;
    }
    else if ( qn == tag_shortcut ) {
	ad.keys = cdata;
	cdata = TQString::null;
    }
    else if ( qn == tag_name ) {
	ad.name = cdata.latin1();
	cdata = TQString::null;
    }
    else if ( qn == tag_group ) {
	ad.group = cdata.latin1();
	cdata = TQString::null;
    }
    else if ( qn == tag_whatsthis ) {
	ad.whatsthis = cdata;
	cdata = TQString::null;
    }
    else if ( qn == tag_statustext ) {
	ad.status = cdata;
	cdata = TQString::null;
    }
    else if ( qn == tag_script ) {
	ad.script.text = cdata;
	cdata = TQString::null;

	if ( !inAction && ad.script.isValid() )
	    actclient->run( ad.script );
    }
    else if ( qn == tag_item ) {
	ad.items += cdata;
	cdata = TQString::null;
    }

    return true;
}

void XMLActionHandler::defineAction()
{
    if ( ad.name.isEmpty() ) {
	kdWarning() << "Attempt to create a TDEAction without setting a name" << endl;
	return;
    }

    if ( ad.text.isEmpty() )
	ad.text = ad.name;

    TDEAction *act = createAction( actclient->actionCollection() );
    if ( act && ad.script.isValid() )
	actclient->bind( act, ad.script );

    ad.clear();
    cdata = TQString::null;
}

TDEAction *XMLActionHandler::createAction( TDEActionCollection *parent )
{
//    kdDebug(80001) << "Creating Action, type is " << type << endl;
//    kdDebug(80001) << "text=" << text << ", icons=" << icons << endl;
//    kdDebug(80001) << "keys=" << keys << ", name=" << name << endl;

    if ( !parent ) {
	kdWarning() << "Create action called but no parent set" << endl;
	return 0;
    }

    TDEAction *act=0;

    if ( ad.type.isEmpty() || (ad.type == "TDEAction") ) {
	act = new TDEAction( ad.text, ad.icons, ad.keys, 0, 0, parent, ad.name.latin1() );
    }
    else if ( ad.type == "TDEToggleAction" ) {
	act = new TDEToggleAction( ad.text, ad.icons, ad.keys, 0, 0, parent, ad.name.latin1() );
    }
    else if ( ad.type == "TDERadioAction" ) {
	TDERadioAction *ra = new TDERadioAction( ad.text, ad.icons, ad.keys, 0, 0, parent, ad.name.latin1() );
	if ( ad.exclusive )
	    ra->setExclusiveGroup( ad.group );

	act = ra;
    }
    else if ( ad.type == "KStdAction" ) {
	for ( int i = KStdAction::ActionNone ; i < KStdAction::ConfigureNotifications ; i++ ) {
	    if ( KStdAction::stdName(static_cast<KStdAction::StdAction>(i)) == ad.name )
		act = KStdAction::create( (KStdAction::StdAction)i, 0, 0, parent );
	}
    }
    else if ( ad.type == "TDEListAction" ) {
	TDEListAction *la = new TDEListAction( ad.text, ad.icons, ad.keys, 0, 0, parent, ad.name.latin1() );
	la->setItems( ad.items );
	ad.items.clear();
	act = la;
    }
    else if ( ad.type == "TDEActionMenu" ) {
	TDEActionMenu *am = new TDEActionMenu( ad.text, ad.icons, parent, ad.name.latin1() );

	for ( TQStringList::Iterator it = ad.items.begin() ; it != ad.items.end() ; ++it ) {
	    TDEAction *a = parent->action( (*it).latin1() );
	    if ( a )
		am->insert( a );
	}
	ad.items.clear();
	act = am;
    }
    else {
	kdWarning() << "Unknown ActionType " << ad.type << endl;
	return 0;
    }

    if ( !act ) {
	kdWarning() << "Unable to create action" << endl;
	return act;
    }

    if ( !ad.group.isEmpty() )
	act->setGroup( ad.group );

    act->setStatusText( ad.status );
    act->setWhatsThis( ad.whatsthis );

    TQObject::connect( actclient, TQT_SIGNAL( destroyed() ), act, TQT_SLOT( deleteLater() ) );

    return act;
}

} // namespace KJSEmbed
