/*
    This file is part of KMail.

    Copyright (c) 2005 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
#ifndef RECIPIENTSPICKER_H
#define RECIPIENTSPICKER_H

#include <config.h> // for TDEPIM_NEW_DISTRLISTS

#include "recipientseditor.h"

#include <tdelistview.h>
#include <tdelistviewsearchline.h>
#include <tdeabc/addressee.h>
#include <tdeabc/stdaddressbook.h>

#include <tqwidget.h>
#include <tqdialog.h>
#include <tqtooltip.h>

class TQComboBox;

#ifdef TDEPIM_NEW_DISTRLISTS
#include <libtdepim/distributionlist.h>
#else
namespace TDEABC {
class DistributionList;
class DistributionListManager;
}
#endif

namespace KPIM {
class  LDAPSearchDialog;
}

class RecipientItem
{
  public:
    typedef TQValueList<RecipientItem *> List;

#ifdef TDEPIM_NEW_DISTRLISTS
    RecipientItem( TDEABC::AddressBook *ab );
    void setDistributionList( KPIM::DistributionList& );
    KPIM::DistributionList& distributionList();
#else
    RecipientItem();
    void setDistributionList( TDEABC::DistributionList * );
    TDEABC::DistributionList * distributionList();
#endif
    void setAddressee( const TDEABC::Addressee &, const TQString &email );

    void setRecipientType( const TQString &type );
    TQString recipientType() const;

    TQString recipient() const;

    TQPixmap icon() const;
    TQString name() const;
    TQString email() const;

    TQString key() const { return mKey; }

    TQString tooltip() const;

  private:
#ifdef TDEPIM_NEW_DISTRLISTS
    TQString createTooltip( KPIM::DistributionList & ) const;
#else
    TQString createTooltip( TDEABC::DistributionList * ) const;
#endif

    TDEABC::Addressee mAddressee;
    TQString mName;
    TQString mEmail;
    TQString mRecipient;
#ifdef TDEPIM_NEW_DISTRLISTS
    KPIM::DistributionList mDistributionList;
    TDEABC::AddressBook *mAddressBook;
#else
    TDEABC::DistributionList *mDistributionList;
#endif
    TQString mType;
    TQString mTooltip;
    
    TQPixmap mIcon;

    TQString mKey;
};

class RecipientViewItem : public TDEListViewItem
{
  public:
    RecipientViewItem( RecipientItem *, TDEListView * );

    RecipientItem *recipientItem() const;

  private:
    RecipientItem *mRecipientItem;
};

class RecipientsListToolTip : public TQToolTip
{
  public:
    RecipientsListToolTip( TQWidget *parent, TDEListView * );

  protected:
    void maybeTip( const TQPoint &pos );

  private:
    TDEListView *mListView;
};

class RecipientsCollection
{
  public:
    RecipientsCollection( const TQString & );
    ~RecipientsCollection();

    void setReferenceContainer( bool );
    bool isReferenceContainer() const;

    void setTitle( const TQString & );
    TQString title() const;

    void addItem( RecipientItem * );

    RecipientItem::List items() const;

    bool hasEquivalentItem( RecipientItem * ) const;
    RecipientItem * getEquivalentItem( RecipientItem *) const;

    void clear();

    void deleteAll();

    TQString id() const;

  private:
    // flag to indicate if this collection contains just references
    // or should manage memory (de)allocation as well.
    bool mIsReferenceContainer;
    TQString mId;
    TQString mTitle;
    TQMap<TQString, RecipientItem *> mKeyMap;
};

class SearchLine : public TDEListViewSearchLine
{
    Q_OBJECT
  
  public:
    SearchLine( TQWidget *parent, TDEListView *listView );

  signals:
    void downPressed();

  protected:
    void keyPressEvent( TQKeyEvent * );
};

using namespace TDEABC;

class RecipientsPicker : public TQDialog
{
    Q_OBJECT
  
  public:
    RecipientsPicker( TQWidget *parent );
    ~RecipientsPicker();

    void setRecipients( const Recipient::List & );
    void updateRecipient( const Recipient & );

    void setDefaultType( Recipient::Type );

  signals:
    void pickedRecipient( const Recipient & );

  protected:
    void initCollections();
    void insertDistributionLists();
    void insertRecentAddresses();
    void insertCollection( RecipientsCollection *coll );

    void keyPressEvent( TQKeyEvent *ev );

    void readConfig();
    void writeConfig();

    void pick( Recipient::Type );

    void setDefaultButton( TQPushButton *button );

    void rebuildAllRecipientsList();

  protected slots:
    void updateList();
    void slotToClicked();
    void slotCcClicked();
    void slotBccClicked();
    void slotPicked( TQListViewItem * );
    void slotPicked();
    void setFocusList();
    void resetSearch();
    void insertAddressBook( AddressBook * );
    void slotSearchLDAP();
    void ldapSearchResult();
  private:
    TDEABC::StdAddressBook *mAddressBook;

    TQComboBox *mCollectionCombo;
    TDEListView *mRecipientList;
    TDEListViewSearchLine *mSearchLine;

    TQPushButton *mToButton;
    TQPushButton *mCcButton;
    TQPushButton *mBccButton;

    TQPushButton *mSearchLDAPButton;
    KPIM::LDAPSearchDialog *mLdapSearchDialog;

    TQMap<int,RecipientsCollection *> mCollectionMap;
    RecipientsCollection *mAllRecipients;
    RecipientsCollection *mDistributionLists;
    RecipientsCollection *mSelectedRecipients;

#ifndef TDEPIM_NEW_DISTRLISTS
    TDEABC::DistributionListManager *mDistributionListManager;
#endif

    Recipient::Type mDefaultType;
};

#endif
