/*
    This file is part of KAddressbook.
    Copyright (c) 2003 Tobias Koenig <tokoe@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

    As a special exception, permission is given to link this program
    with any edition of TQt, and distribute the resulting executable,
    without including the source code for TQt in the source distribution.
*/

#ifndef KAB_CORE_H
#define KAB_CORE_H

#include <config.h> // for TDEPIM_NEW_DISTRLISTS

#ifdef TDEPIM_NEW_DISTRLISTS
#include <libtdepim/distributionlist.h>
#endif

#include <tqobject.h>

#include <tdeabc/field.h>
#include <kcommand.h>
#include <kxmlguiclient.h>
#include <kdemacros.h>

namespace TDEABC {
class AddressBook;
class Resource;
}

class TQWidget;

class TDEActionCollection;
class TDEConfig;
class KURL;

namespace KAB {

class SearchManager;

class KDE_EXPORT Core : public TQObject
{
  TQ_OBJECT
  

  public:
    Core( KXMLGUIClient *client, TQObject *parent, const char *name = 0 );

    /**
      Returns a pointer to the StdAddressBook of the application.
     */
    virtual TDEABC::AddressBook *addressBook() const = 0;

    /**
      Returns a pointer to the TDEConfig object of the application.
     */
    virtual TDEConfig *config() const = 0;

    /**
      Returns a pointer to the global TDEActionCollection object. So
      other classes can register their actions easily.
     */
    virtual TDEActionCollection *actionCollection() const = 0;

    /**
      Returns a pointer to the gui client.
     */
    virtual KXMLGUIClient *guiClient() const { return mGUIClient; }

    /**
      Returns the current sort field.
     */
    virtual TDEABC::Field *currentSortField() const = 0;

    /**
      Returns the uid list of the currently selected contacts.
     */
    virtual TQStringList selectedUIDs() const = 0;

    /**
      Displays a ResourceSelectDialog and returns the selected
      resource or a null pointer if no resource was selected by
      the user.
     */
    virtual TDEABC::Resource *requestResource( TQWidget *parent ) = 0;

    /**
      Returns the parent widget.
     */
    virtual TQWidget *widget() const = 0;

    /**
      Deletes given contacts from the address book.

      @param uids The uids of the contacts, which shall be deleted.
     */
    virtual void deleteContacts( const TQStringList &uids ) = 0;

    /**
      Deletes given contacts from the address book.

      @param uids The uids of the contacts, which shall be deleted.
     */
    virtual void deleteDistributionLists( const TQStringList &uids ) = 0;

#ifdef TDEPIM_NEW_DISTRLISTS
    /**
      Returns all the distribution lists.
     */
    virtual KPIM::DistributionList::List distributionLists() const = 0;


    /**
      Returns the name of all the distribution lists.
     */
    virtual TQStringList distributionListNames() const = 0;

    /**
      sets the distribution list to display. If null, the regular
      address book is to be displayed.  
     */
    virtual void setSelectedDistributionList( const TQString &name ) = 0;
#endif

    //// This class isn't part of interfaces/, so this method here isn't really useful
    virtual SearchManager *searchManager() const = 0;

    virtual KCommandHistory *commandHistory() const = 0;

    signals:
    /**
      Forwarded from SearchManager
      After it is emitted, distributionListNames() might have a different result.
     */
    void contactsUpdated();

  public slots:
    /**
      Is called whenever a contact is selected in the view.
     */
    virtual void setContactSelected( const TQString &uid ) = 0;

    /**
      DCOP method: Adds the given email address to address book.
     */
    virtual void addEmail( const TQString& addr ) = 0;

    /**
      DCOP method: Imports the vCard, located at the given url.
     */
    virtual void importVCard( const KURL& url ) = 0;

    /**
      DCOP method: Imports the given vCard.
     */
    virtual void importVCardFromData( const TQString& vCard ) = 0;

    /**
      DCOP method: Opens contact editor to input a new contact.
     */
    virtual void newContact() = 0;

    /**
      DCOP method: Opens distribution list editor to input a new distribution list.
     */
    virtual void newDistributionList() = 0;

    /**
      DCOP method: Returns the name of the contact, that matches the given
                   phone number.
     */
    virtual TQString getNameByPhone( const TQString& phone ) = 0;

    /**
      Shows an edit dialog for the given uid.
     */
    virtual void editContact( const TQString &uid = TQString() ) = 0;

    /**
      Shows an edit dialog for the given distribution list 
    */
    virtual void editDistributionList( const TQString &name ) = 0;

  private:
    KXMLGUIClient *mGUIClient;
};

}

#endif
