/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
*/

/*
  Copyright (C) 2005 Francois Chazal <neptune3k@free.fr>
  Copyright (C) 2006-2007 Eike Hein <hein@kde.org>
*/


#include "tab_bar.h"
#include "tab_bar.moc"
#include "settings.h"

#include <tqtooltip.h>
#include <tqwhatsthis.h>

#include <tdelocale.h>
#include <tdepopupmenu.h>


TabBar::TabBar(TQWidget* parent, const char* name, bool translucency, const TQString & skin)
    : TranslucentWidget(parent, name, translucency)
{
    loadSkin(skin);

    connect(plus_button, TQT_SIGNAL(clicked()), this, TQT_SIGNAL(addItem()));
    connect(minus_button, TQT_SIGNAL(clicked()), this, TQT_SIGNAL(removeItem()));
    connect(tabs_widget, TQT_SIGNAL(itemSelected(int)), this, TQT_SIGNAL(itemSelected(int)));
    connect(tabs_widget, TQT_SIGNAL(addItem()), this, TQT_SIGNAL(addItem()));
    connect(this, TQT_SIGNAL(updateBackground()), this, TQT_SLOT(slotUpdateBackground()));
}

TabBar::~TabBar()
{
    delete tabs_widget;
    delete plus_button;
    delete minus_button;
}

int TabBar::pressedPosition()
{
    return tabs_widget->pressedPosition();
}


void TabBar::resetPressedPosition()
{
    tabs_widget->resetPressedPosition();
}

void TabBar::addItem(int session_id)
{
    tabs_widget->addItem(session_id);
    tabs_widget->selectItem(session_id);
}

int TabBar::removeItem(int session_id)
{
    return tabs_widget->removeItem(session_id);
}

const TQString TabBar::itemName(int session_id)
{
    return tabs_widget->itemName(session_id);
}

void TabBar::renameItem(int session_id, const TQString& name)
{
    tabs_widget->renameItem(session_id, name);
}

void TabBar::interactiveRename()
{
    tabs_widget->interactiveRename();
}

int TabBar::tabPositionForSessionId(int session_id)
{
    return tabs_widget->tabPositionForSessionId(session_id);
}

int TabBar::sessionIdForTabPosition(int position)
{
    return tabs_widget->sessionIdForTabPosition(position);
}

void TabBar::selectItem(int session_id)
{
    tabs_widget->selectItem(session_id);
}

void TabBar::selectPosition(int position)
{
    tabs_widget->selectPosition(position);
}

void TabBar::slotSelectNextItem()
{
    tabs_widget->selectNextItem();
}

void TabBar::slotSelectPreviousItem()
{
    tabs_widget->selectPreviousItem();
}

void TabBar::slotMoveItemLeft()
{
    tabs_widget->moveItemLeft();
}

void TabBar::slotMoveItemRight()
{
    tabs_widget->moveItemRight();
}

void TabBar::paintEvent(TQPaintEvent*)
{
    TQPainter painter(this);

    erase();

    if (!useTranslucency())
        painter.fillRect(0, 0, width(), height(), Settings::skinbgcolor());

    painter.drawPixmap(0, 0, left_corner);
    painter.drawPixmap(width() - right_corner.width(), 0, right_corner);

    painter.drawTiledPixmap(left_corner.width(), 0, width() -
            left_corner.width() -right_corner.width(), height(), back_image);

    painter.end();
}

void TabBar::resizeEvent(TQResizeEvent*)
{
    plus_button->move(plus_position.x(), plus_position.y());
    minus_button->move(width() - minus_position.x(), minus_position.y());

    tabs_widget->move(tabs_position.x(), tabs_position.y());
    tabs_widget->resize(width() - 2 * tabs_position.x(), tabs_widget->height());
    tabs_widget->refreshBuffer();
}

void TabBar::loadSkin(const TQString& skin)
{
    tabs_widget = new TabbedWidget(this, "Tabbed widget", useTranslucency());
    TQWhatsThis::add(tabs_widget, i18n("The tab bar allows you to switch between sessions."));
    connect(this, TQT_SIGNAL(updateBackground()), tabs_widget, TQT_SLOT(slotUpdateBackground()));

    plus_button = new ImageButton(this, "Plus button", useTranslucency());
    plus_button->setDelayedPopup(true);
    TQToolTip::add(plus_button, i18n("New Session"));
    TQWhatsThis::add(plus_button, i18n("Adds a new session. Press and hold to select session type from menu."));
    connect(this, TQT_SIGNAL(updateBackground()), plus_button, TQT_SLOT(slotUpdateBackground()));

    minus_button = new ImageButton(this, "Minus button", useTranslucency());
    TQToolTip::add(minus_button, i18n("Close Session"));
    TQWhatsThis::add(minus_button, i18n("Closes the active session."));
    connect(this, TQT_SIGNAL(updateBackground()), minus_button, TQT_SLOT(slotUpdateBackground()));

    setPixmaps(skin);

    resize(width(), back_image.height());
}

void TabBar::reloadSkin(const TQString& skin)
{
    setPixmaps(skin);

    resize(width(), back_image.height());

    plus_button->move(plus_position.x(), plus_position.y());
    minus_button->move(width() - minus_position.x(), minus_position.y());

    tabs_widget->move(tabs_position.x(), tabs_position.y());
    tabs_widget->resize(width() - 2 * tabs_position.x(), tabs_widget->height());

    minus_button->repaint();
    plus_button->repaint();
    tabs_widget->refreshBuffer();
    repaint();
}

void TabBar::setPixmaps(const TQString& skin)
{
    TQUrl url(locate("appdata", skin + "/tabs.skin"));
    TDEConfig config(url.path());

    config.setGroup("Tabs");

    tabs_position.setX(config.readNumEntry("x", 20));
    tabs_position.setY(config.readNumEntry("y", 0));

    tabs_widget->setFontColor(TQColor(config.readNumEntry("red", 255),
                                config.readNumEntry("green", 255),
                                config.readNumEntry("blue", 255)));

    tabs_widget->setSeparatorPixmap(url.dirPath() + config.readEntry("separator_image", ""));
    tabs_widget->setSelectedPixmap(url.dirPath() + config.readEntry("selected_background", ""));
    tabs_widget->setSelectedLeftPixmap(url.dirPath() + config.readEntry("selected_left_corner", ""));
    tabs_widget->setSelectedRightPixmap(url.dirPath() + config.readEntry("selected_right_corner", ""));
    tabs_widget->setUnselectedPixmap(url.dirPath() + config.readEntry("unselected_background", ""));

    // Load the background pixmaps.
    config.setGroup("Background");

    tabs_widget->setBackgroundPixmap(url.dirPath() + config.readEntry("back_image", ""));

    back_image.load(url.dirPath() + config.readEntry("back_image", ""));
    left_corner.load(url.dirPath() + config.readEntry("left_corner", ""));
    right_corner.load(url.dirPath() + config.readEntry("right_corner", ""));

    // Load the plus button.
    config.setGroup("PlusButton");

    plus_position.setX(config.readNumEntry("x", 2));
    plus_position.setY(config.readNumEntry("y", 2));

    plus_button->setUpPixmap(url.dirPath() + config.readEntry("up_image", ""));
    plus_button->setOverPixmap(url.dirPath() + config.readEntry("over_image", ""));
    plus_button->setDownPixmap(url.dirPath() + config.readEntry("down_image", ""));

    // Load the minus button.
    config.setGroup("MinusButton");

    minus_position.setX(config.readNumEntry("x", 18));
    minus_position.setY(config.readNumEntry("y", 2));

    minus_button->setUpPixmap(url.dirPath() + config.readEntry("up_image", ""));
    minus_button->setOverPixmap(url.dirPath() + config.readEntry("over_image", ""));
    minus_button->setDownPixmap(url.dirPath() + config.readEntry("down_image", ""));
}

void TabBar::setSessionMenu(TDEPopupMenu* menu)
{
    plus_button->setPopupMenu(menu);
}
