/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-06-14
 * Description : A TQImage loader for DImg framework.
 * 
 * Copyright (C) 2005 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Copyright (C) 2006-2007 by Caulier Gilles <caulier dot gilles at gmail dot com> 
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqimage.h>

// Local includes.

#include "ddebug.h"
#include "dimg.h"
#include "dimgloaderobserver.h"
#include "qimageloader.h"

namespace Digikam
{

TQImageLoader::TQImageLoader(DImg* image)
            : DImgLoader(image)
{
}

bool TQImageLoader::load(const TQString& filePath, DImgLoaderObserver *observer)
{
    // Loading is opaque to us. No support for stopping from observer,
    // progress info are only pseudo values
    TQImage image(filePath);

    if (observer)
        observer->progressInfo(m_image, 0.9);

    if (image.isNull())
    {
        DDebug() << "Cannot loading \"" << filePath << "\" using DImg::TQImageLoader!" << endl;
        return false;
    }

    m_hasAlpha    = image.hasAlphaBuffer();
    TQImage target = image.convertDepth(32);
    
    uint w      = target.width();
    uint h      = target.height();
    uchar* data = new uchar[w*h*4];
    uint*  sptr = (uint*)target.bits();
    uchar* dptr = data;
    
    for (uint i = 0 ; i < w*h ; i++)
    {
        dptr[0] = tqBlue(*sptr);
        dptr[1] = tqGreen(*sptr);
        dptr[2] = tqRed(*sptr);
        dptr[3] = tqAlpha(*sptr);
        
        dptr += 4;
        sptr++;
    }

    if (observer)
        observer->progressInfo(m_image, 1.0);

    imageWidth()  = w;
    imageHeight() = h;
    imageData()   = data;

    // We considering that PNG is the most representative format of an image loaded by TQt
    imageSetAttribute("format", "PNG");

    return true;
}

bool TQImageLoader::save(const TQString& filePath, DImgLoaderObserver *observer)
{
    TQVariant qualityAttr = imageGetAttribute("quality");
    int quality = qualityAttr.isValid() ? qualityAttr.toInt() : 90;
    
    if (quality < 0)
        quality = 90;
    if (quality > 100)
        quality = 100;

    TQVariant formatAttr = imageGetAttribute("format");
    TQCString format     = formatAttr.toCString();

    TQImage image = m_image->copyTQImage();

    if (observer)
        observer->progressInfo(m_image, 0.1);

    // Saving is opaque to us. No support for stopping from observer,
    // progress info are only pseudo values
    bool success = image.save(filePath, format.upper(), quality);
    if (observer && success)
        observer->progressInfo(m_image, 1.0);

    imageSetAttribute("format", format.upper());

    return success;
}

bool TQImageLoader::hasAlpha() const
{
    return m_hasAlpha;
}

}  // NameSpace Digikam
