/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2006 by the KFTPGrabber developers
 * Copyright (C) 2003-2006 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
#ifndef KFTPENGINETHREAD_H
#define KFTPENGINETHREAD_H

#include <tqthread.h>
#include <tqmutex.h>
#include <tqvaluelist.h>
#include <tqdict.h>

#include "event.h"
#include "directorylisting.h"
#include "commands.h"
#include "socket.h"

namespace KFTPEngine {

/**
 * This class represents a socket thread. It serves as a command queue to
 * the underlying socket implementation and also as an abstraction layer
 * to support multiple protocols.
 *
 * @author Jernej Kos <kostko@jweb-network.net>
 */
class Thread : public TQThread
{
friend class EventHandler;
friend class Socket;
public:
    Thread();
    ~Thread();
    
    /**
     * Returns the event handler for this thread. Should be used to connect
     * to any signals this thread may emit.
     *
     * @return A pointer to the EventHandler object
     */
    EventHandler *eventHandler() { return m_eventHandler; }
    
    /**
     * Returns the underlying socket object.
     *
     * @return A pointer to the Socket object
     */
    Socket *socket() { return m_socket; }
    
    /**
     * Prepare the apropriate socket for use.
     *
     * @param url The url that should be used to identify the protocol
     */
    void selectSocketForProtocol(const KURL &url);
    
    /**
     * Schedules a wakeup event to be passed on to the underlying socket.
     *
     * @param event The wakeup event to pass on
     */
    void wakeup(WakeupEvent *event);
    
    void abort();
    void connect(const KURL &url);
    void disconnect();
    void list(const KURL &url);
    void scan(const KURL &url);
    void get(const KURL &source, const KURL &destination);
    void put(const KURL &source, const KURL &destination);
    void remove(const KURL &url);
    void rename(const KURL &source, const KURL &destination);
    void chmod(const KURL &url, int mode, bool recursive = false);
    void mkdir(const KURL &url);
    void raw(const TQString &raw);
    void siteToSite(Thread *thread, const KURL &source, const KURL &destination);
protected:
    void run();
    void event(Event::Type type, TQValueList<EventParameter> params);
    
    EventParameter nextCommandParameter();
protected:
    EventHandler *m_eventHandler;
    Socket *m_socket;
    
    TQMutex m_eventMutex;
    TQMutex m_wakeupMutex;
    TQMutex m_paramsMutex;
    TQMutex m_queueMutex;
    
    TQDict<Socket> m_protocolMap;
    TQValueList<Commands::Type> m_commandQueue;
    TQValueList<EventParameter> m_commandParams;
    WakeupEvent *m_wakeupEvent;
    
    bool m_abortLoop;
    bool m_wakeUpPending;
};

}

#endif
