/*
    securestream.h - Kopete Groupwise Protocol
    Combines a ByteStream with TLS and SASL
  
    Copyright (c) 2004      SUSE Linux AG	 	 http://www.suse.com
    
    Based on Iris, Copyright (C) 2003  Justin Karneges
    
    Kopete (c) 2002-2004 by the Kopete developers <kopete-devel@kde.org>
 
    *************************************************************************
    *                                                                       *
    * This library is free software; you can redistribute it and/or         *
    * modify it under the terms of the GNU Lesser General Public            *
    * License as published by the Free Software Foundation; either          *
    * version 2 of the License, or (at your option) any later version.      *
    *                                                                       *
    *************************************************************************
*/

#ifndef SECURESTREAM_H
#define SECURESTREAM_H

#include <tqca.h>
#include "tlshandler.h"
#include "bytestream.h"

#define USE_TLSHANDLER

#ifdef USE_TLSHANDLER
	class TLSHandler;
#endif

class SecureStream : public ByteStream
{
	TQ_OBJECT
  
public:
	enum Error { ErrTLS = ErrCustom, ErrSASL };
	SecureStream(ByteStream *s);
	~SecureStream();

	void startTLSClient(TQCA::TLS *t, const TQByteArray &spare=TQByteArray());
	void startTLSServer(TQCA::TLS *t, const TQByteArray &spare=TQByteArray());
	void setLayerSASL(TQCA::SASL *s, const TQByteArray &spare=TQByteArray());
#ifdef USE_TLSHANDLER
	void startTLSClient(TLSHandler *t, const TQString &server, const TQByteArray &spare=TQByteArray());
#endif

	void closeTLS();
	int errorCode() const;

	// reimplemented
	bool isOpen() const;
	void write(const TQByteArray &);
	int bytesToWrite() const;

signals:
	void tlsHandshaken();
	void tlsClosed();

private slots:
	void bs_readyRead();
	void bs_bytesWritten(int);

	void layer_tlsHandshaken();
	void layer_tlsClosed(const TQByteArray &);
	void layer_readyRead(const TQByteArray &);
	void layer_needWrite(const TQByteArray &);
	void layer_error(int);

private:
	void linkLayer(TQObject *);
	int calcPrebytes() const;
	void insertData(const TQByteArray &a);
	void writeRawData(const TQByteArray &a);
	void incomingData(const TQByteArray &a);

	class Private;
	Private *d;
};

class LayerTracker
{
public:
	struct Item
	{
		int plain;
		int encoded;
	};
USE_TLSHANDLER
	LayerTracker();

	void reset();
	void addPlain(int plain);
	void specifyEncoded(int encoded, int plain);
	int finished(int encoded);

	int p;
	TQValueList<Item> list;
};


class SecureLayer : public TQObject
{
	TQ_OBJECT
  
public:
	SecureLayer(TQCA::TLS *t);
	SecureLayer(TQCA::SASL *s);
#ifdef USE_TLSHANDLER
	SecureLayer(TLSHandler *t);
#endif
	void init();
	void write(const TQByteArray &a);
	void writeIncoming(const TQByteArray &a);
	int finished(int plain);

	enum { TLS, SASL, TLSH };
	int type;
	union {
		TQCA::TLS *tls;
		TQCA::SASL *sasl;
#ifdef USE_TLSHANDLER
		TLSHandler *tlsHandler;
#endif
	} p;
	LayerTracker layer;
	bool tls_done;
	int prebytes;

signals:
        void tlsHandshaken();
        void tlsClosed(const TQByteArray &);
        void readyRead(const TQByteArray &);
        void needWrite(const TQByteArray &);
        void error(int);

private slots:
        void tls_handshaken();
        void tls_readyRead();
        void tls_readyReadOutgoing(int plainBytes);
        void tls_closed();
        void tls_error(int x);
        void sasl_readyRead();
        void sasl_readyReadOutgoing(int plainBytes);
        void sasl_error(int x);
#ifdef USE_TLSHANDLER
	void tlsHandler_success();
	void tlsHandler_fail();
	void tlsHandler_closed();
	void tlsHandler_readyRead(const TQByteArray &a);
	void tlsHandler_readyReadOutgoing(const TQByteArray &a, int plainBytes);
#endif
	
};

#endif
