;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;February, 1998
;;;***************************************************************

(in-package "ACL2")

(include-book "float")

(defun trunc (x n)
  (* (sgn x) (fl (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n))))

(defthm pos*
    (implies (and (rationalp x)
		  (rationalp y)
		  (> x 0)
		  (> y 0))
	     (> (* x y) 0))
  :rule-classes ())


(defthm trunc-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (> (trunc x n) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo sig fl-weakly-monotonic)
		  :use ((:instance sig-lower-bound)
			(:instance pos* 
				   (x (fl (* (sig x) (expt 2 (1- n))))) 
				   (y (expt 2 (- (1+ (expo x)) n))))
			(:instance sgn+1)
			(:instance expo-monotone (x 1) (y (1- n)))
			(:instance n<=fl (x (sig x)) (n 1))))))

(defthm trunc-neg
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0))
	     (< (trunc x n) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo sig fl-weakly-monotonic)
		  :use ((:instance sig-lower-bound)
			(:instance pos* 
				   (x (fl (* (sig x) (expt 2 (1- n))))) 
				   (y (expt 2 (- (1+ (expo x)) n))))
			(:instance sgn-1)
			(:instance expo-monotone (x 1) (y (1- n)))
			(:instance n<=fl (x (sig x)) (n 1))))))

(defthm trunc-zero
    (implies (and (integerp n)
		  (> n 0))
	     (= (trunc 0 n) 0))
  :rule-classes ())

(defthm sgn-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (sgn (trunc x n))
		    (sgn x)))
  :hints (("Goal" :in-theory (disable trunc)
		  :use ((:instance trunc-pos)
			(:instance trunc-neg)
			(:instance trunc-zero)))))

(defthm abs-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (abs (trunc x n)) (* (fl (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n)))))
  :hints (("Goal" :in-theory (disable expo sig fl-weakly-monotonic)
		  :use ((:instance sig-lower-bound)
			(:instance pos* 
				   (x (fl (* (sig x) (expt 2 (1- n))))) 
				   (y (expt 2 (- (1+ (expo x)) n))))
			(:instance sgn-1)
			(:instance sgn+1)
			(:instance expo-monotone (x 1) (y (1- n)))
			(:instance n<=fl (x (sig x)) (n 1))))))

(defthm trunc-upper-bound
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (<= (abs (trunc x n)) (abs x)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a15 abs expo sig trunc)
		  :use ((:instance fl-def (x (* (expt 2 (1- n)) (sig x))))
			(:instance sig-lower-bound)
			(:instance *-weakly-monotonic
				   (x (expt 2 (- (1+ (expo x)) n)))
				   (y (fl (* (sig x) (expt 2 (1- n)))))
				   (y+ (* (sig x) (expt 2 (1- n)))))
			(:instance fp-abs)
			(:instance expo+ (m (1- n)) (n (- (1+ (expo x)) n)))))))

(defthm rationalp-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (rationalp (trunc x n))))

(defthm trunc-0-0
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (iff (= (trunc x n) 0)
		  (= x 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc)
		  :use ((:instance trunc-pos)
			(:instance trunc-neg)
			(:instance trunc-zero)))))

(defthm trunc-upper-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (<= (trunc x n) x))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable abs-trunc trunc)
		  :use ((:instance trunc-upper-bound)
			(:instance trunc-pos)
			(:instance trunc-0-0)))))

(defthm expo-trunc-upper-bound
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (<= (expo (trunc x n)) (expo x)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo sig trunc)
		  :use ((:instance trunc-upper-bound)
			(:instance trunc-0-0)
			(:instance expo-monotone (x (trunc x n)) (y x))))))

(defthm expo-trunc-lower-bound
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (>= (abs (trunc x n)) (expt 2 (expo x))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo sig trunc)
		  :use ((:instance sig-lower-bound)
			(:instance *-weakly-monotonic
				   (y (expt 2 (1- n)))
				   (y+ (fl (* (sig x) (expt 2 (1- n)))))
				   (x (expt 2 (- (1+ (expo x)) n))))
			(:instance expo+ (m (1- n)) (n (- (1+ (expo x)) n)))
			(:instance fl-monotone (x (expt 2 (1- n))) (y (* (expt 2 (1- n)) (sig x))))))))

(defthm expo-trunc
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (expo (trunc x n)) (expo x)))
  :hints (("Goal" :in-theory (disable abs-trunc expo sig trunc)
		  :use ((:instance expo-trunc-lower-bound)
			(:instance expo-trunc-upper-bound)
			(:instance trunc-0-0)
			(:instance expo-upper-bound (x (trunc x n)))
			(:instance expt-strong-monotone (n (expo x)) (m (1+ (expo (trunc x n)))))))))

(defthm trunc-lower-1-1
    (implies (and (rationalp x)
		  (integerp n))
	     (= (abs x) (* (* (sig x) (expt 2 (1- n))) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable abs expo sig)
		  :use ((:instance fp-abs)))))

(defthm trunc-lower-1-2
    (implies (and (rationalp u)
		  (rationalp v)
		  (rationalp r)
		  (> r 0)
		  (< u (1+ v)))
	     (< (* r u) (* r (1+ v))))
  :rule-classes ())

(defthm trunc-lower-1-3
    (implies (and (rationalp u)
		  (rationalp v)
		  (rationalp r)
		  (> r 0)
		  (< u (1+ v)))
	     (< (* r u) (+ r (* r v))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable *-strongly-monotonic)
		  :use ((:instance trunc-lower-1-2)))))

(defthm trunc-lower-1
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (> (abs (trunc x n)) (- (abs x) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable abs expo sig trunc a15)
		  :use ((:instance trunc-lower-1-1)
			(:instance trunc-lower-1-3
				   (u (* (sig x) (expt 2 (1- n))))
				   (v (fl (* (sig x) (expt 2 (1- n)))))
				   (r (expt 2 (- (1+ (expo x)) n))))
			(:instance fl-def (x (* (expt 2 (1- n)) (sig x))))))))

(defthm trunc-lower-2-1
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (<= (expt 2 (- (1+ (expo x)) n)) (* (abs x) (expt 2 (- 1 n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable abs expo sig a15)
		  :use ((:instance expo-lower-bound)
			(:instance expo+ (m (expo x)) (n (- 1 n)))))))

(defthm trunc-lower-2
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (> (abs (trunc x n)) (* (abs x) (- 1 (expt 2 (- 1 n))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable abs expo sig trunc a15)
		  :use ((:instance trunc-lower-1)
			(:instance trunc-lower-2-1)))))

(defthm trunc-lower-pos
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (> (trunc x n) (* x (- 1 (expt 2 (- 1 n))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable abs-trunc abs-pos expo sig trunc a15)
		  :use ((:instance trunc-lower-2)
			(:instance trunc-pos)))))

(defthm trunc-lower-3
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (abs (trunc x n)) (* (abs x) (- 1 (expt 2 (- 1 n))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable abs expo sig trunc a15)
		  :use ((:instance trunc-lower-1)
			(:instance trunc-0-0)
			(:instance trunc-lower-2-1)))))

(defthm trunc-lower-4-1
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (abs (trunc x n)) (- (abs x) (* (abs x) (expt 2 (- 1 n))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc abs-trunc)
		  :use ((:instance trunc-lower-3)))))

(defthm trunc-lower-4-2
    (implies (and (rationalp x)
		  (< x 0)
		  (integerp n)
		  (> n 0))
	     (>= (trunc x n) x))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc abs-trunc)
		  :use ((:instance trunc-upper-bound)
			(:instance trunc-neg)))))

(defthm trunc-lower-4
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (>= (trunc x n) (- x (* (abs x) (expt 2 (- 1 n))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc abs-trunc)
		  :use ((:instance trunc-lower-4-1)
			(:instance trunc-lower-4-2)
			(:instance trunc-pos)
			(:instance trunc-neg)
			(:instance trunc-0-0)))))

(defthm trunc-diff-1
    (implies (and (rationalp x)
		  (rationalp y)
		  (<= (abs y) (abs x))
		  (iff (< x 0) (< y 0)))
	     (= (abs (- x y)) (- (abs x) (abs y))))
  :rule-classes ())

(defthm trunc-diff
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (abs (- x (trunc x n))) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc abs expo abs-trunc)
		  :use ((:instance trunc-diff-1 (y (trunc x n)))
			(:instance trunc-neg)
			(:instance trunc-pos)
			(:instance trunc-0-0)
			(:instance trunc-upper-bound)
			(:instance trunc-lower-1)))))

(defthm trunc-diff-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (< (- x (trunc x n)) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc expo abs-trunc)
		  :use ((:instance trunc-diff)
			(:instance trunc-pos)
			(:instance trunc-upper-bound)))))


(defthm trunc-diff-expo-1
    (implies (and (rationalp x)
		  (not (= x (trunc x n)))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- x (trunc x n))) (- (expo x) n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc abs expo abs-trunc)
		  :use ((:instance trunc-diff)
			(:instance expo-lower-bound (x (- x (trunc x n))))
			(:instance expt-strong-monotone 
				   (n (expo (- x (trunc x n))))
				   (m (- (1+ (expo x)) n)))))))

(defthm trunc-rewrite
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (equal (trunc x n)
		    (* (sgn x) 
		       (fl (* (expt 2 (- (1- n) (expo x))) (abs x))) 
		       (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable trunc))

(defthm trunc-exactp-1
    (implies (and (rationalp x)
		  (integerp n))
	     (= x (* (sgn x) (* (expt 2 (- (1- n) (expo x))) (abs x)) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance expo+ (n (- (1- n) (expo x))) (m (- (1+ (expo x)) n)))))))

(defthm trunc-exactp-2
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp z)
		  (not (= x 0))
		  (not (= z 0)))
	     (iff (= (* x y z) (* x (fl y) z))
		  (integerp y)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable fl-int fl-int-2 fl)
		  :use ((:instance fl-int-2 (x y))
			(:instance *cancell (x (fl y)) (z (* x z)))))))

(defthm trunc-exactp-3
    (implies (integerp x) (integerp (- x)))
  :rule-classes ())

(defthm trunc-exactp-4
    (implies (rationalp x)
	     (equal (- (- x)) x)))

(defthm trunc-exactp-5
    (implies (rationalp x)
	     (iff (integerp x) (integerp (- x))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a2)
		  :use ((:instance trunc-exactp-3)
			(:instance trunc-exactp-3 (x (- x)))))))

(defthm trunc-exactp-6
    (implies (and (rationalp x)
		  (integerp n))
	     (iff (exactp x n)
		  (integerp (* (abs x) (expt 2 (- (1- n) (expo x)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance trunc-exactp-5 (x (* x (expt 2 (- (1- n) (expo x))))))))))

(defthm trunc-exactp-a
    (implies (and (rationalp x)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (trunc x n))
		  (exactp x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance trunc-exactp-1)
			(:instance trunc-exactp-6)
			(:instance trunc-exactp-2
				   (x (sgn x))
				   (y (* (expt 2 (- (1- n) (expo x))) (abs x)))
				   (z (expt 2 (- (1+ (expo x)) n))))))))

(defthm trunc-diff-expo
    (implies (and (rationalp x)
		  (not (exactp x n))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- x (trunc x n))) (- (expo x) n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable trunc abs exactp2 expo abs-trunc)
		  :use ((:instance trunc-diff-expo-1)
			(:instance trunc-exactp-a)))))

(defthm trunc-exactp-b-1    
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (> n 0))
	     (integerp (* (* (sgn x) (fl y) (expt 2 (- (1- n) (expo x)))) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance integerp-x-y 
				   (x (sgn x))
				   (y (fl (* (expt 2 (- (1- n) (expo x))) (abs x)))))
			(:instance expo+ (n (- (1- n) (expo x))) (m (- (1+ (expo x)) n)))))))

(defthm trunc-exactp-b-2
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (integerp (* (trunc x n) (expt 2 (- (1- n) (expo x))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo sgn)
		  :use ((:instance trunc-exactp-b-1 (y (* (expt 2 (- (1- n) (expo x))) (abs x))))))))

(defthm trunc-exactp-b
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (exactp (trunc x n) n))
  :hints (("Goal" :in-theory (disable expo trunc-rewrite)
		  :use ((:instance trunc-exactp-b-2)))))

(defthm trunc-exactp-c
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (<= a x))
	     (<= a (trunc x n)))
  :hints (("Goal" :in-theory (disable expo exactp2 abs-trunc trunc-rewrite)
		  :use ((:instance trunc-exactp-b)
			(:instance trunc-exactp-a)
			(:instance fp+1 (x (trunc x n)) (y a))
			(:instance trunc-lower-1)
			(:instance trunc-pos)))))

(defthm trunc-monotone
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (> x 0)
		  (> y 0)
		  (> n 0)
		  (<= x y))
	     (<= (trunc x n) (trunc y n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo exactp2 abs-trunc trunc-rewrite)
		  :use ((:instance trunc-exactp-b)
			(:instance trunc-upper-pos)
			(:instance trunc-exactp-c (x y) (a (trunc x n)))))))

(defthm trunc-pos-rewrite
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (equal (trunc x n)
		    (* (fl (* (expt 2 (- (1- n) (expo x))) x))
		       (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable trunc-rewrite))

(defthm trunc-trunc-1
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (* (expt 2 (- (1- m) (expo x)))
			  (* (fl (* (expt 2 (- (1- n) (expo x))) x))
			     (expt 2 (- (1+ (expo x)) n)))))
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo expo-trunc)
		  :use ((:instance trunc-pos)
			(:instance expo-trunc)
			(:instance expo-trunc (x (trunc x n)) (n m))))))

(defthm trunc-trunc-2
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (* (fl (* (expt 2 (- (1- n) (expo x))) x)) (expt 2 (- m n)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance trunc-trunc-1)
			(:instance expo+ (n (- (1- m) (expo x))) (m (- (1+ (expo x)) n)))))))

(defthm expt-inverse
    (implies (integerp n)
	     (equal (/ (expt 2 n))
		    (expt 2 (- n)))))

(defthm trunc-trunc-3
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (/ (fl (* (expt 2 (- (1- n) (expo x))) x)) (expt 2 (- n m)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance trunc-trunc-2)))))

(defthm trunc-trunc-4
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (/ (* (expt 2 (- (1- n) (expo x))) x) (expt 2 (- n m)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable fl/int expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance trunc-trunc-3)
			(:instance fl/int 
				   (x (* (expt 2 (- (1- n) (expo x))) x))
				   (n (expt 2 (- n m))))))))

(defthm trunc-trunc-5
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (trunc (trunc x n) m)
		(* (fl (* (expt 2 (- (1- m) (expo x))) x))
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance trunc-trunc-4)))))

(defthm trunc-trunc
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (equal (trunc (trunc x n) m)
		    (trunc x m)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance trunc-trunc-5)))))

(defthm plus-trunc-1
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y 0)
		  (integerp k)
		  (> k 0)
		  (= n (+ k (- (expo x) (expo y))))
		  (exactp x n))
	     (equal (+ x (trunc y k))
		    (* (+ (* x (expt 2 (- (1- k) (expo y))))
			  (fl (* y (expt 2 (- (1- k) (expo y))))))
		       (expt 2 (- (1+ (expo y)) k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance expo+ (m (- (1- k) (expo y))) (n (- (1+ (expo y)) k)))))))

(defthm plus-trunc-2
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y 0)
		  (integerp k)
		  (> k 0)
		  (= n (+ k (- (expo x) (expo y))))
		  (exactp x n))
	     (equal (+ x (trunc y k))
		    (* (fl (* (+ x y) (expt 2 (- (1- k) (expo y)))))
		       (expt 2 (- (1+ (expo y)) k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable fl+int expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance plus-trunc-1)
			(:instance fl+int 
				   (x (* y (expt 2 (- (1- k) (expo y)))))
				   (n (* x (expt 2 (- (1- k) (expo y))))))))))

(defthm plus-trunc
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y 0)
		  (integerp k)
		  (> k 0)
		  (= n (+ k (- (expo x) (expo y))))
		  (> n 0)
		  (exactp x n))
	     (= (+ x (trunc y k))
		(trunc (+ x y) (+ k (- (expo (+ x y)) (expo y))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance plus-trunc-2)
			(:instance expo-monotone (y (+ x y)))))))

(defthm trunc-plus-1
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e)))
	     (< (expo y) e))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance expo-lower-bound (x y))
			(:instance expt-strong-monotone (n (expo y)) (m e))))))

(defthm trunc-plus-2
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e)))
	     (< (+ (expt 2 e) y) (expt 2 (1+ e))))
  :rule-classes ())

(defthm trunc-plus-3
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e)))
	     (= (expo (+ (expt 2 e) y)) e))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance expo-lower-bound (x (+ (expt 2 e) y)))
			(:instance expo-upper-bound (x (+ (expt 2 e) y)))
			(:instance trunc-plus-2)
			(:instance expt-strong-monotone (n (expo (+ (expt 2 e) y))) (m (1+ e)))
			(:instance expt-strong-monotone (n e) (m (1+ (expo (+ (expt 2 e) y)))))))))

(defthm trunc-plus-4
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e))
		  (integerp m)
		  (> m 0)
		  (integerp k)
		  (> k 0)
		  (<= m (1+ k)))
	     (= (+ (expt 2 e) (trunc y k))
		(trunc (+ (expt 2 e) y) (- (+ k e) (expo y)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite trunc-rewrite)
		  :use ((:instance trunc-plus-1)
			(:instance trunc-plus-3)
			(:instance plus-trunc (x (expt 2 e)) (n (+ k (- (expo (expt 2 e)) (expo y)))))))))

(defthm trunc-plus
    (implies (and (rationalp y)
		  (> y 0)
		  (integerp e)
		  (< y (expt 2 e))
		  (integerp m)
		  (> m 0)
		  (integerp k)
		  (> k 0)
		  (<= m (1+ k)))
	     (= (trunc (+ (expt 2 e) (trunc y k)) m)
		(trunc (+ (expt 2 e) y) m)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-plus-4)
			(:instance trunc-plus-1)
			(:instance trunc-trunc (x (+ (expt 2 e) y)) (n (- (+ k e) (expo y))))))))

(defthm trunc-n+k-1
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= y (- x (trunc x n))))
	     (< y (expt 2 e)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-diff-pos)))))

(defthm trunc-n+k-2
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (trunc (+ (expt 2 e) y) (1+ k))
		(trunc (+ (expt 2 e) z) (1+ k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-1)
			(:instance trunc-upper-pos)
			(:instance trunc-plus (k n) (m (1+ k)))))))

(defthm trunc-n+k-3
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (and (equal (trunc x n) (* (fl (* x (expt 2 (- e)))) (expt 2 e)))
		  (equal (trunc x (+ n k)) (* (fl (* x (expt 2 (- k e)))) (expt 2 (- e k))))))
  :rule-classes ())

(defthm trunc-n+k-4
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (- (fl (* x (expt 2 (- k e))))
		      (* (expt 2 k) (fl (* x (expt 2 (- e))))))
		   (expt 2 (- e k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo))))

(defthm trunc-n+k-5
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (fl (- (* x (expt 2 (- k e)))
			  (* (expt 2 k) (fl (* x (expt 2 (- e)))))))
		   (expt 2 (- e k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite fl+int)
		  :use ((:instance trunc-n+k-4)
			(:instance fl+int 
				   (x (* x (expt 2 (- k e)))) 
				   (n (* (expt 2 k) (fl (* x (expt 2 (- e)))))))))))

(defthm trunc-n+k-6
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (fl (* y (expt 2 (- k e))))
		   (expt 2 (- e k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance trunc-n+k-5)))))

(defthm trunc-n+k-7
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (+ (expt 2 k) (fl (* y (expt 2 (- k e)))))
		      (expt 2 (- e k)))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-6)))))

(defthm trunc-n+k-8
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (fl (+ (expt 2 k) (* y (expt 2 (- k e)))))
		      (expt 2 (- e k)))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite fl+int)
		  :use ((:instance trunc-n+k-7)
			(:instance fl+int (x (* y (expt 2 (- k e)))) (n (expt 2 k)))))))

(defthm trunc-n+k-9
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (fl (* (expt 2 (- k e)) (+ y (expt 2 e))))
		      (expt 2 (- e k)))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-8)
			(:instance expo+ (n e) (m (- k e)))))))

(defthm trunc-n+k-10
    (implies (and (rationalp y)
		  (integerp e)
		  (<= 0 y))
	     (< 0 (+ y (expt 2 e))))
  :rule-classes ())

(defthm trunc-n+k-11
    (implies (and (integerp k)
		  (> k 0)
		  (rationalp y)
		  (> y 0)
		  (integerp e)
		  (= (expo (+ (expt 2 e) y)) e))
	     (= (* (fl (* (expt 2 (- k e)) (+ y (expt 2 e))))
		   (expt 2 (- e k)))
		(trunc (+ (expt 2 e) y) (1+ k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-10)
			(:instance trunc-pos-rewrite (x (+ y (expt 2 e))) (n (1+ k)))))))

(defthm trunc-n+k-12
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (trunc (+ (expt 2 e) y) (1+ k))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-9)
			(:instance trunc-n+k-1)
			(:instance trunc-n+k-11)
			(:instance trunc-plus-3)
			(:instance trunc-diff-pos)
			(:instance trunc-upper-pos)))))

(defthm trunc-n+k-13
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(- (* (sig (trunc (+ (expt 2 e) y) (1+ k))) (expt 2 e))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-12)
			(:instance trunc-n+k-1)
			(:instance trunc-n+k-11)
			(:instance trunc-plus-3)
			(:instance trunc-diff-pos)
			(:instance trunc-pos)
			(:instance trunc-upper-pos)))))

(defthm trunc-n+k-14
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  (not (= x (trunc x n)))
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (1- (sig (trunc (+ (expt 2 e) z) (1+ k))))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable sig expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-2)
			(:instance trunc-n+k-13)))))

(defthm trunc-n+k
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp k)
		  (> k 0)
		  (integerp n)
		  (>= n k)
		  ;;this isn't really needed, but it won't hurt me.
		  (not (exactp x n))          
		  (= e (- (1+ (expo x)) n))
		  (= z (trunc (- x (trunc x n)) n))
		  (= y (- x (trunc x n))))
	     (= (- (trunc x (+ n k)) (trunc x n))
		(* (1- (sig (trunc (+ (expt 2 e) z) (1+ k))))
		   (expt 2 e))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable sig expo trunc-pos-rewrite)
		  :use ((:instance trunc-n+k-14)
			(:instance trunc-exactp-a)))))