/*
 *  Copyright (c) 2006,2007,2010 Cyrille Berger <cberger@cberger.bet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef _GTLCORE_CHANNEL_MATHS_H_
#define _GTLCORE_CHANNEL_MATHS_H_

#include <cmath>

#include <GTLCore/Export.h>
#include <GTLCore/IntegersMath.h>
#include <GTLCore/Lut.h>

#undef _T

namespace GTLCore
{

  /**
  * This is an empty shell that needs to be "specialized" for each possible
  * numerical type (gtl_uint8, gtl_uint16...).
  *
  * It needs to defines some static constant fields :
  * - zeroValue : the zero for this numerical type
  * - unitValue : the maximum value of the normal dynamic range
  * - max : the maximum value
  * - min : the minimum value
  * - epsilon : a value close to zero but different of zero
  * - bits : the bit depth
  *
  * And some types :
  * - compositetype the type used for composite operations (usually one with
  *   a higher bit depth)
  */
  template<typename _T>
  class ChannelMathsTraits
  {
  public:
  };

  template<>
  class GTLCORE_EXPORT ChannelMathsTraits<gtl_uint8>
  {
  public:
      typedef gtl_int32 compositetype;
      static const gtl_uint8 zeroValue = 0;
      static const gtl_uint8 unitValue = 0x00FF;
      static const gtl_uint8 max = 0x00FF;
      static const gtl_uint8 min = 0;
      static const gtl_uint8 epsilon = 1;
      static const gtl_int8 bits = 8;
  };

  template<>
  class GTLCORE_EXPORT ChannelMathsTraits<gtl_uint16>
  {
  public:
      typedef gtl_int64 compositetype;
      static const gtl_uint16 zeroValue = 0;
      static const gtl_uint16 unitValue = 0xFFFF;
      static const gtl_uint16 max = 0xFFFF;
      static const gtl_uint16 min = 0;
      static const gtl_uint16 epsilon = 1;
      static const gtl_int8 bits = 16;
  };

  template<>
  class GTLCORE_EXPORT ChannelMathsTraits<gtl_int16>
  {
  public:
      typedef gtl_int64 compositetype;
      static const gtl_int16 zeroValue = 0;
      static const gtl_int16 unitValue = 32767;
      static const gtl_int16 max = 32767;
      static const gtl_int16 min = -32768;
      static const gtl_int16 epsilon = 1;
      static const gtl_int8 bits = 16;
  };

  template<>
  class GTLCORE_EXPORT ChannelMathsTraits<gtl_uint32>
  {
  public:
      typedef gtl_int64 compositetype;
      static const gtl_uint32 zeroValue = 0;
      static const gtl_uint32 unitValue = 0xFFFFFFFF;
      static const gtl_uint32 max = 0xFFFFFFFF;
      static const gtl_uint32 min = 0;
      static const gtl_uint32 epsilon = 1;
      static const gtl_int8 bits = 32;
  };

  
#ifdef _HAS_HALF_SUPPORT_

  template<>
  class GTLCORE_EXPORT ChannelMathsTraits<half>
  {
  public:
      typedef double compositetype;
      static const half zeroValue;
      static const half unitValue;
      static const half max;
      static const half min;
      static const half epsilon;
      static const gtl_int8 bits = 16;
  };

#endif
  
  template<>
  class GTLCORE_EXPORT ChannelMathsTraits<float>
  {
  public:
      typedef double compositetype;
      static const float zeroValue;
      static const float unitValue;
      static const float max;
      static const float min;
      static const float epsilon;
      static const gtl_int8 bits = 32;
  };

  template<>
  class GTLCORE_EXPORT ChannelMathsTraits<double>
  {
  public:
      typedef double compositetype;
      static const double zeroValue;
      static const double unitValue;
      static const double max;
      static const double min;
      static const double epsilon;
      static const gtl_int8 bits = 64;
  };

  #ifdef _MSC_VER
  // MSVC do not have lrint

  const double _double2fixmagic = 68719476736.0*1.5;
  const gtl_int32 _shiftamt        = 16;                    //16.16 fixed point representation,

  #if Q_BYTE_ORDER == Q_BIG_ENDIAN
          #define iexp_                           0
          #define iman_                           1
  #else
          #define iexp_                           1
          #define iman_                           0
  #endif //BigEndian_

  inline int float2int(double val)
  {
      val = val + _double2fixmagic;
      return ((int*)&val)[iman_] >> _shiftamt; 
  }

  inline int float2int(float val)
  {
      return float2int((double)val);
  }

  #else

  inline int float2int(float x)
  {
      return lrintf(x);
  }

  inline int float2int(double x)
  {
      return lrint(x);
  }

  #endif

  template<typename _T_>
  struct IntegerToFloat {
    inline float operator()(_T_ f) const
    {
      return f / float(ChannelMathsTraits<_T_>::max);
    }
  };

  template<typename _T_>
  struct FloatToInteger {
    inline _T_ operator()(float f) const
    {
      return float2int(f * ChannelMathsTraits<_T_>::max);
    }
  };

  struct Luts {

    static GTLCORE_EXPORT const FullLut< IntegerToFloat<gtl_uint16>, float, gtl_uint16> Uint16ToFloat;
    static GTLCORE_EXPORT const FullLut< IntegerToFloat<gtl_uint8>, float, gtl_uint8> Uint8ToFloat;
  };
  /**
   * Convert a channel from a gamma color space, to a linear.
   */
 template<typename _T_>
  struct GammaToLinearFloat {
    inline GammaToLinearFloat(double _gamma) : m_gamma(_gamma)
    {
    }
    inline _T_ operator()(_T_ v) const
    {
      if (v < 0.03928) return v / 12.92;
      return std::pow (double((v + 0.055) / 1.055), double(m_gamma));
    }
  private:
    double m_gamma;
  };
  /**
   * Convert a channel from a linear color space, to one with gamma.
   */
  template<typename _T_>
  struct LinearToGammaFloat {
    inline LinearToGammaFloat(double _gamma) : m_gamma(_gamma)
    {
    }
    inline _T_ operator()(_T_ v) const
    {
      if( v < 0.00304 ) return 12.92 * v;
      else return 1.055 * pow (double(v), (1.0/m_gamma)) - 0.055;
    }
  private:
    double m_gamma;
  };
  
  template<typename _TOutput_, typename _TInput_, typename _T1_, typename _T2_>
  struct CombinedOperation {
    CombinedOperation(const _T1_& op1, const _T2_& op2) : m_op1(op1), m_op2(op2)
    {
    }
    inline _TOutput_ operator()(_TInput_ v) const
    {
      return m_op2(m_op1(v));
    }
  private:
    _T1_ m_op1;
    _T2_ m_op2;
  };
  
  
  /**
  * This class defines some elementary operations used by various color
  * space. It's intended to be generic, but some specialization exists
  * either for optimization or just for being buildable.
  *
  * @param _T some numerical type with an existing trait
  * @param _Tdst some other numerical type with an existing trait, it is
  *              only needed if different of _T
  */
  template < typename _T, typename _Tdst = _T >
  class ChannelMaths
  {
      typedef ChannelMathsTraits<_T> traits;
      typedef typename traits::compositetype traits_compositetype;
  public:
      inline static traits_compositetype multiply(traits_compositetype a,
              typename  ChannelMathsTraits<_Tdst>::compositetype b) {
          return ((traits_compositetype)a * b) /  ChannelMathsTraits<_Tdst>::unitValue;
      }
      
      inline static traits_compositetype multiply(traits_compositetype a,
              typename  ChannelMathsTraits<_Tdst>::compositetype b,
              typename  ChannelMathsTraits<_Tdst>::compositetype c) {
          return ((traits_compositetype)a * b * c) / (ChannelMathsTraits<_Tdst>::unitValue * ChannelMathsTraits<_Tdst>::unitValue);
      }

      /**
      * Division : (a * MAX ) / b
      * @param a
      * @param b
      */
      inline static _T divide(_T a, _Tdst b) {
          return ((traits_compositetype)a *  ChannelMathsTraits<_Tdst>::unitValue) / b;
      }

      /**
      * Blending : (a * alpha) + b * (1 - alpha)
      * @param a
      * @param b
      * @param alpha
      */
      inline static _T blend(_T a, _T b, _T alpha) {
          traits_compositetype c = (((traits_compositetype)a - (traits_compositetype)b) * alpha) / traits::unitValue;
          return c + b;
      }

      /**
      * This function will scale a value of type _T to fit into a _Tdst.
      */
      inline static _Tdst scaleToA(_T a) {
          return (traits_compositetype)a >> (traits::bits - ChannelMathsTraits<_Tdst>::bits);
      }

      inline static typename  ChannelMathsTraits<_Tdst>::compositetype clamp(typename  ChannelMathsTraits<_Tdst>::compositetype val) {
          return qBound((typename  ChannelMathsTraits<_Tdst>::compositetype) ChannelMathsTraits<_Tdst>::min,
                        val,
                        (typename  ChannelMathsTraits<_Tdst>::compositetype)ChannelMathsTraits<_Tdst>::max);
      }
  };

  //------------------------------ double specialization ------------------------------//
  template<>
  inline gtl_uint8 ChannelMaths<double, gtl_uint8>::scaleToA(double a)
  {
      double v = a * 255;
      return float2int(bound(0.0, v, 255));
  }

  template<>
  inline double ChannelMaths<gtl_uint8, double>::scaleToA(gtl_uint8 a)
  {
      return Luts::Uint8ToFloat(a);
  }

  template<>
  inline gtl_uint16 ChannelMaths<double, gtl_uint16>::scaleToA(double a)
  {
      double v = a * 0xFFFF;
      return float2int(bound(0, v, 0xFFFF));
  }

  template<>
  inline double ChannelMaths<gtl_uint16, double>::scaleToA(gtl_uint16 a)
  {
      return Luts::Uint16ToFloat(a);
  }

  template<>
  inline double ChannelMaths<double>::clamp(double a)
  {
      return a;
  }

  //------------------------------ float specialization ------------------------------//

  template<>
  inline float ChannelMaths<double, float>::scaleToA(double a)
  {
      return (float)a;
  }

  template<>
  inline double ChannelMaths<float, double>::scaleToA(float a)
  {
      return a;
  }

  template<>
  inline gtl_uint16 ChannelMaths<float, gtl_uint16>::scaleToA(float a)
  {
      float v = a * 0xFFFF;
      return (gtl_uint16)float2int(bound( 0, v, 0xFFFF));
  }

  template<>
  inline float ChannelMaths<gtl_uint16, float>::scaleToA(gtl_uint16 a)
  {
      return Luts::Uint16ToFloat(a);
  }

  template<>
  inline gtl_uint8 ChannelMaths<float, gtl_uint8>::scaleToA(float a)
  {
      float v = a * 255;
      return (gtl_uint8)float2int(bound(0, v, 255));
  }

  template<>
  inline float ChannelMaths<gtl_uint8, float>::scaleToA(gtl_uint8 a)
  {
      return Luts::Uint8ToFloat(a);
  }

  template<>
  inline float ChannelMaths<float>::blend(float a, float b, float alpha)
  {
      return (a - b) * alpha + b;
  }

  template<>
  inline double ChannelMaths<float>::clamp(double a)
  {
      return a;
  }

  //------------------------------ half specialization ------------------------------//

#ifdef _HAS_HALF_SUPPORT_

  template<>
  inline half ChannelMaths<double, half>::scaleToA(double a)
  {
      return (half)a;
  }

  template<>
  inline double ChannelMaths<half, double>::scaleToA(half a)
  {
      return a;
  }

  template<>
  inline float ChannelMaths<half, float>::scaleToA(half a)
  {
      return a;
  }

  template<>
  inline half ChannelMaths<float, half>::scaleToA(float a)
  {
      return (half) a;
  }

  template<>
  inline gtl_uint8 ChannelMaths<half, gtl_uint8>::scaleToA(half a)
  {
      half v = a * 255;
      return (gtl_uint8)(CLAMP(v, 0, 255));
  }

  template<>
  inline half ChannelMaths<gtl_uint8, half>::scaleToA(gtl_uint8 a)
  {
      return a *(1.0 / 255.0);
  }
  template<>
  inline gtl_uint16 ChannelMaths<half, gtl_uint16>::scaleToA(half a)
  {
      double v = a * 0xFFFF;
      return (gtl_uint16)(CLAMP(v, 0, 0xFFFF));
  }

  template<>
  inline half ChannelMaths<gtl_uint16, half>::scaleToA(gtl_uint16 a)
  {
      return a *(1.0 / 0xFFFF);
  }

  template<>
  inline half ChannelMaths<half, half>::scaleToA(half a)
  {
      return a;
  }

  template<>
  inline half ChannelMaths<half>::blend(half a, half b, half alpha)
  {
      return (a - b) * alpha + b;
  }

  template<>
  inline double ChannelMaths<half>::clamp(double a)
  {
      return a;
  }


#endif

  //------------------------------ gtl_uint8 specialization ------------------------------//

  template<>
  inline gtl_int32 ChannelMaths<gtl_uint8>::multiply(gtl_int32 a, gtl_int32 b)
  {
      return UINT8_MULT(a, b);
  }


  template<>
  inline gtl_int32 ChannelMaths<gtl_uint8>::multiply(gtl_int32 a, gtl_int32 b, gtl_int32 c)
  {
      return UINT8_MULT3(a, b, c);
  }

  template<>
  inline gtl_uint8 ChannelMaths<gtl_uint8>::divide(gtl_uint8 a, gtl_uint8 b)
  {
      return UINT8_DIVIDE(a, b);
  }

  template<>
  inline gtl_uint8 ChannelMaths<gtl_uint8>::blend(gtl_uint8 a, gtl_uint8 b, gtl_uint8 c)
  {
      return UINT8_BLEND(a, b, c);
  }

  //------------------------------ gtl_uint16 specialization ------------------------------//

  template<>
  inline gtl_int64 ChannelMaths<gtl_uint16>::multiply(gtl_int64 a, gtl_int64 b)
  {
      return UINT16_MULT(a, b);
  }

  template<>
  inline gtl_uint16 ChannelMaths<gtl_uint16>::divide(gtl_uint16 a, gtl_uint16 b)
  {
      return UINT16_DIVIDE(a, b);
  }

  //------------------------------ various specialization ------------------------------//


  // TODO: use more functions from KoIntegersMaths to do the computation

  /// This specialization is needed because the default implementation won't work when scaling up
  template<>
  inline gtl_uint16 ChannelMaths<gtl_uint8, gtl_uint16>::scaleToA(gtl_uint8 a)
  {
      return UINT8_TO_UINT16(a);
  }

  template<>
  inline gtl_uint8 ChannelMaths<gtl_uint16, gtl_uint8>::scaleToA(gtl_uint16 a)
  {
      return UINT16_TO_UINT8(a);
  }


  // Due to once again a bug in gcc, there is the need for those specialized functions:

  template<>
  inline gtl_uint8 ChannelMaths<gtl_uint8, gtl_uint8>::scaleToA(gtl_uint8 a)
  {
      return a;
  }

  template<>
  inline gtl_uint16 ChannelMaths<gtl_uint16, gtl_uint16>::scaleToA(gtl_uint16 a)
  {
      return a;
  }

  template<>
  inline float ChannelMaths<float, float>::scaleToA(float a)
  {
      return a;
  }
}

#endif
