/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "PixelConvertExpressionFactory_p.h"

#include <llvm/Instructions.h>

#include "GTLCore/Type.h"
#include "GTLCore/AST/ConvertExpression.h"
#include "GTLCore/LLVMBackend/ExpressionResult_p.h"
#include "GTLCore/LLVMBackend/ExpressionGenerationContext_p.h"
#include "GTLCore/Macros_p.h"

#include "CodeGenerator_p.h"
#include "Debug.h"
#include "wrappers/PixelWrap_p.h"
#include <GTLCore/LLVMBackend/CodeGenerator_p.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/LLVMBackend/Visitor_p.h>
#include <GTLFragment/wrappers/ColorWrap_p.h>
#include <llvm/Function.h>
#include <GTLCore/Type_p.h>

using namespace OpenShiva;

namespace OpenShiva {
  class PixelConvertExpression : public GTLCore::AST::ConvertExpression {
    public:
      PixelConvertExpression( GTLCore::AST::Expression* _value,  const GTLCore::Type* _dstType ) : GTLCore::AST::ConvertExpression( _value ), m_dstType(_dstType)
      {
      }
      virtual ~PixelConvertExpression() {}
      virtual const GTLCore::Type* type() const { return m_dstType; }
      virtual LLVMBackend::ExpressionResult generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
      {
        if( value()->type()->dataType() == GTLCore::Type::STRUCTURE )
        {
          llvm::Value* pixelPtr = value()->generateValue( _gc, _egc).value();
          llvm::Value* pixelDataPtr = OpenShiva::CodeGenerator::accessPixelDataPtr( _gc, _egc.currentBasicBlock(), pixelPtr);
          if(type()->dataType() == GTLCore::Type::STRUCTURE)
          {
            GTL_ASSERT(type()->structName() == "color");
            llvm::Value* converterDataPtr = OpenShiva::CodeGenerator::accessColorConverterPtr( _gc, _egc.currentBasicBlock(), pixelPtr);

            // Allocate the memory for color structure
            llvm::Value* colorPtr = LLVMBackend::CodeGenerator::allocateMemory(_gc, type()->d->type(_gc.llvmContext()), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), INT32_C(1)), _egc.currentBasicBlock());
      
            // Initialize the structure
            const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor(type());
            _egc.setCurrentBasicBlock(visitor->initialise( _gc, _egc.currentBasicBlock(), colorPtr, type(), std::list<llvm::Value*>()));

            std::vector<llvm::Value*> indexes;
            indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
            indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(ColorWrap::POS_RED) ));
            llvm::Value* rgbPtr = llvm::GetElementPtrInst::Create( colorPtr, indexes, "", _egc.currentBasicBlock() );
            
            //Parameters are AbstractColorConverter*, ptrToColor, ptrTopixeldata, number of channels
            std::vector<llvm::Value*> parameters;
            parameters.push_back( new llvm::LoadInst( converterDataPtr, "PixelConvertExpression::generateValue", _egc.currentBasicBlock() ));
            parameters.push_back(pixelDataPtr);
            parameters.push_back(rgbPtr);
            parameters.push_back(LLVMBackend::CodeGenerator::integerToConstant( _gc.llvmContext(), llvm::cast<llvm::VectorType>(pixelDataPtr->getType()->getContainedType(0))->getNumElements() ) );

            llvm::Function* func = CodeGenerator::createWrapConvertPixelToColorFunction(_gc, pixelDataPtr->getType());
            GTL_COMPARE_FUNCTION_PARAMETERS( func, parameters );
            
            llvm::CallInst *CallFunc = llvm::CallInst::Create(func, parameters, "", _egc.currentBasicBlock());
            CallFunc->setTailCall(false);
            _egc.setCurrentBasicBlock( LLVMBackend::Visitor::getVisitorFor(type())->mark( _gc, _egc.currentBasicBlock(), colorPtr, type(), LLVMBackend::CodeGenerator::integerToConstant( _gc.llvmContext(), INT32_C(-1) ) ) );

            // Mark the color structure for deletion
            _egc.setCurrentBasicBlock(visitor->mark(_gc, _egc.currentBasicBlock(), colorPtr, type(), LLVMBackend::CodeGenerator::integerToConstant(_gc.llvmContext(), -1)));
            
            return LLVMBackend::ExpressionResult(colorPtr, type(), true);
          } else {
            return LLVMBackend::ExpressionResult( new llvm::LoadInst( pixelDataPtr, "", _egc.currentBasicBlock() ) , m_dstType );
          }
        } else {
          return value()->generateValue( _gc, _egc);
        }
      }
      virtual GTLCore::AST::ExpressionResultSP generateValue( GTLCore::AST::GenerationVisitor* _generationVisitor) const
      {
        GTL_ABORT("UNIMPLEMENTED");
		return 0;
      }
    private:
      const GTLCore::Type* m_dstType;
  };
}

PixelConvertExpressionFactory::PixelConvertExpressionFactory()
{
}

PixelConvertExpressionFactory::~PixelConvertExpressionFactory()
{
}

GTLCore::AST::ConvertExpression* PixelConvertExpressionFactory::create( GTLCore::AST::Expression* value, const GTLCore::Type* _dstType ) const
{
  return new PixelConvertExpression( value, _dstType );
}

bool PixelConvertExpressionFactory::canConvertBetween( const GTLCore::Type* srcType, const GTLCore::Type* dstType) const
{
  if(srcType->dataType() == GTLCore::Type::STRUCTURE and srcType->structName().startWith( "pixel" ))
  {
    return srcType->structDataMember(PixelWrap::INDEX_DATA).type() == dstType
           or (dstType->dataType() == GTLCore::Type::STRUCTURE and dstType->structName() == "color");
  } else if(dstType->dataType() == GTLCore::Type::STRUCTURE and dstType->structName().startWith( "pixel" ))
  {
    return dstType->structDataMember(PixelWrap::INDEX_DATA).type() == srcType;
  }
  return false;
}

GTLCore::ConvertCenter::ConversionQuality PixelConvertExpressionFactory::conversionQuality( const GTLCore::Type* _srcType, const GTLCore::Type* _dstType ) const
{
  GTL_ASSERT( canConvertBetween( _srcType, _dstType) );
  UNUSED(_srcType);
  UNUSED(_dstType);
  return GTLCore::ConvertCenter::LOSSLESS_CONVERSION;
}
