///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file AtomsObject.h
 * \brief Contains the definition of the AtomViz::AtomsObject class.
 */

#ifndef __ATOMSOBJECT_H
#define __ATOMSOBJECT_H

#include <atomviz/AtomViz.h>
#include "datachannels/DataChannel.h"
#include "SimulationCell.h"

#include <core/scene/objects/SceneObject.h>
#include <core/gui/properties/PropertiesEditor.h>

namespace AtomViz {

class AtomsObjectEditor;	// defined in AtomsObjectEditor.h

/**
 * \brief Contains all data associated with a simulation dataset including
 *        the atoms, simulation box geometry, data channels, and
 *        atom types.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomsObject : public SceneObject
{
public:

	/// \brief Default constructor that creates an AtomsObject with
	///        no atoms, no data channels and an empty simulation box.
	///
	/// \param isLoading Specifies whether the object's data fields will be initialized from the
	///                  data stored in a scene file after the instance has been created.
	AtomsObject(bool isLoading = false);

	///////////////////// Overrides from SceneObject /////////////////////////

	/// Asks the object for its validity interval at the given time.
	virtual TimeInterval objectValidity(TimeTicks time);

	/// Makes the object render itself into the viewport.
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp);

	/// \brief Asks the object whether it should show in a rendered image.
	virtual bool isRenderable() { return true; }

	/// \brief Renders the object in preview rendering mode using OpenGL.
	/// \param time The animation time at which to render the object
	/// \param view Describes the camera projection.
	/// \param contextNode The node context used to render the object.
	/// \param imageWidth The width of the rendering buffer in pixels.
	/// \param imageHeight The height of the rendering buffer in pixels.
	/// \param glcontext The window that provides the OpenGL rendering context.
	/// \return \c false if the rendering has been aborted by the user; \c true otherwise.
	virtual bool renderPreview(TimeTicks time, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

	/// Returns the bounding box of the object in local object coordinates.
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode);

	/// Indicates if this object should be surrounded by a selection marker when selected.
	/// The default implementation returns true and can be overridden by sub-classes.
	virtual bool showSelectionMarker() { return false; }

	/// \brief Performs a ray intersection calculation.
	/// \param ray The ray to test for intersection with the object. Ray is specified in object space.
	/// \param time The animation at which to do the ray-object intersection test.
	/// \param contextNode The scene nodes to which this scene object belongs to (can be NULL).
	/// \param t If an intersection has been found, the method stores the distance of the intersection in this output parameter.
	/// \param normal If an intersection has been found, the method stores the surface normal at the point of intersection in this output parameter.
	/// \return True if the closest intersection has been found. False if no intersection has been found.
	virtual bool intersectRay(const Ray3& ray, TimeTicks time, ObjectNode* contextNode, FloatType& t, Vector3& normal);

	//////////////////////// AtomsObject specific methods ////////////////////////

	/// \brief Creates a new custom (non-standard) data channel.
	/// \param dataType Specifies the data type (integer, floating-point, ...) of the per-atom elements
	///                 in the new data channel. The data type is specified as identifier according to the
	///                 Qt meta type system.
	/// \param dataTypeSize The size of the data type given by \a dataType in bytes.
	///                     This is necessary because the Qt type system has no function to query
	///                     the size of the type programmaticaly.
	/// \param componentCount The number of components per atom of type \a dataType.
	/// \return The newly created data channel that has been added to the AtomsObject.
	/// \undoable
	/// \sa createStandardDataChannel()
	DataChannel* createCustomDataChannel(int dataType, size_t dataTypeSize, size_t componentCount);

	/// \brief Creates a standard data channel.
	/// \param which The channel to create.
	/// \return The newly created channel or an existing channel if the AtomsObject already
	///         contained the requested standard channel.
	/// \undoable
	/// \sa createCustomDataChannel()
	DataChannel* createStandardDataChannel(DataChannel::DataChannelIdentifier which);

	/// \brief Inserts a data channel.
	/// \param newChannel The channel to be inserted.
	///
	/// If the new channel is one of the standard data channels and the AtomsObject already contains
	/// such a channel then it will be replaced by the new one.
	///
	/// The new data channel must have the correct size, i.e. the same number of atoms as the AtomsObject.
	///
	/// \undoable
	/// \sa createCustomDataChannel(), createStandardDataChannel()
	void insertDataChannel(DataChannel* newChannel);

	/// \brief Inserts a data channel.
	/// \param newChannel The channel to be inserted.
	///
	/// This is the same method as above but takes a smart pointer instead of a raw pointer.
	/// \undoable
	void insertDataChannel(const DataChannel::SmartPtr& newChannel) { insertDataChannel(newChannel.get()); }

	/// \brief Removes a data channel.
	/// \param channel The data channel to be removed.
	/// \undoable
	void removeDataChannel(DataChannel* channel) {
		int index = _dataChannels.indexOf(channel);
		OVITO_ASSERT(index >= 0);
		_dataChannels.remove(index);
	}

	/// \brief Returns the list of data channels.
	/// \return The list of DataChannel objects used by this AtomsObject.
	const DataChannelList& dataChannels() const { return _dataChannels; }

	/// \brief Returns a standard data channel with the given identifier.
	/// \param identifier Specifies the standard channel to return.
	/// \return The requested standard data channel or \c NULL if there is
	///         no such data channel in this AtomsObject.
	DataChannel* getStandardDataChannel(DataChannel::DataChannelIdentifier identifier) const;

	/// \brief Returns the first data channel with the given identifier.
	/// \param name The name to search for.
	/// \return The requested data channel from this AtomsObject or \c NULL if there is
	///         no such data channel in this AtomsObject.
	DataChannel* findDataChannelByName(const QString& name) const;

	/// \brief Looks up a DataChannel based on an offline reference.
	/// \param ref Specifies which data channel to return.
	/// \return The requested data channel or \c NULL if there is no such data channel in this AtomsObject.
	DataChannel* lookupDataChannel(const DataChannelReference& ref) const;

	/// \brief Returns the index of a DataChannel in this AtomsObject.
	/// \param channel The data channel to look up.
	/// \return The zero-based index of the requested data channel in this AtomsObject's array of data channels or
	///         -1 if the AtomsObject does not contain this channel or \a channel is \c NULL.
	int dataChannelIndex(DataChannel* channel) const { return dataChannels().indexOf(channel); }

	/// \brief Returns whether a standard channel is visible.
	/// \param identifier The standard channel to check.
	/// \return \c true if the visibility flag of the given data channel is set;
	///         \c false otherwise or if this AtomsObject does not contain the given data channel.
	bool isStandardChannelVisible(DataChannel::DataChannelIdentifier identifier) const {
		DataChannel* channel = getStandardDataChannel(identifier);
		return (channel != NULL) ? channel->isVisible() : false;
	}

	/// \brief Turns a shallow copy of a data channel into a real copy.
	/// \param channel The channel referenced by this AtomsObject that should be
	///                turned into a deep copy.
	/// \return The deep copy of the specified data channel that has been created to replace the
	///         original shallow copy; or the original channel if it is exclusively used by this
	///         AtomsObject and therefore no copy operation is required.
	/// \undoable
	DataChannel* copyShallowChannel(DataChannel* channel);

	/// \brief Replaces a DataChannel in the AtomsObject with a new one.
	/// \param oldChannel The channel to be replaced.
	/// \param newChannel The new channel.
	/// \undoable
	void replaceDataChannel(DataChannel* oldChannel, const DataChannel::SmartPtr& newChannel);

	/// \brief Resizes the atoms array.
	/// \param newAtomCount The new number of atoms.
	/// \return The old atom count.
	/// \note This method does not record the old state on the undo stack.
	/// \sa atomsCount()
	size_t setAtomsCount(size_t newAtomCount);

	/// \brief Returns the total number of atoms.
	/// \return The number of atoms stored in the data channels of this AtomsObject.
	/// \sa setAtomsCount()
	size_t atomsCount() const { return numAtoms; }

	/// \brief Returns an array of color values that contains the individual color
	///        of each atom in the AtomsObject.
	/// \param[in] time The animation time at which the color should be returned.
	/// \param[in,out] validityInterval This interval is reduced by the method to the
	///                                 interval during which the atom colors do not change.
	/// \return A vector that contains the color for each atom as it is shown in the viewports.
	///
	/// This method gets the atom colors from the color standard data channel if presents
	/// or otherwise uses the colors defined by the atom types.
	QVector<Color> getAtomColors(TimeTicks time, TimeInterval& validityInterval);

	/// \brief Returns an array of values that contains the individual radius
	///        of each atom.
	/// \param[in] time The animation time at which the radii should be returned.
	/// \param[in,out] validityInterval This interval is reduced by the method to the
	///                                 interval during which the atom radii do not change.
	/// \return A vector that contains the radius for each atom as it is shown in the viewports.
	///
	/// This method gets the atom radii from the radius standard data channel if presents
	/// or otherwise uses the radii defined by the atom types.
	QVector<FloatType> getAtomRadii(TimeTicks time, TimeInterval& validityInterval);

	/// \brief Returns the simulation cell.
	/// \return An object that defines the geometry of the simulation cell.
	/// \sa setSimulationCell()
	SimulationCell* simulationCell() const { return _simulationCell; }

	/// \brief Assigns a new simulation cell.
	/// \param newCell The new simulation cell geometry that should replace the old one.
	/// \undoable
	/// \sa simulationCell()
	void setSimulationCell(SimulationCell* newCell) { _simulationCell = newCell; }

	/// \brief Deletes some atoms from the object.
	/// \param mask A bit array with one bit per atom. Atoms for which the
	///             bit is set are deleted from the object.
	/// \return The number of remaining atoms.
	/// \undoable
	size_t deleteAtoms(const dynamic_bitset<>& mask);

	/// \brief Invalidates all temporary information stored with the AtomsObject.
	///
	/// This method must be called each time the atoms have been changed in some
	/// way. It clears the rendering cache so it will be rebuilt on then next
	/// time the object is rendered in the viewports.
	void invalidate();

	/// \brief Returns whether atomic coordinates are saved along with the scene.
	/// \return \c true if data is stored in the scene file; \c false if the data resides in an external file.
	bool serializeAtoms() const { return _serializeAtoms; }

	/// \brief Sets whether atomic coordinates are saved along with the scene.
	/// \param on \c true if data should be stored in the scene file; \c false if the data resides in an external file.
	/// \undoable
	void setSerializeAtoms(bool on);

protected:

	/// From RefMaker.
	virtual void onRefTargetInserted(const PropertyFieldDescriptor& field, RefTarget* newTarget, int listIndex);
	/// From RefMaker.
	virtual void onRefTargetRemoved(const PropertyFieldDescriptor& field, RefTarget* oldTarget, int listIndex);
	/// This method is called when a reference target changes.
	virtual bool onRefTargetMessage(RefTarget* source, RefTargetMessage* msg);

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);
	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

	/// Stores the total number of atoms.
	size_t numAtoms;

	/// Contains all data channels.
	VectorReferenceField<DataChannel> _dataChannels;

	/// The current bounding box of the SceneObject.
	Box3 sceneBoundingBox;

	/// Indicates if the bounding box has to be recomputed.
	TimeInterval sceneBoundingBoxValidity;

	/// The dimensions of the simulation cell containing the atoms.
	ReferenceField<SimulationCell> _simulationCell;

	/// Controls whether the atom data is saved along with the scene.
	PropertyField<bool> _serializeAtoms;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AtomsObject)
	DECLARE_VECTOR_REFERENCE_FIELD(_dataChannels)
	DECLARE_REFERENCE_FIELD(_simulationCell)
	DECLARE_PROPERTY_FIELD(_serializeAtoms)
};

};	// End of namespace AtomViz

#endif // __ATOMSOBJECT_H
