///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/PluginManager.h>
#include "CreationCommandPage.h"

namespace Core {

static QString objectClasses[] = {
	CreationCommandPage::tr("Geometry"),
	CreationCommandPage::tr("Shapes"),
	CreationCommandPage::tr("Lights"),
	CreationCommandPage::tr("Cameras"),
};

static QString objectClassIcons[] = {
		":/core/command_panel/objclass_geometry.png",
		":/core/command_panel/objclass_shapes.png",
		":/core/command_panel/objclass_lights.png",
		":/core/command_panel/objclass_cameras.png",
};

/******************************************************************************
* Initializes the creation panel.
******************************************************************************/
CreationCommandPage::CreationCommandPage() :
	CommandPanelPage(), currentMode(NULL), currentCreationButton(NULL), createButtonGroup(NULL)
{
	QVBoxLayout* layout = new QVBoxLayout();
	layout->setContentsMargins(2,2,2,2);

	OVITO_ASSERT(NUM_OBJ_CLASSES == sizeof(objectClasses)/sizeof(objectClasses[0]));

	// Scan for object types.
    scanInstalledObjectTypes();

	// Create the radio buttons for the object class.
	objClassToolbar = new QToolBar(this);
	objClassToolbar->setMovable(false);
	objClassToolbar->setFloatable(false);
	objClassToolbar->setStyleSheet("QToolBar { padding: 0px; margin: 0px; border: 0px none black; } QToolButton { padding: 0px; margin: 0px }");
	layout->addWidget(objClassToolbar, 0, Qt::AlignHCenter);

	objClassActions = new QActionGroup(this);
	for(size_t i=0; i<NUM_OBJ_CLASSES; i++) {
		QAction* action = new QAction(QIcon(objectClassIcons[i]), objectClasses[i], objClassActions);
		action->setCheckable(true);
		action->setData((int)i);
		objClassActions->addAction(action);
	}
	objClassToolbar->addActions(objClassActions->actions());
	connect(objClassActions, SIGNAL(triggered(QAction*)), this, SLOT(onObjectClassButton(QAction*)));

	// Create the Category box.
	categoryBox = new QComboBox(this);
	layout->addWidget(categoryBox);
	connect(categoryBox, SIGNAL(currentIndexChanged(int)), this, SLOT(onCategorySelected()));

	// Create the properties panel.
	propertiesPanel = new PropertiesPanel(this);
	layout->addWidget(propertiesPanel, 1);

	// Create rollout for object type buttons.
	objectTypePanel = new QWidget();
	QGridLayout* gridLayout = new QGridLayout();
	gridLayout->setContentsMargins(4,4,4,4);
	objectTypePanel->setLayout(gridLayout);
	propertiesPanel->addRollout(objectTypePanel, tr("Object Type"));

	objectTypePanel->setStyleSheet("QPushButton:checked { "
							   		"background-color: palegreen; "
							   		"}");

	setLayout(layout);

	/// Register a listener with the input manager to track the current creation mode.
	connect(&VIEWPORT_INPUT_MANAGER, SIGNAL(inputModeChanged(ViewportInputHandler*, ViewportInputHandler*)), this, SLOT(onInputModeChanged(ViewportInputHandler*, ViewportInputHandler*)));
}

/******************************************************************************
* Resets the creation panel to the initial state.
******************************************************************************/
void CreationCommandPage::reset()
{
	CommandPanelPage::reset();

	// Select initial object class.
	currentObjClass = NUM_OBJ_CLASSES;
	setObjectClass(GEOMETRY);
}

/******************************************************************************
* Is called when the user selects another page.
******************************************************************************/
void CreationCommandPage::onLeave()
{
	endCreation();
	CommandPanelPage::onLeave();
}

/******************************************************************************
* This is called after all changes to the selection set have been completed.
******************************************************************************/
void CreationCommandPage::onSelectionChangeComplete(SelectionSet* newSelection)
{
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != currentMode.get())
		propertiesPanel->setEditObject(NULL);
}

/******************************************************************************
* Selects the given object class.
******************************************************************************/
void CreationCommandPage::setObjectClass(ObjectClass objClass)
{
	if(currentObjClass == objClass) return;
	currentObjClass = objClass;

	// Update contents of the category drop-down box.
    categoryBox->clear();
	for(map<int,ObjectCategory>::iterator iter = categoryLists[currentObjClass].begin(); iter != categoryLists[currentObjClass].end(); ++iter) {
		categoryBox->addItem(iter->second.displayName, qVariantFromValue((void*)&iter->second));
	}

	// Select the first category.
	if(categoryBox->count() != 0)
		categoryBox->setCurrentIndex(0);

	objClassActions->actions().at(currentObjClass)->setChecked(true);

	// Update the displayed button in the object type panel.
	rebuildObjectTypePanel();
}

/******************************************************************************
* Is called when the user selects one of the object class buttons.
******************************************************************************/
void CreationCommandPage::onObjectClassButton(QAction* action)
{
	OVITO_ASSERT(objClassActions->actions().contains(action));
	int id = action->data().toInt();
	OVITO_ASSERT(id >= GEOMETRY && id < NUM_OBJ_CLASSES);
	setObjectClass((ObjectClass)id);
}

/******************************************************************************
* Is called when the user has selected an object category.
******************************************************************************/
void CreationCommandPage::onCategorySelected()
{
	rebuildObjectTypePanel();
}

/******************************************************************************
* Finds all object types provided by the installed plugins.
******************************************************************************/
void CreationCommandPage::scanInstalledObjectTypes()
{
	// Create an iterator that retrieves all object creation modes.
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(CreationMode))) {

		// Extract object class and category from the manifest.
		QDomElement descriptionElement = clazz->getMetaData("Creation-Mode-Description");

		QString objClass = descriptionElement.attribute("Object-Class");
		for(int oc=0; oc<NUM_OBJ_CLASSES; oc++) {
			if(objClass == objectClasses[oc]) {

				// Insert object type into the right category.
				int categoryId = descriptionElement.attribute("Category-Id").toInt();
				ObjectCategory& category = categoryLists[oc][categoryId];
				category.identifier = categoryId;
				category.displayName = descriptionElement.attribute("Category-Name");
				category.classes.push_back(clazz);

                break;
			}
		}
	}
}

/******************************************************************************
* Updates the displayed button in the object type panel.
******************************************************************************/
void CreationCommandPage::rebuildObjectTypePanel()
{
	/// Deactivate the current creation mode.
	endCreation();

	Q_FOREACH(QObject* w, objectTypePanel->children())
		if(w->isWidgetType()) delete w;
	if(createButtonGroup) {
		delete createButtonGroup;
		createButtonGroup = NULL;
	}

	if(categoryBox->currentIndex() < 0)
		return;

	// Get the selected object catgeory.
	ObjectCategory* category = (ObjectCategory*)categoryBox->itemData(categoryBox->currentIndex()).value<void*>();
	if(!category) return;
	CHECK_POINTER(category);

	createButtonGroup = new QButtonGroup(objectTypePanel);
	createButtonGroup->setExclusive(false);

	// Iterate over all creation modes in the selected category.
	for(int index = 0; index < (int)category->classes.size(); index++) {
		PluginClassDescriptor* descriptor = category->classes[index];

        // Extract display name of the creation mode from manifest metadata.
		QDomElement descriptionElement = descriptor->getMetaData("Creation-Mode-Description");
		OVITO_ASSERT(!descriptionElement.isNull());
		QString displayName = descriptionElement.attribute("Display-Name");

		// Create a button that activates the creation mode.
        QPushButton* btn = new QPushButton(displayName, objectTypePanel);
		btn->setCheckable(true);
		createButtonGroup->addButton(btn);
		objectTypePanel->layout()->addWidget(btn);

		// Associate button with the creation mode.
		btn->setProperty("ClassDescriptor", qVariantFromValue((void*)descriptor));
	}

	// Listen for events.
	connect(createButtonGroup, SIGNAL(buttonClicked(QAbstractButton*)), this, SLOT(onCreateObjectButton(QAbstractButton*)));
	// Resize the rollout panel.
	objectTypePanel->updateGeometry();
}

/******************************************************************************
* Is called when the user has clicked on one of the object creation buttons in the object type panel.
******************************************************************************/
void CreationCommandPage::onCreateObjectButton(QAbstractButton* btn)
{
	PluginClassDescriptor* descriptor = (PluginClassDescriptor*)btn->property("ClassDescriptor").value<void*>();
	CHECK_POINTER(descriptor);

	/// Deactivate the current creation mode.
	endCreation();

	try {
		// Create an instance of the creation mode.
		currentMode = static_object_cast<CreationMode>(descriptor->createInstance());
		currentMode->setPropertiesPanel(propertiesPanel);
		currentCreationButton = btn;

		// Activate the new creation mode.
		// This will also toggle the button.
		VIEWPORT_INPUT_MANAGER.pushInputHandler(currentMode);
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
* Deactivates the current creation mode.
******************************************************************************/
void CreationCommandPage::endCreation()
{
	propertiesPanel->setEditObject(NULL);

	if(!currentMode) return;
	CHECK_OBJECT_POINTER(currentMode.get());
	CHECK_POINTER(currentCreationButton);
	OVITO_ASSERT(currentCreationButton->property("ClassDescriptor").value<void*>() == currentMode->pluginClassDescriptor());

	CreationMode::SmartPtr oldMode = currentMode;
	currentMode = NULL;

	// Deactivate the creation mode.
	// This will also toggle the button.
	VIEWPORT_INPUT_MANAGER.removeInputHandler(oldMode.get());
	OVITO_ASSERT(currentCreationButton->isChecked() == false);

	currentCreationButton = NULL;
	CHECK_OBJECT_POINTER(oldMode);
}

/******************************************************************************
* Is called when the active viewport input handler has changed.
******************************************************************************/
void CreationCommandPage::onInputModeChanged(ViewportInputHandler* oldMode, ViewportInputHandler* newMode)
{
	if(newMode != NULL && newMode == currentMode.get()) {
		currentCreationButton->setChecked(true);
	}
	else if(currentCreationButton != NULL) {
		currentCreationButton->setChecked(false);
		endCreation();
	}
}

};
