///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/geometry/SimpleGeometryObject.h>
#include <core/scene/objects/geometry/MeshObject.h>
#include <core/scene/ObjectNode.h>
#include <core/viewport/Viewport.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(SimpleGeometryObject, SceneObject)

/******************************************************************************
* Makes the object render itself into the viewport.
* The viewport transformation is already set up, when this method is called by the
* system. The object has to be rendered in the local object coordinate system.
******************************************************************************/
void SimpleGeometryObject::renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp)
{
	if(!meshValidity.contains(time))
		buildMesh(time, mesh, meshValidity);
	vp->renderNodeMesh(mesh, contextNode);
}

/******************************************************************************
* Renders the object in preview rendering mode using OpenGL.
******************************************************************************/
bool SimpleGeometryObject::renderPreview(TimeTicks time, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext)
{
	SceneObject::SmartPtr meshObject = convertTo(PLUGINCLASSINFO(MeshObject), time);
	if(meshObject)
		return meshObject->renderPreview(time, view, contextNode, imageWidth, imageHeight, glcontext);
	return true;
}

/******************************************************************************
* This asks the object if it supports the conversion to another
******************************************************************************/
bool SimpleGeometryObject::canConvertTo(PluginClassDescriptor* objectClass)
{
	// Can convert to a MeshObject.
	if(PLUGINCLASSINFO(MeshObject)->isKindOf(objectClass)) return true;

	return SceneObject::canConvertTo(objectClass);
}

/******************************************************************************
* Lets the object convert itself to another SceneObject derived type.
******************************************************************************/
SceneObject::SmartPtr SimpleGeometryObject::convertTo(PluginClassDescriptor* objectClass, TimeTicks time)
{
	if(PLUGINCLASSINFO(MeshObject)->isKindOf(objectClass)) {
		if(!meshValidity.contains(time))
			buildMesh(time, mesh, meshValidity);
        MeshObject* meshObj = new MeshObject();
		meshObj->setMesh(mesh);
		meshObj->setObjectValidity(meshValidity);
		return meshObj;
	}
	return SceneObject::convertTo(objectClass, time);
}

};
