/*********************************************************************/
// dar - disk archive - a backup/restoration program
// Copyright (C) 2002-2025 Denis Corbin
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//
// to contact the author, see the AUTHOR file
/*********************************************************************/

    /// \file fichier_libssh.hpp
    /// \brief class fichier_libssh definition. This is a full implementation/inherited class of class fichier_global
    /// this type of object are generated by entrepot_libssh.
    /// \ingroup Private

#ifndef FICHIER_LIBSSH_HPP
#define FICHIER_LIBSSH_HPP


#include "../my_config.h"

extern "C"
{
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
} // end extern "C"

#include "integers.hpp"
#include "user_interaction.hpp"
#include "fichier_global.hpp"
#include "libssh_connection.hpp"
#include "cygwin_adapt.hpp"

#include <string>

namespace libdar
{

	/// \addtogroup Private
	/// @{


	/// filesystem libssh files

#if LIBSSH_AVAILABLE

    class fichier_libssh : public fichier_global
    {
    public:

	    /// constructor
        fichier_libssh(const std::shared_ptr<user_interaction> & dialog,
		       const std::shared_ptr<libssh_connection> & ptr,
		       const std::string & chemin,
		       gf_mode mode,
		       U_I permission,
		       bool fail_if_exists,
		       bool erase);

	    /// copy constructor
	fichier_libssh(const fichier_libssh & ref) = delete;

	    /// move constructor
	fichier_libssh(fichier_libssh && ref) noexcept = delete;

	    /// assignment operator
	fichier_libssh & operator = (const fichier_libssh & ref) = delete;

	    /// move operator
	fichier_libssh & operator = (fichier_libssh && ref) noexcept = delete;

	    /// destructor
	~fichier_libssh();


	    /// set the ownership of the file
	virtual void change_ownership(const std::string & user, const std::string & group) override
	{ throw Efeature(gettext("user/group ownership not implemented for this repository")); };
	    // not implemented, libssh can set ownership based on uid / gid but not based on user/group names

	    /// change the permission of the file
	virtual void change_permission(U_I perm) override;

	    /// return the size of the file
        virtual infinint get_size() const override;

	    /// set posix_fadvise for the whole file
	virtual void fadvise(advise adv) const override {}; // not supported

            // inherited from generic_file
	virtual bool skippable(skippability direction, const infinint & amount) override;
        virtual bool skip(const infinint & pos) override;
        virtual bool skip_to_eof() override;
        virtual bool skip_relative(S_I x) override;
	virtual bool truncatable(const infinint & pos) const override;
        virtual infinint get_position() const override;

    protected:
	    // inherited from generic_file grand-parent class
	virtual void inherited_truncate(const infinint & pos) override;
	virtual void inherited_read_ahead(const infinint & amount) override;
	virtual void inherited_sync_write() override {};
	virtual void inherited_flush_read() override {};
	virtual void inherited_terminate() override { myclose(); };

	    // inherited from fichier_global parent class
	virtual U_I fichier_global_inherited_write(const char *a, U_I size) override;
        virtual bool fichier_global_inherited_read(char *a, U_I size, U_I & read, std::string & message) override;

    private:
	std::shared_ptr<libssh_connection> connect;
	std::string my_path;
	sftp_file sfd;
	infinint current_pos; ///< we cannot rely in libssh to provide the read offset as we also use Async I/O

	    // read ahead structures

	struct rahead
	{
	    sftp_aio handle;

	    rahead(): handle(nullptr) {};
	    rahead(const rahead & ref) = delete;
	    rahead(rahead && ref) noexcept { handle = ref.handle; ref.handle = nullptr; };
	    rahead & operator = (const rahead & ref) = delete;
	    rahead & operator = (rahead && ref) noexcept { std::swap(handle, ref.handle); return *this; };
	    ~rahead() { if(handle != nullptr) sftp_aio_free(handle); };
	};

	static const U_I read_ahead_window_size = 100*1024*1024;
	    /// capping the AIO to memory footprint to 100 Mio

	infinint tora; ///< amount of data to read ahead and not yet pushed to libssh using rareq deque

	std::deque<rahead> rareq; ///< background running requests not yet fetched
	U_I rareq_maxsize;

	char* rabuffer; ///< holds data retreived from the last sftp_aio_wait_read()
	U_I rallocated; ///< allocated size of rabuffer
	U_I rasize;     ///< amount of bytes in rabuffer available for reading (rasize <= rallocated)
	U_I ralu;       ///< bytes of rabuffer already allocaed

	    // private methods

	void myclose();
	void clear_readahead() { rareq.clear(); ralu = rasize = 0; };
	void update_aio_reqs();

	    /// checks libssh is in sync of our current_pos

	    /// \note libssh is not expected to be in sync when we have
	    /// data in rabuffer, pending for reading. It is not expected
	    /// to be neither when we have on-fly async read in rareq queue
	void check_pos_from_libssh();

    };

#endif

	/// @}

} // end of namespace

#endif
