/* Copyright (C) 2023 Open Information Security Foundation
 *
 * You can copy, redistribute or modify this Program under the terms of
 * the GNU General Public License version 2 as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

use super::constant::{EnipCommand, EnipStatus};
use super::enip::EnipTransaction;
use super::parser::{
    cip_segment_type_string, CipData, CipDir, EnipCIP, EnipCipPathSegment, EnipCipRequestPayload,
    EnipCipResponsePayload, EnipHeader, EnipItemPayload, EnipPayload,
};
use crate::detect::EnumString;
use crate::jsonbuilder::{JsonBuilder, JsonError};
use std;

fn log_enip_header(h: &EnipHeader, js: &mut JsonBuilder) -> Result<(), JsonError> {
    if let Some(cmd) = EnipCommand::from_u(h.cmd) {
        js.set_string("command", cmd.to_str())?;
    } else {
        js.set_string("command", &format!("unknown-{}", h.cmd))?;
    }
    if let Some(status) = EnipStatus::from_u(h.status) {
        js.set_string("status", status.to_str())?;
    } else {
        js.set_string("status", &format!("unknown-{}", h.status))?;
    }
    if h.options != 0 {
        js.set_uint("options", h.options)?;
    }
    Ok(())
}

// These lists come from other open source programs like Wireshark, Zeek or scapy
fn enip_vendorid_string(p: u16) -> Option<&'static str> {
    match p {
        1 => Some("Rockwell Automation/Allen-Bradley"),
        2 => Some("Namco Controls Corp."),
        3 => Some("Honeywell Inc."),
        4 => Some("Parker Hannifin Corp. (Veriflo Division)"),
        5 => Some("Rockwell Automation/Reliance Elec."),
        7 => Some("SMC Corporation"),
        8 => Some("Molex Incorporated"),
        9 => Some("Western Reserve Controls Corp."),
        10 => Some("Advanced Micro Controls Inc. (AMCI)"),
        11 => Some("ASCO Pneumatic Controls"),
        12 => Some("Banner Engineering Corp."),
        13 => Some("Belden Wire & Cable Company"),
        14 => Some("Cooper Interconnect"),
        16 => Some("Daniel Woodhead Co. (Woodhead Connectivity)"),
        17 => Some("Dearborn Group Inc."),
        19 => Some("Helm Instrument Company"),
        20 => Some("Huron Net Works"),
        21 => Some("Lumberg Inc."),
        22 => Some("Online Development Inc.(Automation Value)"),
        23 => Some("Vorne Industries Inc."),
        24 => Some("ODVA Special Reserve"),
        26 => Some("Festo Corporation"),
        30 => Some("Unico Inc."),
        31 => Some("Ross Controls"),
        34 => Some("Hohner Corp."),
        35 => Some("Micro Mo Electronics Inc."),
        36 => Some("MKS Instruments Inc."),
        37 => Some("Yaskawa Electric America (formerly Magnetek Drives)"),
        39 => Some("AVG Automation (Uticor)"),
        40 => Some("Wago Corporation"),
        41 => Some("Kinetics (Unit Instruments)"),
        42 => Some("IMI Norgren Limited"),
        43 => Some("BALLUFF Inc."),
        44 => Some("Yaskawa Electric America Inc."),
        45 => Some("Eurotherm Controls Inc"),
        46 => Some("ABB Industrial Systems"),
        47 => Some("Omron Corporation"),
        48 => Some("Turck Inc."),
        49 => Some("Grayhill Inc."),
        50 => Some("Real Time Automation (C&ID)"),
        52 => Some("Numatics Inc."),
        53 => Some("Lutze Inc."),
        56 => Some("Softing GmbH"),
        57 => Some("Pepperl + Fuchs"),
        58 => Some("Spectrum Controls Inc."),
        59 => Some("D.I.P. Inc. MKS Inst."),
        60 => Some("Applied Motion Products Inc."),
        61 => Some("Sencon Inc."),
        62 => Some("High Country Tek"),
        63 => Some("SWAC Automation Consult GmbH"),
        64 => Some("Clippard Instrument Laboratory"),
        68 => Some("Eaton Electrical"),
        71 => Some("Toshiba International Corp."),
        72 => Some("Control Technology Incorporated"),
        73 => Some("TCS (NZ) Ltd."),
        74 => Some("Hitachi Ltd."),
        75 => Some("ABB Robotics Products AB"),
        76 => Some("NKE Corporation"),
        77 => Some("Rockwell Software Inc."),
        78 => Some("Escort Memory Systems (A Datalogic Group Co.)"),
        79 => Some("Berk-Tek"),
        80 => Some("Industrial Devices Corporation"),
        81 => Some("IXXAT Automation GmbH"),
        82 => Some("Mitsubishi Electric Automation Inc."),
        83 => Some("OPTO-22"),
        86 => Some("Horner Electric"),
        87 => Some("Burkert Werke GmbH & Co. KG"),
        88 => Some("Industrial Indexing Systems, Inc."),
        89 => Some("Industrial Indexing Systems Inc."),
        90 => Some("HMS Industrial Networks AB"),
        91 => Some("Robicon"),
        92 => Some("Helix Technology (Granville-Phillips)"),
        93 => Some("Arlington Laboratory"),
        94 => Some("Advantech Co. Ltd."),
        95 => Some("Square D Company"),
        96 => Some("Digital Electronics Corp."),
        97 => Some("Danfoss"),
        100 => Some("Bosch Rexroth Corporation, Pneumatics"),
        101 => Some("Applied Materials Inc."),
        102 => Some("Showa Electric Wire & Cable Co."),
        103 => Some("Pacific Scientific (API Controls Inc.)"),
        104 => Some("Sharp Manufacturing Systems Corp."),
        105 => Some("Olflex Wire & Cable Inc."),
        107 => Some("Unitrode"),
        108 => Some("Beckhoff Automation GmbH"),
        109 => Some("National Instruments"),
        110 => Some("Mykrolis Corporations (Millipore)"),
        111 => Some("International Motion Controls Corp."),
        113 => Some("SEG Kempen GmbH"),
        116 => Some("MTS Systems Corp."),
        117 => Some("Krones Inc"),
        119 => Some("EXOR Electronic R&D"),
        120 => Some("SIEI S.p.A."),
        121 => Some("KUKA Roboter GmbH"),
        123 => Some("SEC (Samsung Electronics Co. Ltd.)"),
        124 => Some("Binary Electronics Ltd"),
        125 => Some("Flexible Machine Controls"),
        127 => Some("ABB Inc. (Entrelec)"),
        128 => Some("MAC Valves Inc."),
        129 => Some("Auma Actuators Inc."),
        130 => Some("Toyoda Machine Works Ltd."),
        133 => Some("Balogh T.A.G. Corporation"),
        134 => Some("TR Systemtechnik GmbH"),
        135 => Some("UNIPULSE Corporation"),
        138 => Some("Conxall Corporation Inc."),
        141 => Some("Kuramo Electric Co. Ltd."),
        142 => Some("Creative Micro Designs"),
        143 => Some("GE Industrial Systems"),
        144 => Some("Leybold Vacuum GmbH"),
        145 => Some("Siemens Energy & Automation/Drives"),
        146 => Some("Kodensha Ltd."),
        147 => Some("Motion Engineering Inc."),
        148 => Some("Honda Engineering Co. Ltd."),
        149 => Some("EIM Valve Controls"),
        150 => Some("Melec Inc."),
        151 => Some("Sony Manufacturing Systems Corporation"),
        152 => Some("North American Mfg."),
        153 => Some("WATLOW"),
        154 => Some("Japan Radio Co. Ltd."),
        155 => Some("NADEX Co. Ltd."),
        156 => Some("Ametek Automation & Process Technologies"),
        157 => Some("FACTS, Inc."),
        158 => Some("KVASER AB"),
        159 => Some("IDEC IZUMI Corporation"),
        160 => Some("Mitsubishi Heavy Industries Ltd."),
        161 => Some("Mitsubishi Electric Corporation"),
        162 => Some("Horiba-STEC Inc."),
        163 => Some("ESD Electronic System Design GmbH"),
        164 => Some("DAIHEN Corporation"),
        165 => Some("Tyco Valves & Controls/Keystone"),
        166 => Some("EBARA Corporation"),
        169 => Some("Hokuyo Electric Co. Ltd."),
        170 => Some("Pyramid Solutions Inc."),
        171 => Some("Denso Wave Incorporated"),
        172 => Some("HLS Hard-Line Solutions Inc."),
        173 => Some("Caterpillar Inc."),
        174 => Some("PDL Electronics Ltd."),
        176 => Some("Red Lion Controls"),
        177 => Some("ANELVA Corporation"),
        178 => Some("Toyo Denki Seizo KK"),
        179 => Some("Sanyo Denki Co. Ltd."),
        180 => Some("Advanced Energy Japan K.K. (Aera Japan)"),
        181 => Some("Pilz GmbH & Co"),
        182 => Some("Marsh Bellofram-Bellofram PCD Division"),
        184 => Some("M-SYSTEM Co. Ltd."),
        185 => Some("Nissin Electric Co. Ltd."),
        186 => Some("Hitachi Metals Ltd."),
        187 => Some("Oriental Motor Company"),
        188 => Some("A&D Co. Ltd."),
        189 => Some("Phasetronics Inc."),
        190 => Some("Cummins Engine Company"),
        191 => Some("Deltron Inc."),
        192 => Some("Geneer Corporation"),
        193 => Some("Anatol Automation Inc."),
        196 => Some("Medar Inc."),
        197 => Some("Comdel Inc."),
        198 => Some("Advanced Energy Industries Inc."),
        200 => Some("DAIDEN Co. Ltd."),
        201 => Some("CKD Corporation"),
        202 => Some("Toyo Electric Corporation"),
        204 => Some("AuCom Electronics Ltd."),
        205 => Some("Shinko Electric Co. Ltd."),
        206 => Some("Vector Informatik GmbH"),
        208 => Some("Moog Inc."),
        209 => Some("Contemporary Controls"),
        210 => Some("Tokyo Sokki Kenkyujo Co. Ltd."),
        211 => Some("Schenck-AccuRate Inc."),
        212 => Some("The Oilgear Company"),
        214 => Some("ASM Japan K.K."),
        215 => Some("HIRATA Corp."),
        216 => Some("SUNX Limited"),
        217 => Some("Meidensha Corp."),
        218 => Some("NIDEC SANKYO CORPORATION (Sankyo Seiki Mfg. Co. Ltd.)"),
        219 => Some("KAMRO Corp."),
        220 => Some("Nippon System Development Co. Ltd."),
        221 => Some("EBARA Technologies Inc."),
        224 => Some("SG Co. Ltd."),
        225 => Some("Vaasa Institute of Technology"),
        226 => Some("MKS Instruments (ENI Technology)"),
        227 => Some("Tateyama System Laboratory Co. Ltd."),
        228 => Some("QLOG Corporation"),
        229 => Some("Matric Limited Inc."),
        230 => Some("NSD Corporation"),
        232 => Some("Sumitomo Wiring Systems Ltd."),
        233 => Some("Group 3 Technology Ltd."),
        234 => Some("CTI Cryogenics"),
        235 => Some("POLSYS CORP"),
        236 => Some("Ampere Inc."),
        238 => Some("Simplatroll Ltd."),
        241 => Some("Leading Edge Design"),
        242 => Some("Humphrey Products"),
        243 => Some("Schneider Automation Inc."),
        244 => Some("Westlock Controls Corp."),
        245 => Some("Nihon Weidmuller Co. Ltd."),
        246 => Some("Brooks Instrument (Div. of Emerson)"),
        248 => Some("Moeller GmbH"),
        249 => Some("Varian Vacuum Products"),
        250 => Some("Yokogawa Electric Corporation"),
        251 => Some("Electrical Design Daiyu Co. Ltd."),
        252 => Some("Omron Software Co. Ltd."),
        253 => Some("BOC Edwards"),
        254 => Some("Control Technology Corporation"),
        255 => Some("Bosch Rexroth"),
        256 => Some("Turck"),
        257 => Some("Control Techniques PLC"),
        258 => Some("Hardy Instruments Inc."),
        259 => Some("LS Industrial Systems"),
        260 => Some("E.O.A. Systems Inc."),
        262 => Some("New Cosmos Electric Co. Ltd."),
        263 => Some("Sense Eletronica LTDA"),
        264 => Some("Xycom Inc."),
        265 => Some("Baldor Electric"),
        267 => Some("Patlite Corporation"),
        269 => Some("Mogami Wire & Cable Corporation"),
        270 => Some("Welding Technology Corporation (WTC)"),
        272 => Some("Deutschmann Automation GmbH"),
        273 => Some("ICP Panel-Tec Inc."),
        274 => Some("Bray Controls USA"),
        276 => Some("Status Technologies"),
        277 => Some("Trio Motion Technology Ltd."),
        278 => Some("Sherrex Systems Ltd."),
        279 => Some("Adept Technology Inc."),
        280 => Some("Spang Power Electronics"),
        282 => Some("Acrosser Technology Co. Ltd."),
        283 => Some("Hilscher GmbH"),
        284 => Some("IMAX Corporation"),
        285 => Some("Electronic Innovation Inc. (Falter Engineering)"),
        286 => Some("Netlogic Inc."),
        287 => Some("Bosch Rexroth Corporation, Indramat"),
        290 => Some("Murata  Machinery Ltd."),
        291 => Some("MTT Company Ltd."),
        292 => Some("Kanematsu Semiconductor Corp."),
        293 => Some("Takebishi Electric Sales Co."),
        294 => Some("Tokyo Electron Device Ltd."),
        295 => Some("PFU Limited"),
        296 => Some("Hakko Automation Co. Ltd."),
        297 => Some("Advanet Inc."),
        298 => Some("Tokyo Electron Software Technologies Ltd."),
        300 => Some("Shinagawa Electric Wire Co. Ltd."),
        301 => Some("Yokogawa M&C Corporation"),
        302 => Some("KONAN Electric Co. Ltd."),
        303 => Some("Binar Elektronik AB"),
        304 => Some("Furukawa Electric Co."),
        305 => Some("Cooper Energy Services"),
        306 => Some("Schleicher GmbH & Co."),
        307 => Some("Hirose Electric Co. Ltd."),
        308 => Some("Western Servo Design Inc."),
        309 => Some("Prosoft Technology"),
        311 => Some("Towa Shoko Co. Ltd."),
        312 => Some("Kyopal Co. Ltd."),
        313 => Some("Extron Co."),
        314 => Some("Wieland Electric GmbH"),
        315 => Some("SEW Eurodrive GmbH"),
        316 => Some("Aera Corporation"),
        317 => Some("STA Reutlingen"),
        319 => Some("Fuji Electric Co. Ltd."),
        322 => Some("ifm efector, inc."),
        324 => Some("IDEACOD-Hohner Automation S.A."),
        325 => Some("CommScope Inc."),
        326 => Some("GE Fanuc Automation North America Inc."),
        327 => Some("Matsushita Electric Industrial Co. Ltd."),
        328 => Some("Okaya Electronics Corporation"),
        329 => Some("KASHIYAMA Industries Ltd."),
        330 => Some("JVC"),
        331 => Some("Interface Corporation"),
        332 => Some("Grape Systems Inc."),
        334 => Some("KEBA AG"),
        335 => Some("Toshiba IT & Control Systems Corporation"),
        336 => Some("Sanyo Machine Works Ltd."),
        337 => Some("Vansco Electronics Ltd."),
        338 => Some("Dart Container Corp."),
        339 => Some("Livingston & Co. Inc."),
        340 => Some("Alfa Laval LKM A/S"),
        341 => Some("BF ENTRON Ltd. (British Federal)"),
        342 => Some("Bekaert Engineering NV"),
        343 => Some("Ferran  Scientific Inc."),
        344 => Some("KEBA AG"),
        345 => Some("Endress + Hauser"),
        346 => Some("Lincoln Electric Company"),
        347 => Some("ABB ALSTOM Power UK Ltd. (EGT)"),
        348 => Some("Berger Lahr GmbH"),
        350 => Some("Federal Signal Corp."),
        351 => Some("Kawasaki Robotics (USA) Inc."),
        352 => Some("Bently Nevada Corporation"),
        354 => Some("FRABA Posital GmbH"),
        355 => Some("Elsag Bailey Inc."),
        356 => Some("Fanuc Robotics America"),
        358 => Some("Surface Combustion Inc."),
        360 => Some("AILES Electronics Ind. Co. Ltd."),
        361 => Some("Wonderware Corporation"),
        362 => Some("Particle Measuring Systems Inc."),
        365 => Some("BITS Co. Ltd."),
        366 => Some("Japan Aviation Electronics Industry Ltd."),
        367 => Some("Keyence Corporation"),
        368 => Some("Kuroda Precision Industries Ltd."),
        369 => Some("Mitsubishi Electric Semiconductor Application"),
        370 => Some("Nippon Seisen Cable Ltd."),
        371 => Some("Omron ASO Co. Ltd."),
        372 => Some("Seiko Seiki Co. Ltd."),
        373 => Some("Sumitomo Heavy IndustriesLtd."),
        374 => Some("Tango Computer Service Corporation"),
        375 => Some("Technology Service Inc."),
        376 => Some("Toshiba Information Systems (Japan) Corporation"),
        377 => Some("TOSHIBA Schneider Inverter Corporation"),
        378 => Some("Toyooki Kogyo Co. Ltd."),
        379 => Some("XEBEC"),
        380 => Some("Madison Cable Corporation"),
        381 => Some("Hitati Engineering & Services Co. Ltd."),
        382 => Some("TEM-TECH Lab Co. Ltd."),
        383 => Some("International Laboratory Corporation"),
        384 => Some("Dyadic Systems Co. Ltd."),
        385 => Some("SETO Electronics Industry Co. Ltd."),
        386 => Some("Tokyo Electron Kyushu Limited"),
        387 => Some("KEI System Co. Ltd."),
        389 => Some("Asahi Engineering Co. Ltd."),
        390 => Some("Contrex Inc."),
        391 => Some("Paradigm Controls Ltd."),
        393 => Some("Ohm Electric Co. Ltd."),
        394 => Some("RKC Instrument Inc."),
        395 => Some("Suzuki Motor Corporation"),
        396 => Some("Custom Servo Motors Inc."),
        397 => Some("PACE Control Systems"),
        398 => Some("Selectron Systems AG"),
        400 => Some("LINTEC Co. Ltd."),
        401 => Some("Hitachi Cable Ltd."),
        402 => Some("BUSWARE Direct"),
        403 => Some("Eaton Electric B.V. (formerly Holec Holland N.V.)"),
        404 => Some("VAT Vakuumventile AG"),
        405 => Some("Scientific Technologies Incorporated"),
        406 => Some("Alfa Instrumentos Eletronicos Ltda."),
        407 => Some("TWK Elektronik GmbH"),
        408 => Some("ABB Welding Systems AB"),
        409 => Some("BYSTRONIC Maschinen AG"),
        410 => Some("Kimura Electric Co. Ltd."),
        411 => Some("Nissei Plastic Industrial Co. Ltd."),
        413 => Some("Kistler-Morse Corporation"),
        414 => Some("Proteous Industries Inc."),
        415 => Some("IDC Corporation"),
        416 => Some("Nordson Corporation"),
        417 => Some("Rapistan Systems"),
        418 => Some("LP-Elektronik GmbH"),
        419 => Some("GERBI & FASE S.p.A. (Fase Saldatura)"),
        420 => Some("Phoenix Digital Corporation"),
        421 => Some("Z-World Engineering"),
        422 => Some("Honda R&D Co. Ltd."),
        423 => Some("Bionics Instrument Co. Ltd."),
        424 => Some("Teknic Inc."),
        425 => Some("R.Stahl Inc."),
        427 => Some("Ryco Graphic Manufacturing Inc."),
        428 => Some("Giddings & Lewis Inc."),
        429 => Some("Koganei Corporation"),
        431 => Some("Nichigoh Communication Electric Wire Co. Ltd."),
        433 => Some("Fujikura Ltd."),
        434 => Some("AD Link Technology Inc."),
        435 => Some("StoneL Corporation"),
        436 => Some("Computer Optical Products Inc."),
        437 => Some("CONOS Inc."),
        438 => Some("Erhardt + Leimer GmbH"),
        439 => Some("UNIQUE Co. Ltd."),
        440 => Some("Roboticsware Inc."),
        441 => Some("Nachi Fujikoshi Corporation"),
        442 => Some("Hengstler GmbH"),
        443 => Some("Vacon Plc"),
        444 => Some("SUNNY GIKEN Inc."),
        445 => Some("Lenze Drive Systems GmbH"),
        446 => Some("CD Systems B.V."),
        447 => Some("FMT/Aircraft Gate Support Systems AB"),
        448 => Some("Axiomatic Technologies Corp"),
        449 => Some("Embedded System Products Inc."),
        451 => Some("Mencom Corporation"),
        452 => Some("Kollmorgen"),
        453 => Some("Matsushita Welding Systems Co. Ltd."),
        454 => Some("Dengensha Mfg. Co. Ltd."),
        455 => Some("Quinn Systems Ltd."),
        456 => Some("Tellima Technology Ltd."),
        457 => Some("MDT Software"),
        458 => Some("Taiwan Keiso Co. Ltd."),
        459 => Some("Pinnacle Systems"),
        460 => Some("Ascom Hasler Mailing Sys."),
        461 => Some("INSTRUMAR Limited"),
        463 => Some("Navistar International Transportation Corp."),
        464 => Some("Huettinger Elektronik GmbH + Co. KG"),
        465 => Some("OCM Technology Inc."),
        466 => Some("Professional Supply Inc."),
        467 => Some("Control Solutions"),
        468 => Some("Baumer IVO GmbH & Co. KG"),
        469 => Some("Worcester Controls Corporation"),
        470 => Some("Pyramid Technical Consultants Inc."),
        471 => Some("Eilersen Electric A/S"),
        472 => Some("Apollo Fire Detectors Limited"),
        473 => Some("Avtron Manufacturing Inc."),
        475 => Some("Tokyo Keiso Co. Ltd."),
        476 => Some("Daishowa Swiki Co. Ltd."),
        477 => Some("Kojima Instruments Inc."),
        478 => Some("Shimadzu Corporation"),
        479 => Some("Tatsuta Electric Wire & Cable Co. Ltd."),
        480 => Some("MECS Corporation"),
        481 => Some("Tahara Electric"),
        482 => Some("Koyo Electronics"),
        483 => Some("Clever Devices"),
        484 => Some("GCD Hardware & Software GmbH"),
        486 => Some("Miller Electric Mfg. Co."),
        487 => Some("GEA Tuchenhagen GmbH"),
        488 => Some("Riken Keiki Co., Ltd."),
        489 => Some("Keisokugiken Corporation"),
        490 => Some("Fuji Machine Mfg. Co. Ltd."),
        492 => Some("Nidec-Shimpo Corp."),
        493 => Some("UTEC Corporation"),
        494 => Some("Sanyo Electric Co. Ltd."),
        497 => Some("Okano Electric Wire Co. Ltd."),
        498 => Some("Shimaden Co. Ltd."),
        499 => Some("Teddington Controls Ltd"),
        501 => Some("VIPA GmbH"),
        502 => Some("Warwick Manufacturing Group"),
        503 => Some("Danaher Controls"),
        506 => Some("American Science & Engineering"),
        507 => Some("Accutron Controls International Inc."),
        508 => Some("Norcott Technologies Ltd."),
        509 => Some("TB Woods Inc."),
        510 => Some("Proportion-Air Inc."),
        511 => Some("SICK Stegmann GmbH"),
        513 => Some("Edwards Signaling"),
        514 => Some("Sumitomo Metal Industries Ltd."),
        515 => Some("Cosmo Instruments Co. Ltd."),
        516 => Some("Denshosha Co. Ltd."),
        517 => Some("Kaijo Corp."),
        518 => Some("Michiproducts Co. Ltd."),
        519 => Some("Miura Corporation"),
        520 => Some("TG Information Network Co. Ltd."),
        521 => Some("Fujikin Inc."),
        522 => Some("Estic Corp."),
        523 => Some("GS Hydraulic Sales"),
        524 => Some("Leuze Electronic GmbH & Co. KG"),
        525 => Some("MTE Limited"),
        526 => Some("Hyde Park Electronics Inc."),
        527 => Some("Pfeiffer Vacuum GmbH"),
        528 => Some("Cyberlogic Technologies"),
        529 => Some("OKUMA Corporation FA Systems Division"),
        531 => Some("Hitachi Kokusai Electric Co. Ltd."),
        532 => Some("SHINKO TECHNOS Co. Ltd."),
        533 => Some("Itoh Electric Co. Ltd."),
        534 => Some("Colorado Flow Tech Inc."),
        535 => Some("Love Controls Division/Dwyer Inst."),
        536 => Some("Alstom Drives and Controls"),
        537 => Some("The Foxboro Company"),
        538 => Some("Tescom Corporation"),
        540 => Some("Atlas Copco Controls UK"),
        542 => Some("Autojet Technologies"),
        543 => Some("Prima Electronics S.p.A."),
        544 => Some("PMA GmbH"),
        545 => Some("Shimafuji Electric Co. Ltd."),
        546 => Some("Oki Electric Industry Co. Ltd."),
        547 => Some("Kyushu Matsushita Electric Co. Ltd."),
        548 => Some("Nihon Electric Wire & Cable Co. Ltd."),
        549 => Some("Tsuken Electric Ind Co. Ltd."),
        550 => Some("Tamadic Co."),
        551 => Some("MAATEL SA"),
        552 => Some("OKUMA America"),
        553 => Some("Control Techniques PLC-NA"),
        554 => Some("TPC Wire & Cable"),
        555 => Some("ATI Industrial Automation"),
        556 => Some("Microcontrol (Australia) Pty Ltd."),
        557 => Some("Serra Soldadura, S.A."),
        558 => Some("Southwest Research Institute"),
        559 => Some("Cabinplant International"),
        560 => Some("Sartorius Mechatronics T&H GmbH"),
        561 => Some("Comau S.p.A. Robotics & Final Assembly Division"),
        562 => Some("Phoenix Contact"),
        563 => Some("Yokogawa MAT Corporation"),
        564 => Some("Asahi Sangyo Co., Ltd."),
        566 => Some("Akita Myotoku Ltd."),
        567 => Some("OBARA Corp."),
        568 => Some("Suetron Electronic GmbH"),
        570 => Some("Serck Controls Limited"),
        571 => Some("Fairchild Industrial Products Company"),
        572 => Some("ARO S.A."),
        573 => Some("M2C GmbH"),
        574 => Some("Shin Caterpillar Mitsubishi Ltd."),
        575 => Some("Santest Co. Ltd."),
        576 => Some("Cosmotechs Co. Ltd."),
        577 => Some("Hitachi Electric Systems"),
        578 => Some("Smartscan Ltd."),
        579 => Some("Woodhead Software & Electronics France"),
        580 => Some("Athena Controls Inc."),
        581 => Some("Syron Engineering & Manufacturing Inc."),
        582 => Some("Asahi Optical Co. Ltd."),
        583 => Some("Sansha Electric Mfg. Co. Ltd."),
        584 => Some("Nikki Denso Co. Ltd."),
        585 => Some("Star Micronics => Some( Co. Ltd."),
        586 => Some("Ecotecnia Socirtat Corp."),
        587 => Some("AC Technology Corp."),
        588 => Some("West Instruments Limited"),
        589 => Some("NTI Limited"),
        590 => Some("Delta Computer Systems Inc."),
        591 => Some("FANUC Ltd."),
        592 => Some("Hearn-Gu Lee"),
        593 => Some("ABB Automation Products"),
        594 => Some("Orion Machinery Co. Ltd."),
        596 => Some("Wire-Pro Inc."),
        597 => Some("Beijing Huakong Technology Co. Ltd."),
        598 => Some("Yokoyama Shokai Co. Ltd."),
        599 => Some("Toyogiken Co. Ltd."),
        600 => Some("Coester Equipamentos Eletronicos Ltda."),
        601 => Some("Kawasaki Heavy Industries, Ltd."),
        602 => Some("Electroplating Engineers of Japan Ltd."),
        603 => Some("ROBOX S.p.A."),
        604 => Some("Spraying Systems Company"),
        605 => Some("Benshaw Inc."),
        606 => Some("ZPA-DP A.S."),
        607 => Some("Wired Rite Systems"),
        608 => Some("Tandis Research Inc."),
        609 => Some("SSD Drives GmbH"),
        610 => Some("ULVAC Japan Ltd."),
        611 => Some("DYNAX Corporation"),
        612 => Some("Nor-Cal Products Inc."),
        613 => Some("Aros Electronics AB"),
        614 => Some("Jun-Tech Co. Ltd."),
        615 => Some("HAN-MI Co. Ltd."),
        616 => Some("uniNtech (formerly SungGi Internet)"),
        617 => Some("Hae Pyung Electronics Research Institute"),
        618 => Some("Milwaukee Electronics"),
        619 => Some("OBERG Industries"),
        620 => Some("Parker Hannifin/Compumotor Division"),
        621 => Some("TECHNO DIGITAL CORPORATION"),
        622 => Some("Network Supply Co. Ltd."),
        623 => Some("Union Electronics Co. Ltd."),
        624 => Some("Tritronics Services PM Ltd."),
        625 => Some("Rockwell Automation-Sprecher+Schuh"),
        626 => Some("Matsushita Electric Industrial Co. Ltd. Motor Co."),
        627 => Some("Rolls-Royce Energy Systems Inc."),
        628 => Some("Jeongil Intercom Co., Ltd."),
        629 => Some("Interroll Corp."),
        630 => Some("Hubbell Wiring Device-Kellems (Delaware)"),
        631 => Some("Intelligent Motion Systems"),
        633 => Some("INFICON AG"),
        634 => Some("Hirschmann Inc."),
        635 => Some("The Siemon Company"),
        636 => Some("YAMAHA Motor Co. Ltd."),
        637 => Some("aska corporation"),
        638 => Some("Woodhead Connectivity"),
        639 => Some("Trimble AB"),
        640 => Some("Murrelektronik GmbH"),
        641 => Some("Creatrix Labs Inc."),
        642 => Some("TopWorx"),
        643 => Some("Kumho Industrial Co. Ltd."),
        644 => Some("Wind River Systems Inc."),
        645 => Some("Bihl & Wiedemann GmbH"),
        646 => Some("Harmonic Drive Systems Inc."),
        647 => Some("Rikei Corporation"),
        648 => Some("BL Autotec Ltd."),
        649 => Some("Hana Information & Technology Co. Ltd."),
        650 => Some("Seoil Electric Co. Ltd."),
        651 => Some("Fife Corporation"),
        652 => Some("Shanghai Electrical Apparatus Research Institute"),
        653 => Some("Detector Electronics"),
        654 => Some("Parasense Development Centre"),
        657 => Some("Six Tau S.p.A."),
        658 => Some("Aucos GmbH"),
        659 => Some("Rotork Controls"),
        660 => Some("Automationdirect.com"),
        661 => Some("Thermo BLH"),
        662 => Some("System Controls Ltd."),
        663 => Some("Univer S.p.A."),
        664 => Some("MKS-Tenta Technology"),
        665 => Some("Lika Electronic SNC"),
        666 => Some("Mettler-Toledo Inc."),
        667 => Some("DXL USA Inc."),
        668 => Some("Rockwell Automation/Entek IRD Intl."),
        669 => Some("Nippon Otis Elevator Company"),
        670 => Some("Sinano Electric => Some( Co. Ltd."),
        671 => Some("Sony Manufacturing Systems"),
        673 => Some("Contec Co. Ltd."),
        674 => Some("Automated Solutions"),
        675 => Some("Controlweigh"),
        677 => Some("Fincor Electronics"),
        678 => Some("Cognex Corporation"),
        679 => Some("Qualiflow"),
        680 => Some("Weidmuller Inc."),
        681 => Some("Morinaga Milk Industry Co. Ltd."),
        682 => Some("Takagi Industrial Co. Ltd."),
        683 => Some("Wittenstein AG"),
        684 => Some("Sena Technologies Inc."),
        686 => Some("APV Products Unna"),
        687 => Some("Creator Teknisk Utvedkling AB"),
        689 => Some("Mibu Denki Industrial Co. Ltd."),
        690 => Some("Takamastsu Machineer Section"),
        691 => Some("Startco Engineering Ltd."),
        693 => Some("Holjeron"),
        694 => Some("ALCATEL High Vacuum Technology"),
        695 => Some("Taesan LCD Co. Ltd."),
        696 => Some("POSCON"),
        697 => Some("VMIC"),
        698 => Some("Matsushita Electric Works Ltd."),
        699 => Some("IAI Corporation"),
        700 => Some("Horst GmbH"),
        701 => Some("MicroControl GmbH & Co."),
        702 => Some("Leine & Linde AB"),
        704 => Some("EC Elettronica Srl"),
        705 => Some("VIT Software HB"),
        706 => Some("Bronkhorst High-Tech B.V."),
        707 => Some("Optex Co. Ltd."),
        708 => Some("Yosio Electronic Co."),
        709 => Some("Terasaki Electric Co. Ltd."),
        710 => Some("Sodick Co. Ltd."),
        711 => Some("MTS Systems Corporation-Automation Division"),
        712 => Some("Mesa Systemtechnik"),
        713 => Some("SHIN HO SYSTEM Co. Ltd."),
        714 => Some("Goyo Electronics Co. Ltd."),
        715 => Some("Loreme"),
        716 => Some("SAB Brockskes GmbH & Co. KG"),
        717 => Some("Trumpf Laser GmbH + Co. KG"),
        718 => Some("Niigata Electronic Instruments Co. Ltd."),
        719 => Some("Yokogawa Digital Computer Corporation"),
        720 => Some("O.N. Electronic Co. Ltd."),
        721 => Some("Industrial Control  Communication Inc."),
        722 => Some("ABB Inc."),
        723 => Some("ElectroWave USA Inc."),
        724 => Some("Industrial Network Controls, LC"),
        725 => Some("KDT Systems Co. Ltd."),
        726 => Some("SEFA Technology Inc."),
        727 => Some("Nippon POP Rivets and Fasteners Ltd."),
        728 => Some("Yamato Scale Co. Ltd."),
        729 => Some("Zener Electric"),
        730 => Some("GSE Scale Systems"),
        731 => Some("ISAS (Integrated Switchgear & Sys. Pty Ltd.)"),
        732 => Some("Beta LaserMike Limited"),
        733 => Some("TOEI Electric Co. Ltd."),
        734 => Some("Hakko Electronics Co. Ltd"),
        736 => Some("RFID Inc."),
        737 => Some("Adwin Corporation"),
        738 => Some("Osaka Vacuum Ltd."),
        739 => Some("A-Kyung Motion Inc."),
        740 => Some("Camozzi S.P.A."),
        741 => Some("Crevis Co., Ltd."),
        742 => Some("Rice Lake Weighing Systems"),
        743 => Some("Linux Network Services"),
        744 => Some("KEB Antriebstechnik GmbH"),
        745 => Some("Hagiwara Electric Co. Ltd."),
        746 => Some("Glass Inc. International"),
        748 => Some("DVT Corporation"),
        749 => Some("Woodward Governor"),
        750 => Some("Mosaic Systems Inc."),
        751 => Some("Laserline GmbH"),
        752 => Some("COM-TEC Inc."),
        753 => Some("Weed Instrument"),
        754 => Some("Prof-face European Technology Center"),
        755 => Some("Fuji Automation Co. Ltd."),
        756 => Some("Matsutame Co. Ltd."),
        757 => Some("Hitachi Via Mechanics Ltd."),
        758 => Some("Dainippon Screen Mfg. Co. Ltd."),
        759 => Some("FLS Automation A/S"),
        760 => Some("ABB Stotz Kontakt GmbH"),
        761 => Some("Technical Marine Service"),
        762 => Some("Advanced Automation Associates Inc."),
        763 => Some("Baumer Ident GmbH"),
        764 => Some("Tsubakimoto Chain Co."),
        766 => Some("Furukawa Co. Ltd."),
        767 => Some("Active Power"),
        768 => Some("CSIRO Mining Automation"),
        769 => Some("Matrix Integrated Systems"),
        770 => Some("Digitronic Automationsanlagen GmbH"),
        771 => Some("SICK STEGMANN Inc."),
        772 => Some("TAE-Antriebstechnik GmbH"),
        773 => Some("Electronic Solutions"),
        774 => Some("Rocon L.L.C."),
        775 => Some("Dijitized Communications Inc."),
        776 => Some("Asahi Organic Chemicals Industry Co. Ltd."),
        777 => Some("Hodensha"),
        778 => Some("Harting Inc. NA"),
        779 => Some("Kubler GmbH"),
        780 => Some("Yamatake Corporation"),
        781 => Some("JEOL"),
        782 => Some("Yamatake Industrial Systems Co.Ltd."),
        783 => Some("HAEHNE Elektronische Messgerate GmbH"),
        784 => Some("Ci Technologies Pty Ltd (for Pelamos Industries)"),
        785 => Some("N. SCHLUMBERGER & CIE"),
        786 => Some("Teijin Seiki Co. Ltd."),
        787 => Some("DAIKIN Industries Ltd."),
        788 => Some("RyuSyo Industrial Co. Ltd."),
        789 => Some("Saginomiya Seisakusho, Inc."),
        790 => Some("Seishin Engineering Co. Ltd."),
        791 => Some("Japan Support System Ltd."),
        792 => Some("Decsys"),
        793 => Some("Metronix Messgerate u. Elektronik GmbH"),
        794 => Some("ROPEX Industrie - Elektronik GmbH"),
        795 => Some("Vaccon Company Inc."),
        796 => Some("Siemens Energy & Automation Inc."),
        797 => Some("Ten X Technology Inc."),
        798 => Some("Tyco Electronics"),
        799 => Some("Delta Power Electronics Center"),
        800 => Some("Denker"),
        801 => Some("Autonics Corporation"),
        802 => Some("JFE Electronic Engineering Pty. Ltd."),
        804 => Some("Electro-Sensors Inc."),
        805 => Some("Digi International Inc."),
        806 => Some("Texas Instruments"),
        807 => Some("ADTEC Plasma Technology Co. Ltd."),
        808 => Some("SICK AG"),
        809 => Some("Ethernet Peripherals Inc."),
        810 => Some("Animatics Corporation"),
        812 => Some("Process Control Corporation"),
        813 => Some("SystemV. Inc."),
        814 => Some("Danaher Motion SRL"),
        815 => Some("SHINKAWA Sensor Technology Inc."),
        816 => Some("Tesch GmbH & Co. KG"),
        818 => Some("Trend Controls Systems Ltd."),
        819 => Some("Guangzhou ZHIYUAN Electronic Co. Ltd."),
        820 => Some("Mykrolis Corporation"),
        821 => Some("Bethlehem Steel Corporation"),
        822 => Some("KK ICP"),
        823 => Some("Takemoto Denki Corporation"),
        824 => Some("The Montalvo Corporation"),
        826 => Some("LEONI Special Cables GmbH"),
        828 => Some("ONO SOKKI CO., LTD."),
        829 => Some("Rockwell Samsung Automation"),
        830 => Some("SHINDENGEN ELECTRIC MFG. CO. LTD"),
        831 => Some("Origin Electric Co. Ltd."),
        832 => Some("Quest Technical Solutions Inc."),
        833 => Some("LS CableLtd."),
        834 => Some("Enercon-Nord Electronic GmbH"),
        835 => Some("Northwire Inc."),
        836 => Some("Engel Elektroantriebe GmbH"),
        837 => Some("The Stanley Works"),
        838 => Some("Celesco Transducer Products Inc."),
        839 => Some("Chugoku Electric Wire and Cable Co."),
        840 => Some("Kongsberg Simrad AS"),
        841 => Some("Panduit Corporation"),
        842 => Some("Spellman High Voltage Electronics Corp."),
        843 => Some("Kokusai Electric Alpha Co. Ltd."),
        844 => Some("Brooks Automation Inc."),
        845 => Some("ANYWIRE CORPORATION"),
        846 => Some("Honda Electronics Co. Ltd"),
        847 => Some("REO Elektronik AG"),
        848 => Some("Fusion UV Systems Inc."),
        849 => Some("ASI Advanced Semiconductor Instruments GmbH"),
        850 => Some("Datalogic Inc."),
        851 => Some("SoftPLC Corporation"),
        852 => Some("Dynisco Instruments LLC"),
        853 => Some("WEG Industrias SA"),
        854 => Some("Frontline Test Equipment Inc."),
        855 => Some("Tamagawa Seiki Co. Ltd."),
        856 => Some("Multi Computing Co. Ltd."),
        857 => Some("RVSI"),
        858 => Some("Commercial Timesharing Inc."),
        859 => Some("Tennessee Rand Automation LLC"),
        860 => Some("Wacogiken Co. Ltd"),
        861 => Some("Reflex Integration Inc."),
        862 => Some("Siemens AG, A&D PI Flow Instruments"),
        863 => Some("G. Bachmann Electronic GmbH"),
        864 => Some("NT International"),
        865 => Some("Schweitzer Engineering Laboratories"),
        866 => Some("ATR Industrie-Elektronik GmbH Co."),
        867 => Some("PLASMATECH Co. Ltd."),
        869 => Some("GEMU GmbH & Co. KG"),
        870 => Some("Alcorn McBride Inc."),
        871 => Some("Mori Seiki Co., Ltd."),
        872 => Some("NodeTech Systems Ltd."),
        873 => Some("Emhart Teknologies"),
        874 => Some("Cervis Inc."),
        875 => Some("FieldServer Technologies (Div Sierra Monitor Corp)"),
        876 => Some("NEDAP Power Supplies"),
        877 => Some("Nippon Sanso Corporation"),
        878 => Some("Mitomi Giken Co. Ltd."),
        879 => Some("PULS GmbH"),
        881 => Some("Japan Control Engineering Ltd."),
        882 => Some("Embedded Systems Korea (Former Zues Emtek Co Ltd.)"),
        883 => Some("Automa SRL"),
        884 => Some("Harms+Wende GmbH & Co KG"),
        885 => Some("SAE-STAHL GmbH"),
        886 => Some("Microwave Data Systems"),
        887 => Some("Bernecker + Rainer Industrie-Elektronik GmbH"),
        888 => Some("Hiprom Technologies"),
        890 => Some("Nitta Corporation"),
        891 => Some("Kontron Modular Computers GmbH"),
        892 => Some("Marlin Controls"),
        893 => Some("ELCIS s.r.l."),
        894 => Some("Acromag Inc."),
        895 => Some("Avery Weigh-Tronix"),
        899 => Some("Practicon Ltd"),
        900 => Some("Schunk GmbH & Co. KG"),
        901 => Some("MYNAH Technologies"),
        902 => Some("Defontaine Groupe"),
        903 => Some("Emerson Process Management Power & Water Solutions"),
        904 => Some("F.A. Elec"),
        905 => Some("Hottinger Baldwin Messtechnik GmbH"),
        906 => Some("Coreco Imaging Inc."),
        907 => Some("London Electronics Ltd."),
        908 => Some("HSD SpA"),
        909 => Some("Comtrol Corporation"),
        910 => Some("TEAM => Some( S.A. (Tecnica Electronica de Automatismo Y Medida)"),
        911 => Some("MAN B&W Diesel Ltd. Regulateurs Europa"),
        914 => Some("Micro Motion Inc."),
        915 => Some("Eckelmann AG"),
        916 => Some("Hanyoung Nux"),
        917 => Some("Ransburg Industrial Finishing KK"),
        918 => Some("Kun Hung Electric Co. Ltd."),
        919 => Some("Brimos wegbebakening b.v."),
        920 => Some("Nitto Seiki Co. Ltd."),
        921 => Some("PPT Vision Inc."),
        922 => Some("Yamazaki Machinery Works"),
        923 => Some("SCHMIDT Technology GmbH"),
        924 => Some("Parker Hannifin SpA (SBC Division)"),
        925 => Some("HIMA Paul Hildebrandt GmbH"),
        926 => Some("RivaTek Inc."),
        927 => Some("Misumi Corporation"),
        928 => Some("GE Multilin"),
        929 => Some("Measurement Computing Corporation"),
        930 => Some("Jetter AG"),
        931 => Some("Tokyo Electronics Systems Corporation"),
        932 => Some("Togami Electric Mfg. Co. Ltd."),
        933 => Some("HK Systems"),
        934 => Some("CDA Systems Ltd."),
        935 => Some("Aerotech Inc."),
        936 => Some("JVL Industrie Elektronik A/S"),
        937 => Some("NovaTech Process Solutions LLC"),
        939 => Some("Cisco Systems"),
        940 => Some("Grid Connect"),
        941 => Some("ITW Automotive Finishing"),
        942 => Some("HanYang System"),
        943 => Some("ABB K.K. Technical Center"),
        944 => Some("Taiyo Electric Wire & Cable Co.Ltd."),
        946 => Some("SEREN IPS INC"),
        947 => Some("Belden CDT Electronics Division"),
        948 => Some("ControlNet International"),
        949 => Some("Gefran S.P.A."),
        950 => Some("Jokab Safety AB"),
        951 => Some("SUMITA OPTICAL GLASS => Some( INC."),
        952 => Some("Biffi Italia srl"),
        953 => Some("Beck IPC GmbH"),
        954 => Some("Copley Controls Corporation"),
        955 => Some("Fagor Automation S. Coop."),
        956 => Some("DARCOM"),
        957 => Some("Frick Controls (div. of York International)"),
        958 => Some("SymCom Inc."),
        959 => Some("Infranor"),
        960 => Some("Kyosan CableLtd."),
        961 => Some("Varian Vacuum Technologies"),
        962 => Some("Messung Systems"),
        963 => Some("Xantrex Technology Inc."),
        964 => Some("StarThis Inc."),
        965 => Some("Chiyoda Co.Ltd."),
        966 => Some("Flowserve Corporation"),
        967 => Some("Spyder Controls Corp."),
        968 => Some("IBA AG"),
        969 => Some("SHIMOHIRA ELECTRIC MFG.CO.,LTD"),
        971 => Some("Siemens L&A"),
        972 => Some("Micro Innovations AG"),
        973 => Some("Switchgear & Instrumentation"),
        974 => Some("PRE-TECH CO. => Some( LTD."),
        975 => Some("National Semiconductor"),
        976 => Some("Invensys Process Systems"),
        977 => Some("Ametek HDR Power Systems"),
        979 => Some("TETRA-K Corporation"),
        980 => Some("C & M Corporation"),
        981 => Some("Siempelkamp Maschinen"),
        983 => Some("Daifuku America Corporation"),
        984 => Some("Electro-Matic Products Inc."),
        985 => Some("BUSSAN MICROELECTRONICS CORP."),
        986 => Some("ELAU AG"),
        987 => Some("Hetronic USA"),
        988 => Some("NIIGATA POWER SYSTEMS Co.Ltd."),
        989 => Some("Software Horizons Inc."),
        990 => Some("B3 Systems Inc."),
        991 => Some("Moxa Networking Co.Ltd."),
        993 => Some("S4 Integration"),
        994 => Some("Elettro Stemi S.R.L."),
        995 => Some("AquaSensors"),
        996 => Some("Ifak System GmbH"),
        997 => Some("SANKEI MANUFACTURING Co.,LTD."),
        998 => Some("Emerson Network Power Co.Ltd."),
        999 => Some("Fairmount Automation Inc."),
        1000 => Some("Bird Electronic Corporation"),
        1001 => Some("Nabtesco Corporation"),
        1002 => Some("AGM Electronics Inc."),
        1003 => Some("ARCX Inc."),
        1004 => Some("DELTA I/O Co."),
        1005 => Some("Chun IL Electric Ind. Co."),
        1006 => Some("N-Tron"),
        1007 => Some("Nippon Pneumatics/Fludics System CO., LTD."),
        1008 => Some("DDK Ltd."),
        1009 => Some("Seiko Epson Corporation"),
        1010 => Some("Halstrup-Walcher GmbH"),
        1011 => Some("ITT"),
        1012 => Some("Ground Fault Systems bv"),
        1013 => Some("Scolari Engineering S.p.A."),
        1014 => Some("Vialis Traffic bv"),
        1015 => Some("Weidmueller Interface GmbH & Co. KG"),
        1016 => Some("Shanghai Sibotech Automation Co. Ltd."),
        1017 => Some("AEG Power Supply Systems GmbH"),
        1018 => Some("Komatsu Electronics Inc."),
        1019 => Some("Souriau"),
        1020 => Some("Baumuller Chicago Corp."),
        1021 => Some("J. Schmalz GmbH"),
        1022 => Some("SEN Corporation"),
        1023 => Some("Korenix Technology Co. Ltd"),
        1024 => Some("Cooper Power Tools"),
        1025 => Some("INNOBIS"),
        1026 => Some("Shinho System"),
        1027 => Some("Xm Services Ltd."),
        1028 => Some("KVC Co.Ltd."),
        1029 => Some("Sanyu Seiki Co.Ltd."),
        1030 => Some("TuxPLC"),
        1031 => Some("Northern Network Solutions"),
        1032 => Some("Converteam GmbH"),
        1033 => Some("Symbol Technologies"),
        1034 => Some("S-TEAM Lab"),
        1035 => Some("Maguire Products Inc."),
        1036 => Some("AC&T"),
        1037 => Some("MITSUBISHI HEAVY INDUSTRIES, LTD. KOBE SHIPYARD & MACHINERY WORKS"),
        1038 => Some("Hurletron Inc."),
        1039 => Some("Chunichi Denshi Co.Ltd"),
        1040 => Some("Cardinal Scale Mfg. Co."),
        1041 => Some("BTR NETCOM via RIA Connect Inc."),
        1042 => Some("Base2"),
        1043 => Some("ASRC Aerospace"),
        1044 => Some("Beijing Stone Automation"),
        1045 => Some("Changshu Switchgear Manufacture Ltd."),
        1046 => Some("METRONIX Corp."),
        1047 => Some("WIT"),
        1048 => Some("ORMEC Systems Corp."),
        1049 => Some("ASATech (China) Inc."),
        1050 => Some("Controlled Systems Limited"),
        1051 => Some("Mitsubishi Heavy Ind. Digital System Co.Ltd. (M.H.I.)"),
        1052 => Some("Electrogrip"),
        1053 => Some("TDS Automation"),
        1054 => Some("T&C Power Conversion Inc."),
        1055 => Some("Robostar Co.Ltd"),
        1056 => Some("Scancon A/S"),
        1057 => Some("Haas Automation Inc."),
        1058 => Some("Eshed Technology"),
        1059 => Some("Delta Electronic Inc."),
        1060 => Some("Innovasic Semiconductor"),
        1061 => Some("SoftDEL Systems Limited"),
        1062 => Some("FiberFin Inc."),
        1063 => Some("Nicollet Technologies Corp."),
        1064 => Some("B.F. Systems"),
        1065 => Some("Empire Wire and Supply LLC"),
        1066 => Some("ENDO KOGYO CO., LTD"),
        1067 => Some("Elmo Motion Control LTD"),
        1069 => Some("Asahi Keiki Co. Ltd."),
        1070 => Some("Joy Mining Machinery"),
        1071 => Some("MPM Engineering Ltd."),
        1072 => Some("Wolke Inks & Printers GmbH"),
        1073 => Some("Mitsubishi Electric Engineering Co. Ltd."),
        1074 => Some("COMET AG"),
        1075 => Some("Real Time Objects & Systems, LLC"),
        1076 => Some("MISCO Refractometer"),
        1077 => Some("JT Engineering Inc."),
        1078 => Some("Automated Packing Systems"),
        1079 => Some("Niobrara R&D Corp."),
        1080 => Some("Garmin Ltd."),
        1081 => Some("Japan Mobile Platform Co. Ltd"),
        1082 => Some("Advosol Inc."),
        1083 => Some("ABB Global Services Limited"),
        1084 => Some("Sciemetric Instruments Inc."),
        1085 => Some("Tata Elxsi Ltd."),
        1086 => Some("TPC Mechatronics => Some( Co. Ltd."),
        1087 => Some("Cooper Bussmann"),
        1088 => Some("Trinite Automatisering B.V."),
        1089 => Some("Peek Traffic B.V."),
        1090 => Some("Acrison Inc."),
        1091 => Some("Applied Robotics Inc."),
        1092 => Some("FireBus Systems Inc."),
        1093 => Some("Beijing Sevenstar Huachuang Electronics"),
        1094 => Some("Magnetek"),
        1095 => Some("Microscan"),
        1096 => Some("Air Water Inc."),
        1097 => Some("Sensopart Industriesensorik GmbH"),
        1098 => Some("Tiefenbach Control Systems GmbH"),
        1099 => Some("INOXPA S.A"),
        1100 => Some("Zurich University of Applied Sciences"),
        1101 => Some("Ethernet Direct"),
        1102 => Some("GSI-Micro-E Systems"),
        1103 => Some("S-Net Automation Co.Ltd."),
        1104 => Some("Power Electronics S.L."),
        1105 => Some("Renesas Technology Corp."),
        1106 => Some("NSWCCD-SSES"),
        1107 => Some("Porter Engineering Ltd."),
        1108 => Some("Meggitt Airdynamics Inc."),
        1109 => Some("Inductive Automation"),
        1110 => Some("Neural ID"),
        1111 => Some("EEPod LLC"),
        1112 => Some("Hitachi Industrial Equipment Systems Co. Ltd."),
        1113 => Some("Salem Automation"),
        1114 => Some("port GmbH"),
        1115 => Some("B & PLUS"),
        1116 => Some("Graco Inc."),
        1117 => Some("Altera Corporation"),
        1118 => Some("Technology Brewing Corporation"),
        1121 => Some("CSE Servelec"),
        1124 => Some("Fluke Networks"),
        1125 => Some("Tetra Pak Packaging Solutions SPA"),
        1126 => Some("Racine Federated Inc."),
        1127 => Some("Pureron Japan Co. Ltd."),
        1130 => Some("Brother Industries Ltd."),
        1132 => Some("Leroy Automation"),
        1134 => Some("THK CO., LTD."),
        1137 => Some("TR-Electronic GmbH"),
        1138 => Some("ASCON S.p.A."),
        1139 => Some("Toledo do Brasil Industria de Balancas Ltda."),
        1140 => Some("Bucyrus DBT Europe GmbH"),
        1141 => Some("Emerson Process Management Valve Automation"),
        1142 => Some("Alstom Transport"),
        1144 => Some("Matrox Electronic Systems"),
        1145 => Some("Littelfuse"),
        1146 => Some("PLASMART Inc."),
        1147 => Some("Miyachi Corporation"),
        1150 => Some("Promess Incorporated"),
        1151 => Some("COPA-DATA GmbH"),
        1152 => Some("Precision Engine Controls Corporation"),
        1153 => Some("Alga Automacao e controle LTDA"),
        1154 => Some("U.I. Lapp GmbH"),
        1155 => Some("ICES"),
        1156 => Some("Philips Lighting bv"),
        1157 => Some("Aseptomag AG"),
        1158 => Some("ARC Informatique"),
        1159 => Some("Hesmor GmbH"),
        1160 => Some("Kobe SteelLtd."),
        1161 => Some("FLIR Systems"),
        1162 => Some("Simcon A/S"),
        1163 => Some("COPALP"),
        1164 => Some("Zypcom Inc."),
        1165 => Some("Swagelok"),
        1166 => Some("Elspec"),
        1167 => Some("ITT Water & Wastewater AB"),
        1168 => Some("Kunbus GmbH Industrial Communication"),
        1170 => Some("Performance Controls Inc."),
        1171 => Some("ACS Motion ControlLtd."),
        1173 => Some("IStar Technology Limited"),
        1174 => Some("Alicat Scientific Inc."),
        1176 => Some("ADFweb.com SRL"),
        1177 => Some("Tata Consultancy Services Limited"),
        1178 => Some("CXR Ltd."),
        1179 => Some("Vishay Nobel AB"),
        1181 => Some("SolaHD"),
        1182 => Some("Endress+Hauser"),
        1183 => Some("Bartec GmbH"),
        1185 => Some("AccuSentry Inc."),
        1186 => Some("Exlar Corporation"),
        1187 => Some("ILS Technology"),
        1188 => Some("Control Concepts Inc."),
        1190 => Some("Procon Engineering Limited"),
        1191 => Some("Hermary Opto Electronics Inc."),
        1192 => Some("Q-Lambda"),
        1194 => Some("VAMP Ltd"),
        1195 => Some("FlexLink"),
        1196 => Some("Office FA.com Co. Ltd."),
        1197 => Some("SPMC (Changzhou) Co. Ltd."),
        1198 => Some("Anton Paar GmbH"),
        1199 => Some("Zhuzhou CSR Times Electric Co. Ltd."),
        1200 => Some("DeStaCo"),
        1201 => Some("Synrad Inc"),
        1202 => Some("Bonfiglioli Vectron GmbH"),
        1203 => Some("Pivotal Systems"),
        1204 => Some("TKSCT"),
        1205 => Some("Randy Nuernberger"),
        1206 => Some("CENTRALP"),
        1207 => Some("Tengen Group"),
        1208 => Some("OES Inc."),
        1209 => Some("Actel Corporation"),
        1210 => Some("Monaghan Engineering Inc."),
        1211 => Some("wenglor sensoric gmbh"),
        1212 => Some("HSA Systems"),
        1213 => Some("MK Precision Co.Ltd."),
        1214 => Some("Tappan Wire and Cable"),
        1215 => Some("Heinzmann GmbH & Co. KG"),
        1216 => Some("Process Automation International Ltd."),
        1217 => Some("Secure Crossing"),
        1218 => Some("SMA Railway Technology GmbH"),
        1219 => Some("FMS Force Measuring Systems AG"),
        1220 => Some("ABT Endustri Enerji Sistemleri Sanayi Tic. Ltd. Sti."),
        1221 => Some("MagneMotion Inc."),
        1222 => Some("STS Co.Ltd."),
        1223 => Some("MERAK SIC => Some( SA"),
        1224 => Some("ABOUNDI Inc."),
        1225 => Some("Rosemount Inc."),
        1226 => Some("GEA FES Inc."),
        1227 => Some("TMG Technologie und Engineering GmbH"),
        1228 => Some("embeX GmbH"),
        1229 => Some("GH Electrotermia, S.A."),
        1230 => Some("Tolomatic"),
        1231 => Some("Dukane"),
        1232 => Some("Elco (Tian Jin) Electronics Co.Ltd."),
        1233 => Some("Jacobs Automation"),
        1234 => Some("Noda Radio Frequency Technologies Co.Ltd."),
        1235 => Some("MSC Tuttlingen GmbH"),
        1236 => Some("Hitachi Cable Manchester"),
        1237 => Some("ACOREL SAS"),
        1238 => Some("Global Engineering Solutions Co.Ltd."),
        1239 => Some("ALTE Transportation, S.L."),
        1240 => Some("Penko Engineering B.V."),
        1241 => Some("Z-Tec Automation Systems Inc."),
        1242 => Some("ENTRON Controls LLC"),
        1243 => Some("Johannes Huebner Fabrik Elektrischer Maschinen GmbH"),
        1244 => Some("RF IDeas, Inc."),
        1245 => Some("Pentronic AB"),
        1246 => Some("SCA Schucker GmbH & Co. KG"),
        1247 => Some("TDK-Lambda"),
        1250 => Some("Altronic LLC"),
        1251 => Some("Siemens AG"),
        1252 => Some("Liebherr Transportation Systems GmbH & Co KG"),
        1254 => Some("SKF USA Inc."),
        1256 => Some("LMI Technologies"),
        1259 => Some("EN Technologies Inc."),
        1261 => Some("CEPHALOS Automatisierung mbH"),
        1262 => Some("Atronix Engineering, Inc."),
        1263 => Some("Monode Marking Products, Inc."),
        1265 => Some("Quabbin Wire & Cable Co., Inc."),
        1266 => Some("GPSat Systems Australia"),
        1269 => Some("Tri-Tronics Co., Inc."),
        1270 => Some("Rovema GmbH"),
        1272 => Some("IEP GmbH"),
        1277 => Some("Control Chief Corporation"),
        1280 => Some("Jacktek Systems Inc."),
        1282 => Some("PRIMES GmbH"),
        1283 => Some("Branson Ultrasonics"),
        1284 => Some("DEIF A/S"),
        1285 => Some("3S-Smart Software Solutions GmbH"),
        1287 => Some("Smarteye Corporation"),
        1288 => Some("Toshiba Machine"),
        1289 => Some("eWON"),
        1290 => Some("OFS"),
        1291 => Some("KROHNE"),
        1293 => Some("General Cable Industries, Inc."),
        1295 => Some("Kistler Instrumente AG"),
        1296 => Some("YJS Co., Ltd."),
        1301 => Some("Xylem Analytics Germany GmbH"),
        1302 => Some("Lenord, Bauer & Co. GmbH"),
        1303 => Some("Carlo Gavazzi Controls"),
        1304 => Some("Faiveley Transport"),
        1306 => Some("vMonitor"),
        1307 => Some("Kepware Technologies"),
        1308 => Some("duagon AG"),
        1310 => Some("Xylem Water Solutions"),
        1311 => Some("Automation Professionals, LLC"),
        1313 => Some("CEIA SpA"),
        1314 => Some("Marine Technologies LLC"),
        1315 => Some("Alphagate Automatisierungstechnik GmbH"),
        1316 => Some("Mecco Partners, LLC"),
        1317 => Some("LAP GmbH Laser Applikationen"),
        1318 => Some("ABB S.p.A. - SACE Division"),
        1319 => Some("ABB S.p.A. - SACE Division"),
        1322 => Some("Thermo Ramsey Inc."),
        1323 => Some("Helmholz GmbH & Co. KG"),
        1324 => Some("EUCHNER GmbH + Co. KG"),
        1325 => Some("AMK GmbH & Co. KG"),
        1326 => Some("Badger Meter"),
        1328 => Some("Fisher-Rosemount Systems, Inc."),
        1329 => Some("LJU Automatisierungstechnik GmbH"),
        1330 => Some("Fairbanks Scales, Inc."),
        1331 => Some("Imperx, Inc."),
        1332 => Some("FRONIUS International GmbH"),
        1333 => Some("Hoffman Enclosures"),
        1334 => Some("Elecsys Corporation"),
        1335 => Some("Bedrock Automation"),
        1336 => Some("RACO Manufacturing and Engineering"),
        1337 => Some("Hein Lanz Industrial Tech."),
        1338 => Some("Synopsys, Inc. (formerly Codenomicon)"),
        1341 => Some("Sensirion AG"),
        1342 => Some("SIKO GmbH"),
        1344 => Some("GRUNDFOS"),
        1346 => Some("Beijer Electronics Products AB"),
        1348 => Some("AIMCO"),
        1350 => Some("Coval Vacuum Managers"),
        1351 => Some("Powell Industries"),
        1353 => Some("IPDisplays"),
        1354 => Some("SCAIME SAS"),
        1355 => Some("Metal Work SpA"),
        1356 => Some("Telsonic AG"),
        1358 => Some("Hauch & Bach ApS"),
        1359 => Some("Pago AG"),
        1360 => Some("ULTIMATE Europe Transportation Equipment GmbH"),
        1362 => Some("Enovation Controls"),
        1363 => Some("Lake Cable LLC"),
        1367 => Some("Laird"),
        1368 => Some("Nanotec Electronic GmbH & Co. KG"),
        1369 => Some("SAMWON ACT Co., Ltd."),
        1370 => Some("Aparian Inc."),
        1371 => Some("Cosys Inc."),
        1372 => Some("Insight Automation Inc."),
        1374 => Some("FASTECH"),
        1375 => Some("K.A. Schmersal GmbH & Co. KG"),
        1377 => Some("Chromalox"),
        1378 => Some("SEIDENSHA ELECTRONICS CO., LTD"),
        1380 => Some("Don Electronics Ltd"),
        1381 => Some("burster gmbh & co kg"),
        1382 => Some("Unitronics (1989) (RG) LTD"),
        1383 => Some("OEM Technology Solutions"),
        1384 => Some("Allied Motion"),
        1385 => Some("Mitron Oy"),
        1386 => Some("Dengensha TOA"),
        1387 => Some("Systec Systemtechnik und Industrieautomation GmbH"),
        1389 => Some("Jenny Science AG"),
        1390 => Some("Baumer Optronic GmbH"),
        1391 => Some("Invertek Drives Ltd"),
        1392 => Some("High Grade Controls Corporation"),
        1394 => Some("Ishida Europe Limited"),
        1396 => Some("Actia Systems"),
        1398 => Some("Beijing Tiandi-Marco Electro-Hydraulic Control System Co., Ltd."),
        1399 => Some("Universal Robots A/S"),
        1401 => Some("Dialight"),
        1402 => Some("E-T-A Elektrotechnische Apparate GmbH"),
        1403 => Some("Kemppi Oy"),
        1404 => Some("Tianjin Geneuo Technology Co., Ltd."),
        1405 => Some("ORing Industrial Networking Corp."),
        1406 => Some("Benchmark Electronics"),
        1408 => Some("ELAP S.R.L."),
        1409 => Some("Applied Mining Technologies"),
        1410 => Some("KITZ SCT Corporation"),
        1411 => Some("VTEX Corporation"),
        1412 => Some("ESYSE GmbH Embedded Systems Engineering"),
        1413 => Some("Automation Controls"),
        1415 => Some("Cincinnati Test Systems"),
        1417 => Some("Zumbach Electronics Corp."),
        1418 => Some("Emerson Process Management"),
        1419 => Some("CCS Inc."),
        1420 => Some("Videojet, Inc."),
        1421 => Some("Zebra Technologies"),
        1422 => Some("Anritsu Infivis"),
        1423 => Some("Dimetix AG"),
        1424 => Some("General Measure (China)"),
        1425 => Some("Fortress Interlocks"),
        1427 => Some("Task Force Tips"),
        1428 => Some("SERVO-ROBOT INC."),
        1429 => Some("Flow Devices and Systems, Inc."),
        1430 => Some("nLIGHT, Inc."),
        1431 => Some("Microchip Technology Inc."),
        1432 => Some("DENT Instruments"),
        1433 => Some("CMC Industrial Electronics Ltd."),
        1434 => Some("Accutron Instruments Inc."),
        1435 => Some("Kaeser Kompressoren SE"),
        1436 => Some("Optoelectronics"),
        1437 => Some("Coherix, Inc."),
        1438 => Some("FLSmidth A/S"),
        1439 => Some("Kyland Corporation"),
        1440 => Some("Cole-Parmer Instrument Company"),
        1441 => Some("Wachendorff Automation GmbH & Co., KG"),
        1442 => Some("SMAC Moving Coil Actuators"),
        1444 => Some("PushCorp, Inc."),
        1445 => Some("Fluke Process Instruments GmbH"),
        1446 => Some("Mini Motor srl"),
        1447 => Some("I-CON Industry Tech."),
        1448 => Some("Grace Engineered Products, Inc."),
        1449 => Some("Zaxis Inc."),
        1450 => Some("Lumasense Technologies"),
        1451 => Some("Domino Printing"),
        1452 => Some("LightMachinery Inc"),
        1453 => Some("DEUTA-WERKE GmbH"),
        1454 => Some("Altus Sistemas de Automação S.A."),
        1455 => Some("Criterion NDT"),
        1456 => Some("InterTech Development Company"),
        1457 => Some("Action Labs, Incorporated"),
        1458 => Some("Perle Systems Limited"),
        1459 => Some("Utthunga Technologies Pvt Ltd."),
        1460 => Some("Dong IL Vision, Co., Ltd."),
        1461 => Some("Wipotec Wiege-und Positioniersysteme GmbH"),
        1462 => Some("Atos spa"),
        1463 => Some("Solartron Metrology LTD"),
        1464 => Some("Willowglen Systems Inc."),
        1465 => Some("Analog Devices"),
        1466 => Some("Power Electronics International, Inc."),
        1468 => Some("Campbell Wrapper Corporatio"),
        1469 => Some("Herkules-Resotec Elektronik GmbH"),
        1470 => Some("aignep spa"),
        1471 => Some("SHANGHAI CARGOA M.&E.EQUIPMENT CO. LTD"),
        1472 => Some("PMV Automation AB"),
        1473 => Some("K-Patents Oy"),
        1474 => Some("Dynatronix"),
        1475 => Some("Atop Technologies"),
        1476 => Some("Bitronics, LLC."),
        1477 => Some("Delta Tau Data Systems"),
        1478 => Some("WITZ Corporation"),
        1479 => Some("AUTOSOL"),
        1480 => Some("ADB Safegate"),
        1481 => Some("VersaBuilt, Inc"),
        1482 => Some("Visual Technologies, Inc."),
        1483 => Some("Artis GmbH"),
        1484 => Some("Reliance Electric Limited"),
        1485 => Some("Vanderlande"),
        1486 => Some("Packet Power"),
        1487 => Some("ima-tec gmbh"),
        1488 => Some("Vision Automation A/S"),
        1489 => Some("PROCENTEC BV"),
        1490 => Some("HETRONIK GmbH"),
        1491 => Some("Lanmark Controls Inc."),
        1492 => Some("profichip GmbH"),
        1493 => Some("flexlog GmbH"),
        1494 => Some("YUCHANGTECH"),
        1495 => Some("Dynapower Company"),
        1496 => Some("TAKIKAWA ENGINEERING"),
        1497 => Some("Ingersoll Rand"),
        1498 => Some("ASA-RT s.r.l"),
        1499 => Some("Trumpf Laser- und Systemtectechnik Gmbh"),
        1500 => Some("SYNTEC TECHNOLOGY CORPORATION COMPANY"),
        1501 => Some("Rinstrum"),
        1502 => Some("Symbotic LLC"),
        1503 => Some("GE Healthcare Life Sciences"),
        1504 => Some("BlueBotics SA"),
        1505 => Some("Dynapar Corporation"),
        1506 => Some("Blum-Novotest"),
        1507 => Some("CIMON"),
        1508 => Some("Dalian SeaSky Automation Co., Ltd."),
        1509 => Some("Rethink Robotics, Inc."),
        1510 => Some("Ingeteam"),
        1511 => Some("TOSEI ENGINEERING CORP."),
        1512 => Some("SAMSON AG"),
        1513 => Some("TGW Mechanics GmbH"),
        _ => None,
    }
}

fn enip_devicetype_string(p: u16) -> Option<&'static str> {
    match p {
        0 => Some("Generic Device (deprecated)"),
        1 => Some("Control Station (deprecated)"),
        2 => Some("AC Drive Device"),
        3 => Some("Motor Overload"),
        4 => Some("Limit Switch"),
        5 => Some("Inductive Proximity Switch"),
        6 => Some("Photoelectric Sensor"),
        7 => Some("General Purpose Discrete I/O"),
        8 => Some("Encoder (deprecated)"),
        9 => Some("Resolver"),
        10 => Some("General Purpose Analog I/O (deprecated)"),
        12 => Some("Communications Adapter"),
        13 => Some("Barcode Scanner (deprecated)"),
        14 => Some("Programmable Logic Controller"),
        16 => Some("Position Controller"),
        17 => Some("Weigh Scale (deprecated)"),
        18 => Some("Message Display (deprecated)"),
        19 => Some("DC Drive"),
        20 => Some("Servo Drives (deprecated)"),
        21 => Some("Contactor"),
        22 => Some("Motor Starter"),
        23 => Some("Softstart Starter"),
        24 => Some("Human-Machine Interface"),
        25 => Some("Pneumatic Valve(s) (deprecated)"),
        26 => Some("Mass Flow Controller"),
        27 => Some("Pneumatic Valve(s)"),
        28 => Some("Vacuum Pressure Gauge"),
        29 => Some("Process Control Value"),
        30 => Some("Residual Gas Analyzer"),
        31 => Some("DC Power Generator"),
        32 => Some("RF Power Generator"),
        33 => Some("Turbomolecular Vacuum Pump"),
        34 => Some("Encoder"),
        35 => Some("Safety Discrete I/O Device"),
        36 => Some("Fluid Flow Controller"),
        37 => Some("CIP Motion Drive"),
        38 => Some("CompoNet Repeater"),
        39 => Some("Mass Flow Controller Enhanced"),
        40 => Some("CIP Modbus Device"),
        41 => Some("CIP Modbus Translator"),
        42 => Some("Safety Analog I/O Device"),
        43 => Some("Generic Device (keyable)"),
        44 => Some("Managed Ethernet Switch"),
        45 => Some("CIP Motion Safety Drive Device"),
        46 => Some("Safety Drive Device"),
        47 => Some("CIP Motion Encoder"),
        48 => Some("CIP Motion Converter"),
        49 => Some("CIP Motion I/O"),
        50 => Some("ControlNet Physical Layer Component"),
        100 => Some("In-Sight 2000 Series"),
        200 => Some("Embedded Component"),
        150 => Some("PowerFlex 525"),
        773 => Some("DataMan Series Reader"),
        _ => None,
    }
}

fn cip_service_string(p: u8) -> Option<&'static str> {
    match p {
        0x01 => Some("Get Attributes All"),
        0x02 => Some("Set Attributes All"),
        0x03 => Some("Get Attribute List"),
        0x04 => Some("Set Attribute List"),
        0x05 => Some("Reset"),
        0x06 => Some("Start"),
        0x07 => Some("Stop"),
        0x08 => Some("Create"),
        0x09 => Some("Delete"),
        0x0A => Some("Multiple Service Packet"),
        0x0D => Some("Apply Attributes"),
        0x0E => Some("Get Attributes Single"),
        0x10 => Some("Set Attributes Single"),
        0x11 => Some("Find Next Object Instance"),
        0x15 => Some("Restore"),
        0x16 => Some("Save"),
        0x17 => Some("No Forward Open"),
        0x18 => Some("Get Member"),
        0x19 => Some("Set Member"),
        0x1A => Some("Insert Member"),
        0x1B => Some("Remove Member"),
        0x1C => Some("Group Sync"),
        0x1D => Some("Get Connection Point Member List"),
        // these next ones are class-specific
        0x4B => Some("Execute PCCC Service"),
        0x4C => Some("Read Data"),
        0x4D => Some("Write Data"),
        0x4E => Some("Read Write Modify Data"),
        0x52 => Some("Read Data Fragmented"),
        0x53 => Some("Write Data Fragmented"),
        0x54 => Some("Forward Open"),
        _ => None,
    }
}

fn cip_status_string(p: u8) -> Option<&'static str> {
    match p {
        0x00 => Some("Success"),
        0x01 => Some("Connection Failure"),
        0x02 => Some("Resource Unavailable"),
        0x03 => Some("Invalid Parameter Value"),
        0x04 => Some("Path Segment Error"),
        0x05 => Some("Path Destination Unknown"),
        0x06 => Some("Partial Transfer"),
        0x07 => Some("Connection Lost"),
        0x08 => Some("Service Not Supported"),
        0x09 => Some("Invalid Attribute Value"),
        0x0A => Some("Attribute List Error"),
        0x0B => Some("Already In Requested Mode/State"),
        0x0C => Some("Object State Conflict"),
        0x0D => Some("Object Already Exists"),
        0x0E => Some("Attribute Not Settable"),
        0x0F => Some("Privilege Violation"),
        0x10 => Some("Device State Conflict"),
        0x11 => Some("Reply Data Too Large"),
        0x12 => Some("Fragmentation Of A Primitive Value"),
        0x13 => Some("Not Enough Data"),
        0x14 => Some("Attribute Not Supported"),
        0x15 => Some("Too Much Data"),
        0x16 => Some("Object Does Not Exist"),
        0x17 => Some("Service Fragmentation Sequence Not In Progress"),
        0x18 => Some("No Stored Attribute Data"),
        0x19 => Some("Store Operation Failure"),
        0x1A => Some("Routing Failure, Request Packet Too Large"),
        0x1B => Some("Routing Failure, Response Packet Too Large"),
        0x1C => Some("Missing Attribute List Entry Data"),
        0x1D => Some("Invalid Attribute Value List"),
        0x1E => Some("Embedded Service Error"),
        0x1F => Some("Vendor Specific Error"),
        0x20 => Some("Invalid Parameter"),
        0x21 => Some("Write-Once Value Or Medium Already Written"),
        0x22 => Some("Invalid Reply Received"),
        0x23 => Some("Buffer Overflow"),
        0x24 => Some("Invalid Message Format"),
        0x25 => Some("Key Failure In Path"),
        0x26 => Some("Path Size Invalid"),
        0x27 => Some("Unexpected Attribute In List"),
        0x28 => Some("Invalid Member ID"),
        0x29 => Some("Member Not Settable"),
        0x2A => Some("Group 2 Only Server General Failure"),
        0x2B => Some("Unknown Modbus Error"),
        0x2C => Some("Attribute Not Gettable"),
        0xFF => Some("Still Processing"),
        _ => None,
    }
}

fn cip_status_extended_string(p: &[u8]) -> Option<&'static str> {
    if p.len() == 2 {
        let val = ((p[1] as u16) << 8) | (p[0] as u16);
        return match val {
        0x100 => Some( "Connection in use or duplicate Forward Open"),
        0x103 => Some( "Transport class and trigger combination not supported"),
        0x106 => Some( "Ownership conflict"),
        0x107 => Some( "Target connection not found"),
        0x108 => Some( "Invalid network connection parameter"),
        0x109 => Some( "Invalid connection size"),
        0x110 => Some( "Target for connection not configured"),
        0x111 => Some( "RPI not supported"),
        0x112 => Some( "RPI value(s) not acceptable"),
        0x113 => Some( "Out of connections"),
        0x114 => Some( "Vendor ID or product code mismatch"),
        0x115 => Some( "Device type mismatch"),
        0x116 => Some( "Revision mismatch"),
        0x117 => Some( "Invalid produced or consumed application path"),
        0x118 => Some( "Invalid or inconsistent configuration application path"),
        0x119 => Some( "Non-listen only connection not opened"),
        0x11A => Some( "Target object out of connections"),
        0x11B => Some( "RPI is smaller than the production inhibit time"),
        0x11C => Some( "Transport class not supported"),
        0x11D => Some( "Production trigger not supported"),
        0x11E => Some( "Direction not supported"),
        0x11F => Some( "Invalid O->T Fixed/Variable"),
        0x120 => Some( "Invalid T->O Fixed/Variable"),
        0x121 => Some( "Invalid O->T Priority"),
        0x122 => Some( "Invalid T->O Priority"),
        0x123 => Some( "Invalid O->T connection type"),
        0x124 => Some( "Invalid T->O connection type"),
        0x125 => Some( "Invalid O->T redundant owner"),
        0x126 => Some( "Invalid configuration size"),
        0x127 => Some( "Invalid O->T size"),
        0x128 => Some( "Invalid T->O size"),
        0x129 => Some( "Invalid configuration application path"),
        0x12A => Some( "Invalid consuming application path"),
        0x12B => Some( "Invalid producing application path"),
        0x12C => Some( "Configuration symbol does not exist"),
        0x12D => Some( "Consuming symbol does not exist"),
        0x12E => Some( "Producing symbol does not exist"),
        0x12F => Some( "Inconsistent application path combination"),
        0x130 => Some( "Inconsistent consume data format"),
        0x131 => Some( "Inconsistent produce data format"),
        0x132 => Some( "NULL ForwardOpen not supported"),
        0x203 => Some( "Connection timed out"),
        0x204 => Some( "Unconnected request timed out"),
        0x205 => Some( "Parameter error in unconnected request"),
        0x206 => Some( "Message too large for UnconnectedSend"),
        0x207 => Some( "Unconnected acknowledged without reply"),
        0x301 => Some( "No buffer memory available"),
        0x302 => Some( "Network bandwidth not available for data"),
        0x303 => Some( "No consumed connection ID filter available"),
        0x304 => Some( "Not configured to send scheduled priority data"),
        0x305 => Some( "Schedule signature mismatch"),
        0x306 => Some( "Schedule signature validation not possible"),
        0x311 => Some( "Port not available"),
        0x312 => Some( "Link address not valid"),
        0x315 => Some( "Invalid segment in connection path"),
        0x316 => Some( "ForwardClose connection path mismatch"),
        0x317 => Some( "Scheduling not specified"),
        0x318 => Some( "Link address to self invalid"),
        0x319 => Some( "Secondary resources unavailable"),
        0x31A => Some( "Rack connection already established"),
        0x31B => Some( "Module connection already established"),
        0x31C => Some( "Miscellaneous"),
        0x31D => Some( "Redundant connection mismatch"),
        0x31E => Some( "No more user configurable link consumer resources available in the producing module"),
        0x31F => Some( "No more user configurable link consumer resources configured in the producing module"),
        0x800 => Some( "Network link offline"),
        0x801 => Some( "Incompatible Multicast RPI"),
        0x802 => Some( "Invalid Safety Connection Size"),
        0x803 => Some( "Invalid Safety Connection Format"),
        0x804 => Some( "Invalid Time Correction Connection Parameters"),
        0x805 => Some( "Invalid Ping Interval EPI Multiplier"),
        0x806 => Some( "Time Coordination Msg Min Multiplier"),
        0x807 => Some( "Network Time Expectation Multiplier"),
        0x808 => Some( "Timeout Multiplier"),
        0x809 => Some( "Invalid Max Consumer Number"),
        0x80A => Some( "Invalid CPCRC"),
        0x80B => Some( "Time Correction Connection ID Invalid"),
        0x80C => Some( "SCID Mismatch"),
        0x80D => Some( "TUNID not set"),
        0x80E => Some( "TUNID Mismatch"),
        0x80F => Some( "Configuration operation not allowed"),
        0x810 => Some( "No target application data available"),
        0x811 => Some( "No originator application data available"),
        0x812 => Some( "Node address has changed since the network was scheduled"),
        0x813 => Some( "Not configured for off-subnet multicast"),
        0x814 => Some( "Invalid produce/consume data format"),
        _ => None,
        };
    }
    None
}

fn cip_class_string(p: u32) -> Option<&'static str> {
    match p {
        0x00 => Some(""),
        0x01 => Some("Identity"),
        0x02 => Some("Message Router"),
        0x03 => Some("DeviceNet"),
        0x04 => Some("Assembly"),
        0x05 => Some("Connection"),
        0x06 => Some("Connection Manager"),
        0x07 => Some("Register"),
        0x08 => Some("Discrete Input Point"),
        0x09 => Some("Discrete Output Point"),
        0x0A => Some("Analog Input Point"),
        0x0B => Some("Analog Output Point"),
        0x0E => Some("Presence Sensing"),
        0x0F => Some("Parameter"),
        0x10 => Some("Parameter Group"),
        0x12 => Some("Group"),
        0x1D => Some("Discrete Input Group"),
        0x1E => Some("Discrete Output Group"),
        0x1F => Some("Discrete Group"),
        0x20 => Some("Analog Input Group"),
        0x21 => Some("Analog Output Group"),
        0x22 => Some("Analog Group"),
        0x23 => Some("Position Sensor"),
        0x24 => Some("Position Controller Supervisor"),
        0x25 => Some("Position Controller"),
        0x26 => Some("Block Sequencer"),
        0x27 => Some("Command Block"),
        0x28 => Some("Motor Data"),
        0x29 => Some("Control Supervisor"),
        0x2A => Some("AC/DC Drive"),
        0x2B => Some("Acknowledge Handler"),
        0x2C => Some("Overload"),
        0x2D => Some("Softstart"),
        0x2E => Some("Selection"),
        0x30 => Some("S-Device Supervisor"),
        0x31 => Some("S-Analog Sensor"),
        0x32 => Some("S-Analog Actuator"),
        0x33 => Some("S-Single Stage Controller"),
        0x34 => Some("S-Gas Calibration"),
        0x35 => Some("Trip Point"),
        0x37 => Some("File"),
        0x38 => Some("S-Partial Pressure"),
        0x39 => Some("Safety Supervisor"),
        0x3A => Some("Safety Validator"),
        0x3B => Some("Safety Discrete Output Point"),
        0x3C => Some("Safety Discrete Output Group"),
        0x3D => Some("Safety Discrete Input Point"),
        0x3E => Some("Safety Discrete Input Group"),
        0x3F => Some("Safety Dual Channel Output"),
        0x40 => Some("S-Sensor Calibration"),
        0x41 => Some("Event Log"),
        0x42 => Some("Motion Device Axis"),
        0x43 => Some("Time Sync"),
        0x44 => Some("Modbus"),
        0x45 => Some("Originator Connection List"),
        0x46 => Some("Modbus Serial Link"),
        0x47 => Some("Device Level Ring (DLR)"),
        0x48 => Some("QoS"),
        0x49 => Some("Safety Analog Input Point"),
        0x4A => Some("Safety Analog Input Group"),
        0x4B => Some("Safety Dual Channel Analog Input"),
        0x4C => Some("SERCOS III Link"),
        0x4D => Some("Target Connection List"),
        0x4E => Some("Base Energy"),
        0x4F => Some("Electrical Energy"),
        0x50 => Some("Non-Electrical Energy"),
        0x51 => Some("Base Switch"),
        0x52 => Some("SNMP"),
        0x53 => Some("Power Management"),
        0x54 => Some("RSTP Bridge"),
        0x55 => Some("RSTP Port"),
        0x56 => Some("PRP/HSR Protocol"),
        0x57 => Some("PRP/HSR Nodes Table"),
        0x58 => Some("Safety Feedback"),
        0x59 => Some("Safety Dual Channel Feedback"),
        0x5A => Some("Safety Stop Functions"),
        0x5B => Some("Safety Limit Functions"),
        0x5C => Some("Power Curtailment"),
        0x5D => Some("CIP Security"),
        0x5E => Some("EtherNet/IP Security"),
        0x5F => Some("Certificate Management"),
        0x67 => Some("PCCC Class"),
        0xF0 => Some("ControlNet"),
        0xF1 => Some("ControlNet Keeper"),
        0xF2 => Some("ControlNet Scheduling"),
        0xF3 => Some("Connection Configuration"),
        0xF4 => Some("Port"),
        0xF5 => Some("TCP/IP Interface"),
        0xF6 => Some("Ethernet Link"),
        0xF7 => Some("CompoNet"),
        0xF8 => Some("CompoNet Repeater"),
        _ => None,
    }
}

fn log_cip_path_segment(c: &EnipCipPathSegment) -> Result<JsonBuilder, JsonError> {
    let mut js = JsonBuilder::try_new_object()?;
    match cip_segment_type_string(c.segment_type) {
        Some(val) => {
            js.set_string("segment_type", val)?;
        }
        None => {
            js.set_string("segment_type", &format!("unknown-{}", c.segment_type))?;
        }
    }
    js.set_uint("value", c.value)?;
    js.close()?;
    Ok(js)
}

fn log_cip(c: &EnipCIP, js: &mut JsonBuilder) -> Result<(), JsonError> {
    for item in c.items.iter() {
        if let EnipItemPayload::Data(d) = &item.payload {
            js.open_object("cip")?;
            log_cip_data(&d.cip, js)?;
            js.close()?;
            return Ok(());
        }
    }
    return Ok(());
}

fn log_cip_data(d: &CipData, js: &mut JsonBuilder) -> Result<(), JsonError> {
    match cip_service_string(d.service) {
        Some(val) => {
            js.set_string("service", val)?;
        }
        None => {
            js.set_string("service", &format!("unknown-{}", d.service))?;
        }
    }
    match &d.cipdir {
        CipDir::Response(resp) => {
            match cip_status_string(resp.status) {
                Some(val) => {
                    js.set_string("status", val)?;
                }
                None => {
                    js.set_string("status", &format!("unknown-{}", resp.status))?;
                }
            }
            if !resp.status_extended.is_empty() {
                js.set_hex("status_extended", &resp.status_extended)?;
                if let Some(val) = cip_status_extended_string(&resp.status_extended) {
                    js.set_string("status_extended_meaning", val)?;
                }
            }
            if let EnipCipResponsePayload::Multiple(m) = &resp.payload {
                if !m.packet_list.is_empty() {
                    js.open_array("multiple")?;
                    for p in m.packet_list.iter() {
                        let mut js2 = JsonBuilder::try_new_object()?;
                        // parsing has already limited recursion
                        log_cip_data(p, &mut js2)?;
                        js2.close()?;
                        js.append_object(&js2)?;
                    }
                    js.close()?;
                }
            }
        }
        CipDir::Request(req) => {
            if !req.path.is_empty() {
                js.open_array("path")?;
                for seg in req.path.iter() {
                    js.append_object(&log_cip_path_segment(seg)?)?;
                }
                js.close()?;
                for i in 0..req.path.len() {
                    //take last class
                    let idx = req.path.len() - 1 - i;
                    if (req.path[idx].segment_type & 0xFC) == 0x20 {
                        match cip_class_string(req.path[idx].value) {
                            Some(val) => {
                                js.set_string("class_name", val)?;
                            }
                            None => {
                                js.set_string(
                                    "class_name",
                                    &format!("unknown-{}", req.path[idx].value),
                                )?;
                            }
                        }
                        break;
                    }
                }
            }
            if let EnipCipRequestPayload::Multiple(m) = &req.payload {
                if !m.packet_list.is_empty() {
                    js.open_array("multiple")?;
                    for p in m.packet_list.iter() {
                        let mut js2 = JsonBuilder::try_new_object()?;
                        // parsing has already limited recursion
                        log_cip_data(p, &mut js2)?;
                        js2.close()?;
                        js.append_object(&js2)?;
                    }
                    js.close()?;
                }
            }
        }
        CipDir::None => {}
    }
    Ok(())
}

fn log_enip(tx: &EnipTransaction, js: &mut JsonBuilder) -> Result<(), JsonError> {
    js.open_object("enip")?;
    if let Some(ref request) = tx.request {
        js.open_object("request")?;
        log_enip_header(&request.header, js)?;
        match &request.payload {
            EnipPayload::Cip(cip) => {
                log_cip(cip, js)?;
            }
            EnipPayload::RegisterSession(rs) => {
                js.open_object("register_session")?;
                js.set_uint("protocol_version", rs.protocol_version)?;
                js.set_uint("options", rs.options)?;
                js.close()?;
            }
            _ => {}
        }
        js.close()?;
    }
    if let Some(ref response) = tx.response {
        js.open_object("response")?;
        log_enip_header(&response.header, js)?;
        match &response.payload {
            EnipPayload::RegisterSession(rs) => {
                js.open_object("register_session")?;
                js.set_uint("protocol_version", rs.protocol_version)?;
                js.set_uint("options", rs.options)?;
                js.close()?;
            }
            EnipPayload::Cip(cip) => {
                log_cip(cip, js)?;
            }
            EnipPayload::ListServices(lsp) if !lsp.is_empty() => {
                if let EnipItemPayload::Services(ls) = &lsp[0].payload {
                    js.open_object("list_services")?;
                    js.set_uint("protocol_version", ls.protocol_version)?;
                    js.set_uint("capabilities", ls.capabilities)?;
                    js.set_string("service_name", &String::from_utf8_lossy(&ls.service_name))?;
                    js.close()?;
                }
            }
            EnipPayload::ListIdentity(lip) if !lip.is_empty() => {
                if let EnipItemPayload::Identity(li) = &lip[0].payload {
                    js.open_object("identity")?;
                    js.set_uint("protocol_version", li.protocol_version)?;
                    js.set_string(
                        "revision",
                        &format!("{}.{}", li.revision_major, li.revision_minor),
                    )?;
                    match enip_vendorid_string(li.vendor_id) {
                        Some(val) => {
                            js.set_string("vendor_id", val)?;
                        }
                        None => {
                            js.set_string("vendor_id", &format!("unknown-{}", li.vendor_id))?;
                        }
                    }
                    match enip_devicetype_string(li.device_type) {
                        Some(val) => {
                            js.set_string("device_type", val)?;
                        }
                        None => {
                            js.set_string("device_type", &format!("unknown-{}", li.device_type))?;
                        }
                    }
                    js.set_uint("product_code", li.product_code)?;
                    js.set_uint("status", li.status)?;
                    js.set_uint("serial", li.serial)?;
                    js.set_string("product_name", &String::from_utf8_lossy(&li.product_name))?;
                    js.set_uint("state", li.state)?;
                    js.close()?;
                }
            }
            _ => {}
        }
        js.close()?;
    }
    js.close()?;
    Ok(())
}

#[no_mangle]
pub unsafe extern "C" fn SCEnipLoggerLog(
    tx: *mut std::os::raw::c_void, js: &mut JsonBuilder,
) -> bool {
    let tx = cast_pointer!(tx, EnipTransaction);
    if tx.request.is_none() && tx.response.is_none() {
        return false;
    }
    log_enip(tx, js).is_ok()
}
