/*****************************************************************************
 * telnet.c
 * Provide a telnet interface
 *****************************************************************************
 * Copyright (C) 1998, 1999, 2000, 2001 VideoLAN
 * $Id: telnet.c,v 1.15 2001/04/30 02:29:03 lool Exp $
 *
 * Authors: Damien Lucas <nitrox@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <stdio.h>                                                /* sprintf */
#include <stdlib.h>                                          /* free, malloc */
#include <string.h>                                                 /* bzero */
#include <unistd.h>                                    /* close, read, write */
#include <sys/socket.h>                                 /* socket, bind, ... */
#include <arpa/inet.h>                                            /* types.h */
#include <arpa/telnet.h>                                   /* WILL, IAC, ... */
#include <crypt.h>                                                  /* crypt */

#include "../types.h"
#include "../logger.h"
#include "telnet.h"
#include "interface.h"

#define PORT 7891
#define MAXCONNECTIONS 20 // how many connections are queued

/* TELNET Interface */
#define WELCOME    "\nVideoLAN Channel Server Administration System\n"
#define LOGIN      "Login: "
#define PASSWD     "Password: "
#define PROMPT     "@CHANNELServer # "

#define KEY_UP          65         //
#define KEY_DOWN        66         /* Various codes sent after the 0x1b byte */
#define KEY_LEFT        67         //
#define KEY_RIGHT       68         //


/* Protoypes of static functions further defined */

static void IF_telnet_init(int isocket);
static char* IF_telnet_login(int isocket);
static void IF_telnet_commands(int iSocket, char* sUser);

static void Menu_init(struct IF_Menu* mMenu);
static void Menu_send(int iSocket, struct IF_Menu mMenu);

static ERR_CODE TELNET_stop(int iSocket);
static ERR_CODE TELNET_reload(int iSocket);
static ERR_CODE TELNET_info(int iSocket);

static char* Catch_Word(int iSocket);
static char* Catch_Passwd(int iSocket);
static VS_MachineId Catch_MAC(int iSocket);

static void Request_send(int iSocket, char bAction, char bOption);
static void Message_send(int iSocket, char* sMessage, size_t size);
static void Prompt_send(int iSocket, char* sUser);
  
static unsigned int uStop;


/*****************************************************************************
 * IF_telnet() Main function of the telnet interface
 *****************************************************************************/
void* IF_telnet(void* ar)
{
  int sock,newsock ;
  struct sockaddr_in addr;
  struct sockaddr incoming;
  char* sUser;
  int size;
  unsigned int stop;
  int rc;
  int iOpt;
  
  stop=0;
  size=sizeof(struct sockaddr_in);
  iOpt=1;
  
  /* Initialization */
  sock=socket(AF_INET,SOCK_STREAM,0);
  if(sock==-1)
  {
    VS_log(LOGERROR,TELNET,"Error creating socket !");
    return NULL;
  }

  
  addr.sin_family = AF_INET;
  addr.sin_port = htons(PORT);
  addr.sin_addr.s_addr = INADDR_ANY;
  bzero(&(addr.sin_zero),8);
  
  setsockopt(sock,SOL_SOCKET, SO_REUSEADDR, &iOpt,sizeof(iOpt));
  rc=bind(sock,(struct sockaddr *)&addr,sizeof(struct sockaddr));
  if(rc==-1)
  {
    VS_log(LOGERROR,TELNET,"Unable to bind socket");
    return NULL;
  }
  
  
  rc=listen(sock,MAXCONNECTIONS);
  if(rc==-1)
  {
    VS_log(LOGERROR,TELNET,"Unable to listen to the socket");
    return NULL;
  }

  /* New connection */
  while(1)
  {
    newsock=accept(sock,&incoming,&size);
    IF_telnet_init(newsock);
    sUser=IF_telnet_login(newsock);
    if(sUser!=NULL)
    {
      IF_telnet_commands(newsock,sUser);
    }
    
    close(newsock);
  }

  return NULL;
}


/*****************************************************************************
 * IF_telnet_init()
 *****************************************************************************
 * Send the welcome message
 * Send the terminal specifications
 *****************************************************************************/
static void IF_telnet_init(int isocket)
{
  VS_log(LOGDEBUG, TELNET, "Connection attempt");
  
  /* Send the Welcome message */
  Message_send(isocket, WELCOME, strlen(WELCOME));
    
  /* Need to negociate Terminal ... */
  Request_send(isocket, WILL, TELOPT_ECHO);
  Request_send(isocket, WILL, TELOPT_SGA);
  Request_send(isocket, WILL, TELOPT_OUTMRK);
  Request_send(isocket, DONT, TELOPT_LINEMODE);
}


/*****************************************************************************
 * IF_telnet_login()
 *****************************************************************************
 * Send the login sequence
 * Get the login and the passwd
 *****************************************************************************/
static char* IF_telnet_login (int iSocket)
{
  char* sUser;
  char* sPasswd;
  sUser=NULL;
  sPasswd=NULL;
  
  Message_send(iSocket, LOGIN, strlen(LOGIN));
  sUser=Catch_Word(iSocket);
  if(sUser!=NULL)
  {
    Message_send(iSocket, PASSWD, strlen(PASSWD));
    sPasswd=Catch_Passwd(iSocket);
    if(sPasswd != NULL)
    {
      FILE * Fpasswd_file;
      char sCryptedPasswd[14];                         /* stored MD5 passord */
      char sSalt[2];                          /* password encryption entropy */
      char sName[9];   /* just to compare with the name in the password file */
      if((Fpasswd_file = fopen(PASSWD_FILE, "r")) == NULL)
      {
        VS_log(LOGDEBUG,TELNET, "Password file open error : %s",PASSWD_FILE);
        free(sUser);
        free(sPasswd);
        return NULL;
      }
      VS_log(LOGDEBUG,TELNET, "Connection opened for %s/%s",sUser,sPasswd);
      do
      {
        if (fscanf(Fpasswd_file, "%8s%13s", sName, sCryptedPasswd) != 2)
        {
          VS_log(LOGDEBUG,TELNET, "User %s not found or unknown", sUser);
          free(sUser);
          free(sPasswd);
          return NULL;
        }
      } while (strcmp(sUser, sName));
      sscanf(sCryptedPasswd, "%2s", sSalt);	
      VS_log(LOGDEBUG,TELNET,"%s",crypt(sPasswd,sSalt));
      if (strncmp(crypt(sPasswd, sSalt), sCryptedPasswd,strlen(sCryptedPasswd)))
      {
        VS_log(LOGDEBUG,TELNET, "Wrong password for user %s", sUser);
        free(sUser);
        free(sPasswd);
        return NULL;
      }
      VS_log(LOGDEBUG,TELNET, "User %s, login sucessful", sUser);
      free(sPasswd);
    }
  }
  return sUser;
}

/*****************************************************************************
 * IF_telnet_commands()
 *****************************************************************************
 * Once logged main function
 * Catch the commands and interprete them
 *****************************************************************************/
static void IF_telnet_commands (int iSocket, char* sUser)
{
  char* sCmd;
  struct IF_Menu mMenu;
  struct IFT_Command* z; 

  uStop=0;
  z=NULL;
  
  while(!uStop)
  {
    Menu_init(&mMenu);
    Menu_send(iSocket,mMenu);
    Prompt_send(iSocket,sUser);
    sCmd=Catch_Word(iSocket);
    for(z=mMenu.Command;z!=NULL;z=z->next)
    {
      if(!strcmp(sCmd,z->sName))
      {
        z->function(iSocket);
      }
    }
    free(sCmd);
  }  
}

/*****************************************************************************
 * Menu_init()
 *****************************************************************************
 * Fill the struct mMenu
 *                       TODO from a menuconfig
 *****************************************************************************/
static void Menu_init (struct IF_Menu* mMenu)
{
  struct IFT_Command* mMenu_current;/* pointer on current chained list cell  */

  mMenu->Command = malloc(sizeof(struct IFT_Command));

  mMenu_current = mMenu->Command;

  mMenu_current->iIndex='0';
  mMenu_current->function=TELNET_stop;
  sprintf(mMenu_current->sName,"logout");
  sprintf(mMenu_current->sDescr,"Close the current connection");

  mMenu_current->next = malloc(sizeof(struct IFT_Command));
  mMenu_current = mMenu_current->next;
  
  mMenu_current->iIndex='1';
  mMenu_current->function=TELNET_reload;
  sprintf(mMenu_current->sName,"reload");
  sprintf(mMenu_current->sDescr,"Reload the database");

  mMenu_current->next = malloc(sizeof(struct IFT_Command));
  mMenu_current = mMenu_current->next;

  mMenu_current->iIndex='2';
  mMenu_current->function=TELNET_info;
  sprintf(mMenu_current->sName,"info");
  sprintf(mMenu_current->sDescr,"Informations for a MAC address");
  
  mMenu_current->next=NULL;

}

/*****************************************************************************
 * TELNET_stop()
 *****************************************************************************
 * Give the order to stop the telnetd using uStop
 *****************************************************************************/
static ERR_CODE TELNET_stop(int iSocket)
{
  uStop=1;
  return 0;
}

/*****************************************************************************
 * TELNET_reload()
 *****************************************************************************
 * Give the order to reload the DB calling IF_reload() 
 *****************************************************************************/
static ERR_CODE TELNET_reload(int iSocket)
{
  IF_reload();
  return 0;
}

/*****************************************************************************
 * TELNET_info()
 *****************************************************************************
 * Catch a MAC using Catch_MAC()
 * Get info about it calling IF_info()
 *****************************************************************************/
static ERR_CODE TELNET_info(int iSocket)
{
  char * sMessage;
  VS_MachineId mMac;
  
  mMac = Catch_MAC(iSocket);
  sMessage = IF_info(mMac);
  
  Message_send(iSocket,sMessage,strlen(sMessage));
  Message_send(iSocket,"",0);
  return 0;
}

/*****************************************************************************
 * Menu_send()
 *****************************************************************************
 * Send the Menu using Message_send() 
 *****************************************************************************/
static void Menu_send (int iSocket, struct IF_Menu mMenu)
{
  struct IFT_Command* z;

  Message_send(iSocket,"\r\n",2);
  Message_send(iSocket,
  "------------------------- VLCS menu ------------------------------\r\n",68); 
  for(z=mMenu.Command; z!=NULL; z=z->next)
  {
    Message_send(iSocket,&(z->iIndex), 1);
    Message_send(iSocket,"     - ",8);
    Message_send(iSocket,z->sName,strlen(z->sName));
    Message_send(iSocket,"                  - ",21-strlen(z->sName));
    Message_send(iSocket,z->sDescr,strlen(z->sDescr));
    Message_send(iSocket,"\r\n",2);
  }
}



/*****************************************************************************
 * Catch_Word()
 *****************************************************************************
 * Read in the socket and catch chars with local echo
 * Return after a \r\n sequence
 *****************************************************************************/
static char* Catch_Word (int iSocket)
{
  unsigned char bBuff;
  char* sData;
  int iPos;
  
  sData=malloc(300*sizeof(char));
  iPos=0;

  while(iPos<254)
  {
    read(iSocket, &bBuff,1);
    switch(bBuff)
    {
      case IAC:
        read(iSocket, &bBuff,1);
        read(iSocket, &bBuff,1);
        break;                                      //TODO Terminal negociation
      
      case '\r':
        sData[iPos]='\0';
        read(iSocket, &bBuff,1);         //Need to read next byte: should be \n
        Message_send(iSocket,"\r\n",2);
        return sData;

      case 0x7f:
        if(iPos > 0)
        {
          unsigned char b4Buff[4];
	  iPos--;
          sData[iPos]='\0';
          b4Buff[0]=0x08;
          b4Buff[1]=0x20;
          b4Buff[2]=0x08;
          Message_send(iSocket,b4Buff,3);
        }
	break;
         
      case 0x1b:
        break;                              // TODO Add test 0x1b 91 KEY_UP ../

      default:
        if (((bBuff>='0')&&(bBuff<='9')) || ((bBuff>='A')&&(bBuff<='Z')) || \
            ((bBuff>='a')&&(bBuff<='z')))
        {
          sData[iPos]=bBuff;
          iPos++;
          Message_send(iSocket,&bBuff,1);
        }
        break;
    }
  }
  return NULL;
}


/*****************************************************************************
 * Catch_MAC()
 *****************************************************************************
 * Read in the socket and catch chars '0'..'9', ':' with local echo
 * Return after a \r\n sequence
 *****************************************************************************/
static VS_MachineId Catch_MAC (int iSocket)
{
  unsigned char bBuff;
  char* sData;
  char end_loop = 0;
  int iPos = 0;
  int iIndex1, iIndex2;
  VS_MachineId mMac;
  
  sData = malloc(18 * sizeof(char));

  while (!end_loop)
  {
    read(iSocket, &bBuff, 1);
    switch(bBuff)
    {
      case IAC:
        read(iSocket, &bBuff, 1);
        read(iSocket, &bBuff, 1);
        break;                                      //TODO Terminal negociation
      
      case '\r':
        sData[iPos]='\0';
        read(iSocket, &bBuff, 1);        //Need to read next byte: should be \n
        Message_send(iSocket,"\r\n", 2);
        end_loop = 1;
        break;

      case 0x7f:                                                   // Backspace
        if(iPos > 0)
        {
          iPos--;
          sData[iPos]='\0';
          Message_send(iSocket, "\x8\x20\x8", 3);
        }                // {0x08, 0x20, 0x08} =  Backspace - Space - Backspace
        break;

      case 0x1b:
        break;
          
      default:
        if (((bBuff>='0')&&(bBuff<='9')) || ((bBuff>='A')&&(bBuff<='F')) || \
             (bBuff==':'))
        {
          if (iPos <= 16)
          {
            sData[iPos] = bBuff;
            iPos++;
            sData[iPos] = '\0';
            Message_send(iSocket, &bBuff, 1);
          }
        }
        break;
    }
  }
        // this removes ':' chars of MAC address TODO fill with 0 between 2 ':'
  for (iIndex1 = 0, iIndex2 = 0; iIndex1 < iPos; iIndex1++)
  {
    if (sData[iIndex1] != ':')
    {
      sData[iIndex2] = sData[iIndex1];
      iIndex2++;
    }
  }
  sData[iIndex2] = '\0';
  sscanf(sData, "%llx", &mMac);
  return mMac;                                          // sData is null padded
}

/*****************************************************************************
 * Catch_Passwd()
 *****************************************************************************
 * Read in the socket and catch chars BUT without any local echo
 * Return after a \r\n sequence
 *****************************************************************************/
static char* Catch_Passwd (int iSocket)
{
  unsigned char bBuff;
  char* sData;
  int iPos;
  
  sData=malloc(300*sizeof(char));
  if(sData==NULL)
  {
    VS_log(LOGERROR,TELNET,"Unable to allocate memory");
    return NULL;
  }

  iPos=0;

  while(iPos<254)
  {
    read(iSocket, &bBuff, 1);
    switch(bBuff)
    {
      case IAC:
        read(iSocket, &bBuff, 2);
        break;                                      //TODO Terminal negociation
      
      case '\r':
        sData[iPos]='\0';
        read(iSocket, &bBuff, 1);        //Need to read next byte: should be \n
        Message_send(iSocket,"\r\n", 2);
        return sData;

      default:
        sData[iPos]=bBuff;
        iPos++;
        break;
    }
  }
  return NULL;
}


/*****************************************************************************
 * Request_send()
 *****************************************************************************
 * Send all commands to the telnet client
 * A command should begin with IAC, then Action, then Option
 *****************************************************************************/
static void Request_send (int iSocket, char bAction, char bOption)
{
  unsigned char req[3];
  
  req[0]=IAC;
  req[1]=bAction;
  req[2]=bOption;
  
  write(iSocket, req, 3);
}



/*****************************************************************************
 * Message_send()
 *****************************************************************************
 * Write in the socket
 *****************************************************************************/
static void Message_send (int iSocket, char* sMessage, size_t size)
{
  write(iSocket, sMessage,size);
}

/*****************************************************************************
 * Prompt_send()
 *****************************************************************************
 * Send a prompt, with the PROMPT def and the sUser variable
 * Begin with a \r\n sequence
 *****************************************************************************/
static void Prompt_send (int iSocket, char* sUser)
{
  Message_send(iSocket,"\r\n",2);
  Message_send(iSocket, sUser,strlen(sUser));
  Message_send(iSocket, PROMPT,strlen(PROMPT));
}
