#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2011 Yesudeep Mangalapilly <yesudeep@gmail.com>
# Copyright 2012 Google, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


"""
:module: watchdog.observers.polling
:synopsis: Polling emitter implementation.
:author: yesudeep@google.com (Yesudeep Mangalapilly)

Classes
-------
.. autoclass:: PollingEmitter
   :members:
   :show-inheritance:
"""

from __future__ import with_statement

import time
import threading

from watchdog.utils.dirsnapshot import DirectorySnapshot, DirectorySnapshotDiff
from watchdog.observers.api import\
  EventEmitter,\
  BaseObserver,\
  DEFAULT_OBSERVER_TIMEOUT,\
  DEFAULT_EMITTER_TIMEOUT
from watchdog.events import\
  DirMovedEvent,\
  DirDeletedEvent,\
  DirCreatedEvent,\
  DirModifiedEvent,\
  FileMovedEvent,\
  FileDeletedEvent,\
  FileCreatedEvent,\
  FileModifiedEvent


class PollingEmitter(EventEmitter):
  """
  Platform-independent emitter that polls a directory to detect file
  system changes.
  """

  def __init__(self, event_queue, watch, timeout=DEFAULT_EMITTER_TIMEOUT):
    EventEmitter.__init__(self, event_queue, watch, timeout)
    self._snapshot = DirectorySnapshot(watch.path, watch.is_recursive)
    self._lock = threading.Lock()

  def on_thread_exit(self):
    self._snapshot = None


  def queue_events(self, timeout):
    with self._lock:
      # We don't want to hit the disk continuously.
      # timeout behaves like an interval for polling emitters.
      time.sleep(timeout)

      # Get event diff between fresh snapshot and previous snapshot.
      # Update snapshot.
      new_snapshot = DirectorySnapshot(self.watch.path, self.watch.is_recursive)
      events = DirectorySnapshotDiff(self._snapshot, new_snapshot)
      self._snapshot = new_snapshot

      # Files.
      for src_path in events.files_deleted:
        self.queue_event(FileDeletedEvent(src_path))
      for src_path in events.files_modified:
        self.queue_event(FileModifiedEvent(src_path))
      for src_path in events.files_created:
        self.queue_event(FileCreatedEvent(src_path))
      for src_path, dest_path in events.files_moved:
        self.queue_event(FileMovedEvent(src_path, dest_path))

      # Directories.
      for src_path in events.dirs_deleted:
        self.queue_event(DirDeletedEvent(src_path))
      for src_path in events.dirs_modified:
        self.queue_event(DirModifiedEvent(src_path))
      for src_path in events.dirs_created:
        self.queue_event(DirCreatedEvent(src_path))
      for src_path, dest_path in events.dirs_moved:
        self.queue_event(DirMovedEvent(src_path, dest_path))


class PollingObserver(BaseObserver):
  """
  Observer thread that schedules watching directories and dispatches
  calls to event handlers.
  """

  def __init__(self, timeout=DEFAULT_OBSERVER_TIMEOUT):
    BaseObserver.__init__(self, emitter_class=PollingEmitter, timeout=timeout)


